// Copyright (C) 2015 The Syncthing Authors.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this file,
// You can obtain one at https://mozilla.org/MPL/2.0/.

// Checks for files missing copyright notice
package meta

import (
	"bufio"
	"fmt"
	"os"
	"path/filepath"
	"regexp"
	"strings"
	"testing"
)

// File extensions to check
var copyrightCheckExts = map[string]bool{
	".go": true,
}

// Directories to search
var copyrightCheckDirs = []string{".", "../cmd", "../lib", "../test", "../script"}

// Valid copyright headers, searched for in the top five lines in each file.
var copyrightRegexps = []string{
	`Copyright`,
	`package auto`,
	`automatically generated by genxdr`,
	`generated by protoc`,
	`^// Code generated .* DO NOT EDIT\.$`,
}

var copyrightRe = regexp.MustCompile(strings.Join(copyrightRegexps, "|"))

func TestCheckCopyright(t *testing.T) {
	for _, dir := range copyrightCheckDirs {
		err := filepath.Walk(dir, checkCopyright)
		if err != nil {
			t.Error(err)
		}
	}
}

func checkCopyright(path string, info os.FileInfo, err error) error {
	if err != nil {
		return err
	}
	if !info.Mode().IsRegular() {
		return nil
	}
	if !copyrightCheckExts[filepath.Ext(path)] {
		return nil
	}

	fd, err := os.Open(path)
	if err != nil {
		return err
	}
	defer fd.Close()

	scanner := bufio.NewScanner(fd)
	for i := 0; scanner.Scan() && i < 5; i++ {
		if copyrightRe.MatchString(scanner.Text()) {
			return nil
		}
	}

	return fmt.Errorf("missing copyright in %s?", path)
}
