TGax Video Traffic Model
------------------------

This section describes the implementation of the |ns3| TGax video traffic generator model, which
is based on the IEEE 802.11-14/0571r12 TGAX evaluation methodology [TGAX_EVAL]_. The model generates
buffered video streaming traffic patterns (notionally generated by services such as YouTube
or Netflix) suitable for evaluating IEEE 802.11 networks and other wireless systems.

The TGax video traffic model is implemented in the ``src/applications/`` folder.

The video traffic generator models the behavior of a video streaming source, producing
traffic that simulates video frames being generated at the application layer, fragmented
at the transport layer, and experiencing network latency as they traverse the Internet
before reaching the access point for transmission to clients.

The model implements a Weibull distribution for video frame size generation and a Gamma
distribution for network latency modeling, as specified in the IEEE 802.11-14/0571r12
evaluation methodology. The latter is disabled by default since the model is primarily intended
for use with a link simulator that does not model queues and TCP behavior.

Scope and Limitations
~~~~~~~~~~~~~~~~~~~~~

The TGax video model implements the buffered video streaming and multicast video streaming
traffic models from the IEEE 802.11-14/0571r12 specification. The following describes
what is covered and what is not:

**Supported Features:**

- Eight predefined traffic model profiles (BV1-BV6 for buffered video, MC1-MC2 for multicast)
- Custom traffic model configuration with user-defined parameters
- Weibull distribution for video frame size generation
- Gamma distribution for network latency modeling (mean ~14.8ms)
- TCP and UDP protocol support
- Multicast video modes
- Automatic TCP segment size handling for large video frames
- Comprehensive trace sources for monitoring frame generation and packet transmission

**Known Limitations:**

- The Wireless Display (WD1-WD4) traffic models from the specification are not implemented
- The Video Conferencing (VC) traffic model is not explicitly implemented (users can simulate
  this by creating bidirectional BV1 streams)
- TCP acknowledgment traffic follows ns-3's built-in TCP behavior rather than the explicit
  40-byte/1ms delay model in the specification
- Network latency is not regenerated if it would cause packets to arrive after simulation end

Model Description
~~~~~~~~~~~~~~~~~

The ``TgaxVideoTraffic`` application is a source application that generates video streaming
traffic patterns according to the IEEE 802.11-14/0571r12 evaluation methodology. The application
models video streaming services such as YouTube and Netflix.

**Traffic Generation Process:**

The model follows a three-step process:

1. **Frame Size Generation**: At the application layer, video frame sizes (in bytes) are
   generated according to a Weibull distribution with parameters that vary by target bit rate.

2. **Frame Transmission**: Video frames are sent to the transport layer. For TCP, frames
   larger than the segment size (1500 bytes) are automatically fragmented.

3. **Network Latency**: Each packet experiences network latency drawn from a Gamma distribution
   with mean approximately 14.8ms, simulating the delay encountered as video traffic
   traverses multiple hops in the Internet before reaching the access point.

(note, if code review comment accepted, step 3 should be described as optional)

**Predefined Traffic Models:**

The implementation provides eight predefined traffic model profiles, matching those in the
IEEE specification:

*Buffered Video (BV) Models:*

.. list-table:: Buffered Video Traffic Models
   :header-rows: 1

   * - Model ID
     - Bit Rate
     - Weibull Scale (lambda)
     - Weibull Shape (k)
   * - BV1
     - 2 Mbps
     - 6950
     - 0.8099
   * - BV2
     - 4 Mbps
     - 13900
     - 0.8099
   * - BV3
     - 6 Mbps
     - 20850
     - 0.8099
   * - BV4
     - 8 Mbps
     - 27800
     - 0.8099
   * - BV5
     - 10 Mbps
     - 34750
     - 0.8099
   * - BV6
     - 15.6 Mbps
     - 54210
     - 0.8099

*Multicast Video (MC) Models:*

.. list-table:: Multicast Video Traffic Models
   :header-rows: 1

   * - Model ID
     - Bit Rate
     - Weibull Scale (lambda)
     - Weibull Shape (k)
   * - MC1
     - 3 Mbps
     - 10425
     - 0.8099
   * - MC2
     - 6 Mbps
     - 20850
     - 0.8099

*Custom model:*

The default CUSTOM model has parameter defaults corresponding to BV1 model, but is intended
to be used and reconfigured when users may want to use their own model parameters.

**Network Latency Model:**

The network latency experienced by each packet is modeled using a Gamma distribution:

* **Shape parameter (k)**: 0.2463
* **Scale parameter (theta)**: 60.227
* **Mean latency**: k * theta = 14.834 ms

This models the delay variation experienced as video packets traverse multiple network
hops between the video server and the access point.

Configuration and Usage
***********************

The video traffic generator is instantiated as an application through the ``ApplicationHelper``.
Here is a basic example using UDP:

::

  // Create nodes
  NodeContainer nodes;
  nodes.Create(2);

  // Install Internet stack and assign IP addresses
  InternetStackHelper internet;
  internet.Install(nodes);

  Ipv4AddressHelper ipv4;
  ipv4.SetBase("10.0.0.0", "255.255.255.0");
  ipv4.Assign(devices);

  // Configure video application with BV3 (6 Mbps) profile
  ApplicationHelper sourceHelper(TgaxVideoTraffic::GetTypeId());
  sourceHelper.SetAttribute("Protocol", TypeIdValue(UdpSocketFactory::GetTypeId()));
  sourceHelper.SetAttribute("Remote", AddressValue(InetSocketAddress(Ipv4Address("10.0.0.2"), 5000)));
  sourceHelper.SetAttribute("TrafficModelClassIdentifier", StringValue("BV3"));

  auto sourceApps = sourceHelper.Install(nodes.Get(0));
  sourceApps.Start(Seconds(1.0));
  sourceApps.Stop(Seconds(11.0));

  // Create receiver
  PacketSinkHelper sinkHelper("ns3::UdpSocketFactory", InetSocketAddress(Ipv4Address::GetAny(), 5000));
  auto sinkApps = sinkHelper.Install(nodes.Get(1));
  sinkApps.Start(Seconds(0.0));
  sinkApps.Stop(Seconds(12.0));

For custom traffic parameters:

::

  // Configure custom video parameters
  ApplicationHelper sourceHelper(TgaxVideoTraffic::GetTypeId());
  sourceHelper.SetAttribute("Protocol", TypeIdValue(UdpSocketFactory::GetTypeId()));
  sourceHelper.SetAttribute("Remote", AddressValue(InetSocketAddress(Ipv4Address("10.0.0.2"), 5000)));
  sourceHelper.SetAttribute("TrafficModelClassIdentifier", StringValue("Custom"));
  sourceHelper.SetAttribute("CustomVideoBitRate", DataRateValue(DataRate("5Mbps")));
  sourceHelper.SetAttribute("CustomWeibullScale", DoubleValue(17375));
  sourceHelper.SetAttribute("CustomWeibullShape", DoubleValue(0.8099));

Attributes
~~~~~~~~~~

The video traffic generator supports UDP and TCP protocols through the ``Protocol`` attribute:

* **ns3::UdpSocketFactory**: UDP socket (recommended for video streaming simulation)
* **ns3::TcpSocketFactory**: TCP socket (realistic for actual video streaming behavior)

The ``Remote`` address must be configured appropriately:

* For UDP/TCP: Use ``InetSocketAddress`` for IPv4 or ``Inet6SocketAddress`` for IPv6

The following additional attributes control the behavior of the video traffic generator:

**Protocol Configuration:**

* ``Protocol``: The socket factory type to use. Default: ``ns3::UdpSocketFactory``
* ``Remote``: The remote address and port for the video destination

**Traffic Model Selection:**

* ``TrafficModelClassIdentifier``: The traffic model to use. Options:
  BV1, BV2, BV3, BV4, BV5, BV6, MC1, MC2, or Custom. Default: Custom

  The BV1-BV6 and MC1-MC2 options correspond to the predefined traffic models from the
  IEEE 802.11-14/0571r12 specification. The Custom option is an implementation extension
  (not part of the IEEE specification) that allows users to specify their own Weibull
  distribution parameters for experimentation or interpolation between defined bit rates.

**Custom Model Parameters (only used when TrafficModelClassIdentifier is Custom):**

* ``CustomVideoBitRate``: The target video bit rate. Default: 2 Mbps
* ``CustomWeibullScale``: Scale parameter (lambda) for the Weibull distribution. Default: 6950
* ``CustomWeibullShape``: Shape parameter (k) for the Weibull distribution. Default: 0.8099

Note: The default values for the Custom model match the BV1 parameters from the specification.

**Network Latency Parameters:**

* ``GammaShape``: Shape parameter (k) for the Gamma distribution for network latency. Default: 0.2463
* ``GammaScale``: Scale parameter (theta) for the Gamma distribution. Default: 60.227

Trace Sources
~~~~~~~~~~~~~

The video traffic generator provides two main trace sources for monitoring simulation behavior:

**TxWithLatency:**

::

  typedef void (*TxTracedCallback)(Ptr<const Packet> packet, Time latency);

This trace source is fired when a packet is transmitted. It provides both the packet being
transmitted and the network latency value that was applied to that packet. This is useful
for analyzing packet transmission patterns and the distribution of network latency.

**VideoFrameGenerated:**

::

  typedef void (*FrameGeneratedCallback)(uint32_t frameSize);

This trace source is fired when a video frame is generated at the application layer.
It provides the size of the generated frame in bytes. This is useful for analyzing the
frame size distribution and verifying that it matches the expected Weibull distribution.

Examples
~~~~~~~~

The ``tgax-video-example.cc`` program demonstrates basic usage of the video traffic generator.
This example sets up a simple Wi-Fi network (AP and STAs) with video streaming traffic
flowing from the AP to the station(s). The example supports:

* All predefined traffic models (BV1-BV6, MC1-MC2)
* Both UDP and TCP protocols
* Multicast delivery to multiple receivers

To run with default settings (BV1, 2 Mbps over UDP):

::

  $ ./ns3 run tgax-video-example

To run with a higher bit rate profile:

::

  $ ./ns3 run "tgax-video-example --model=BV3"

To run with TCP:

::

  $ ./ns3 run "tgax-video-example --protocol=tcp"

To run a multicast scenario:

::

  $ ./ns3 run "tgax-video-example --model=MC1"

Tests
~~~~~

Unit tests for the video traffic model are provided in
``src/applications/test/tgax-video-traffic-test-suite.cc``. These tests verify:

* All predefined traffic models (BV1-BV6) work correctly with both TCP and UDP
* Multicast models (MC1, MC2) deliver traffic to multiple receivers
* Custom traffic model parameters are applied correctly
* Measured bit rate matches expected bit rate (within 5% tolerance)
* Inter-frame duration matches expected value
* Average network latency matches specification (14.834ms)
* All transmitted bytes are received

To run the video traffic tests:

::

  $ ./test.py -s applications-tgax-video-traffic

Verification
~~~~~~~~~~~~

The video traffic model has been checked against the IEEE 802.11-14/0571r12 specification
by comparing:

* All predefined model parameters (Weibull scale and shape) against Table 5 in the specification
* Network latency Gamma distribution parameters against the specification
* Measured bit rates against expected bit rates for each traffic model
* Frame size distribution against expected Weibull properties
* Network latency distribution against expected Gamma properties (mean ~14.8ms)

The test suite demonstrates that the model correctly implements the buffered video
streaming traffic model, produces appropriate frame sizes, and generates traffic
patterns consistent with real video streaming applications.

Video Conferencing Simulation
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Although the Video Conferencing (VC) traffic model is not explicitly implemented as a
separate class, users can simulate video conferencing scenarios by creating bidirectional
BV1 streams. According to the specification:

* Traffic from AP to station: Use the BV1 model (with network latency)
* Traffic from station to AP: Use the BV1 model (without network latency, or set GammaScale to 0)

