/* YACC parser for C++ syntax.
   Copyright (C) 1988, 1989 Free Software Foundation, Inc.
   Hacked by Michael Tiemann (tiemann@mcc.com)

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


/* This grammar is based on the GNU CC grammar.  */

%{
#include "config.h"
#include "tree.h"
#include "input.h"
#include "cplus-parse.h"
#include "cplus-tree.h"
#include "assert.h"

/* C++ extensions */
extern tree ridpointers[];	/* need this up here */

#include <stdio.h>

#include <errno.h>

extern int errno;

extern int end_of_file;

void yyerror ();

/* true iff ok to declare a new varaible in a stmt decl.  */
static int stmt_decl_ok = 1;

/* Cause the `yydebug' variable to be defined.  */
#define YYDEBUG 1
%}

%start program

%union {long itype; tree ttype; enum tree_code code; }

/* All identifiers that are not reserved words
   and are not declared typedefs in the current block */
%token IDENTIFIER

/* All identifiers that are declared typedefs in the current block.
   In some contexts, they are treated just like IDENTIFIER,
   but they can also serve as typespecs in declarations.  */
%token TYPENAME

/* Reserved words that specify storage class.
   yylval contains an IDENTIFIER_NODE which indicates which one.  */
%token SCSPEC

/* Reserved words that specify type.
   yylval contains an IDENTIFIER_NODE which indicates which one.  */
%token TYPESPEC

/* Reserved words that qualify type: "const" or "volatile".
   yylval contains an IDENTIFIER_NODE which indicates which one.  */
%token TYPE_QUAL

/* Character or numeric constants.
   yylval is the node for the constant.  */
%token CONSTANT

/* String constants in raw form.
   yylval is a STRING_CST node.  */
%token STRING

/* "...", used for functions with variable arglists.  */
%token ELLIPSIS

/* the reserved words */
%token SIZEOF ENUM /* STRUCT UNION */ IF ELSE WHILE DO FOR SWITCH CASE DEFAULT
%token BREAK CONTINUE RETURN GOTO ASM TYPEOF ALIGNOF

/* the reserved words... C++ extensions */
%token <ttype> AGGR
%token DELETE NEW OVERLOAD PRIVATE PUBLIC PROTECTED THIS OPERATOR
%token OPERATOR_PUSH OPERATOR_POP DYNAMIC POINTSAT_LEFT_RIGHT
%token <itype> SCOPE

/* Define the operator tokens and their precedences.
   The value is an integer because, if used, it is the tree code
   to use in the expression made from the operator.  */

%left EMPTY			/* used to resolve s/r with epsilon */

/* Add precedence rules to solve dangling else s/r conflict */
%nonassoc IF
%nonassoc ELSE

%left IDENTIFIER TYPENAME TYPENAME_COLON SCSPEC TYPESPEC TYPE_QUAL ENUM AGGR

%left '{' ','

%right <code> ASSIGN '='
%right <code> '?' ':' RANGE
%left <code> OROR
%left <code> ANDAND
%left <code> '|'
%left <code> '^'
%left <code> '&'
%left <code> MIN_MAX
%left <code> EQCOMPARE
%left <code> ARITHCOMPARE
%left <code> LSHIFT RSHIFT
%left <code> '+' '-'
%left <code> '*' '/' '%'
%right <code> UNARY PLUSPLUS MINUSMINUS
%left HYPERUNARY
%left <ttype> PAREN_STAR_PAREN PAREN_X_SCOPE_STAR_PAREN PAREN_X_SCOPE_REF_PAREN
%left <code> POINTSAT '.' '(' '['

%right SCOPE			/* C++ extension */
%nonassoc NEW DELETE
%right DYNAMIC

%type <code> unop

%type <ttype> identifier IDENTIFIER TYPENAME CONSTANT expr nonnull_exprlist exprlist
%type <ttype> expr_no_commas unary_expr cast_expr primary string STRING
%type <ttype> typed_declspecs reserved_declspecs
%type <ttype> typed_typespecs reserved_typespecquals
%type <ttype> declmods typespecqual_reserved
%type <ttype> typespec SCSPEC TYPESPEC TYPE_QUAL nonempty_type_quals maybe_type_qual
%type <itype> initdecls notype_initdecls initdcl	/* C++ modification */
%type <ttype> init initlist maybeasm
%type <ttype> asm_operands nonnull_asm_operands asm_operand asm_clobbers

%type <ttype> declarator notype_declarator after_type_declarator

%type <ttype> structsp opt.component_decl_list component_decl_list component_decl components component_declarator
%type <ttype> enumlist enumerator
%type <ttype> typename absdcl absdcl1 type_quals
%type <ttype> xexpr parmlist parms parm bad_parm

/* C++ extensions */
%token <ttype> TYPENAME_COLON TYPENAME_SCOPE TYPENAME_ELLIPSIS
%token <ttype> PRE_PARSED_FUNCTION_DECL EXTERN_LANG_STRING
%type <ttype> fn.def2 dummy_decl x_typespec return_id
%type <ttype> class_head opt.init base_class_list base_class_visibility_list
%type <ttype> after_type_declarator_no_typename
%type <ttype> component_declarator0 scoped_declarator
%type <ttype> forhead.1 identifier_or_opname operator_name
%type <ttype> new object primary_no_id aggr
%type <itype> LC forhead.2 initdcl0 notype_initdcl0 wrapper member_init_list

%{
/* the declaration found for the last IDENTIFIER token read in.
   yylex must look this up to detect typedefs, which get token type TYPENAME,
   so it is left around in case the identifier is not a typedef but is
   used in a context which makes it a reference to a variable.  */
tree lastiddecl;

tree make_pointer_declarator (), make_reference_declarator ();

tree combine_strings ();
void reinit_parse_for_function ();
void reinit_parse_for_method ();

/* list of types and structure classes of the current declaration */
tree current_declspecs;

int undeclared_variable_notice;	/* 1 if we explained undeclared var errors. */

int yylex ();
%}

%%
program: /* empty */
	| extdefs
		{ finish_file (); }
	;

/* the reason for the strange actions in this rule
 is so that notype_initdecls when reached via datadef
 can find a valid list of type and sc specs in $0. */

extdefs:
	  {$<ttype>$ = NULL_TREE; } extdef
	| extdefs {$<ttype>$ = NULL_TREE; } extdef
	;

extdef:
	  fndef
		{ if (pending_inlines) do_pending_inlines (); }
	| datadef
		{ if (pending_inlines) do_pending_inlines (); }
	| overloaddef
	| ASM '(' string ')' ';'
		{ if (pedantic)
		    warning ("ANSI C forbids use of `asm' keyword");
		  if (TREE_CHAIN ($3)) $3 = combine_strings ($3);
		  assemble_asm ($3); }
	| EXTERN_LANG_STRING '{'
		{ push_lang_context ($1); }
	  extdefs '}'
		{ pop_lang_context (); }
	| EXTERN_LANG_STRING '{' '}'
		{ push_lang_context ($1);
		  pop_lang_context (); }
	| extern_lang_string fndef
		{ if (pending_inlines) do_pending_inlines ();
		  pop_lang_context (); }
	| extern_lang_string datadef
		{ if (pending_inlines) do_pending_inlines ();
		  pop_lang_context (); }
	;

extern_lang_string:
	  EXTERN_LANG_STRING
		{ push_lang_context ($1); }
	;

overloaddef:
	  OVERLOAD ov_identifiers ';'

ov_identifiers: IDENTIFIER
		{ declare_overloaded ($1); }
	| ov_identifiers ',' IDENTIFIER
		{ declare_overloaded ($3); }
	;
	  
dummy_decl: /* empty */
		{ $$ = NULL_TREE; }
	;

datadef:
	  dummy_decl notype_initdecls ';'
		{ if (pedantic)
		    error ("ANSI C forbids data definition lacking type or storage class");
  		  else if (! flag_traditional)
  		    warning ("data definition lacks type or storage class"); }
	| declmods notype_initdecls ';'
		{}
	| typed_declspecs initdecls ';'
		{}
        | declmods ';'
	  { error ("empty declaration"); }
	| typed_declspecs ';'
	  { shadow_tag ($1); }
	| error ';'
	| error '}'
	| ';'
	;

fndef:
	  fn.def1 base_init compstmt_or_error
		{
		  finish_function (lineno, 1);
		  /* finish_function performs these three statements:

		     expand_end_bindings (getdecls (), 1, 0);
		     poplevel (1, 1, 0);

		     expand_end_bindings (0, 0, 0);
		     poplevel (0, 0, 1);
		     */
		}
	| fn.def1 return_init base_init compstmt_or_error
		{
		  finish_function (lineno, 1);
		  /* finish_function performs these three statements:

		     expand_end_bindings (getdecls (), 1, 0);
		     poplevel (1, 1, 0);

		     expand_end_bindings (0, 0, 0);
		     poplevel (0, 0, 1);
		     */
		}
	| fn.def1 xdecls compstmt_or_error
		{ finish_function (lineno, 0); }
	| fn.def1 return_init ';' xdecls compstmt_or_error
		{
		  finish_function (lineno, 0);
		}
	| fn.def1 return_init nodecls compstmt_or_error
		{
		  finish_function (lineno, 0);
		}
	| typed_declspecs declarator error
		{}
	| declmods notype_declarator error
		{}
	| dummy_decl notype_declarator error
		{}
	;

fn.def1:
	  typed_declspecs declarator
		{ if (! start_function ($1, $2, 0))
		    YYERROR;
		  reinit_parse_for_function (); }
	| declmods notype_declarator
		{ if (! start_function ($1, $2, 0))
		    YYERROR;
		  reinit_parse_for_function (); }
	| dummy_decl notype_declarator
		{ if (! start_function (NULL_TREE, $2, 0))
		    YYERROR;
		  reinit_parse_for_function (); }
	| dummy_decl TYPENAME '(' parmlist ')'
		{ if (! start_function (NULL_TREE, build_nt (CALL_EXPR, $2, $4), 0))
		    YYERROR;
		  reinit_parse_for_function (); }
	| PRE_PARSED_FUNCTION_DECL
		{ start_function (NULL_TREE, $1, 1);
		  reinit_parse_for_function (); }
	;

/* more C++ complexity */
fn.def2:
	  typed_declspecs '(' parmlist ')'
		{
		  tree decl = build_nt (CALL_EXPR, TREE_VALUE ($1), $3);
		  $$ = start_method (TREE_CHAIN ($1), decl);
		  if (! $$)
		    YYERROR;
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  reinit_parse_for_method (yychar, $$); }
	| typed_declspecs declarator
		{ $$ = start_method ($1, $2);
		  if (! $$)
		    YYERROR;
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  reinit_parse_for_method (yychar, $$); }
	| declmods '(' parmlist ')'
		{
		  tree decl = build_nt (CALL_EXPR, TREE_VALUE ($1), $3);
		  $$ = start_method (TREE_CHAIN ($1), decl);
		  if (! $$)
		    YYERROR;
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  reinit_parse_for_method (yychar, $$); }
	| declmods declarator
		{ $$ = start_method ($1, $2);
		  if (! $$)
		    YYERROR;
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  reinit_parse_for_method (yychar, $$); }
	| dummy_decl notype_declarator
		{ $$ = start_method (NULL_TREE, $2);
		  if (! $$)
		    YYERROR;
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  reinit_parse_for_method (yychar, $$); }
	;

return_id: RETURN IDENTIFIER
	{
	  if (! current_function_parms_stored)
	    store_parm_decls ();
	  $$ = $2;
	}
	;

return_init: return_id opt.init
		{
		  extern tree value_identifier;
		  tree result;

		  result = DECL_RESULT (current_function_decl);
		  if (DECL_NAME (result) == value_identifier)
		    DECL_NAME (result) = $1;
		  else
		    error ("return identifier `%s' already in place",
			   IDENTIFIER_POINTER (DECL_NAME (result)));
		  store_return_init ($2);
		}
	| return_id '(' exprlist ')'
		{
		  extern tree value_identifier;
		  tree result;

		  result = DECL_RESULT (current_function_decl);
		  if (DECL_NAME (result) == value_identifier)
		    DECL_NAME (result) = $1;
		  else
		    error ("return identifier `%s' already in place",
			   IDENTIFIER_POINTER (DECL_NAME (result)));
		  store_return_init ($3);
		}
	;

base_init:
	  ':' .set_base_init member_init_list
		{ if ($3 == 0)
		    error ("no base initializers given following ':'");
		  setup_vtbl_ptr (); }
	| ':' error
		{ setup_vtbl_ptr (); }
	;

.set_base_init:
	/* empty */
		{
		  char *errmsg = NULL;

		  if (current_class_name == NULL_TREE)
		    errmsg = "base initializers not allowed here";
		  else if (! DECL_CONSTRUCTOR_P (current_function_decl))
		    errmsg = "only constructors take base initializers";

		  if (! current_function_parms_stored)
		    store_parm_decls ();

		  if (DECL_CONSTRUCTOR_P (current_function_decl))
		    {
		      /* Make a contour for the initializer list.  */
		      pushlevel (0);
		      clear_last_expr ();
		      expand_start_bindings (0);
		    }

		  if (errmsg)
		    {
		      error (errmsg);
		      yyerrstatus = 3;
		      YYERROR;
		    }
		}
	;

member_init_list:
	  /* empty */
		{ $$ = 0; }
	| member_init
		{ $$ = 1; }
	| member_init_list ',' member_init
	;

member_init:
	  '(' exprlist ')'
		{
#if 0
		  warning ("old style base class initialization; use `%s (...)'",
			   IDENTIFIER_POINTER (current_class_name));
#endif
		  if ($2 == NULL_TREE)
		    $2 = void_type_node;
		  expand_member_init (C_C_D, NULL_TREE, $2);
		}
	| identifier '(' exprlist ')'
		{
		  if ($3 == NULL_TREE)
		    $3 = void_type_node;
		  expand_member_init (C_C_D, $1, $3);
		}
	| scoped_declarator identifier '(' exprlist ')'
		{
		  tree basetype, base;
		  tree member, scopes = $1;
		  if ($4 == NULL_TREE)
		    $4 = void_type_node;
		  if (TREE_CODE (scopes) == SCOPE_REF)
		    /* just a pain to do this right now.  */
		    abort ();

		  if (! is_aggr_typedef_or_else (scopes))
		    break;
		  basetype = get_base_type (TREE_TYPE (TREE_TYPE (scopes)),
					    current_class_type, 1);
		  if (basetype == error_mark_node)
		    break;
		  if (basetype == 0)
		    {
		      error_not_derived_type (TREE_TYPE (TREE_TYPE (scopes)), current_class_type);
		      break;
		    }

		  base = convert_to_nonzero_pointer (basetype, current_class_decl);
		  expand_member_init (build_indirect_ref (base), $2, $4);
		}
	;

identifier:
	  IDENTIFIER
	| TYPENAME
	;

identifier_or_opname:
	  IDENTIFIER
	| TYPENAME
	| operator_name
		{ $$ = hack_operator ($1); }
	| wrapper IDENTIFIER
		{ $$ = hack_wrapper ($1, NULL_TREE, $2); }
	| wrapper TYPENAME
		{ $$ = hack_wrapper ($1, NULL_TREE, $2); }
	| wrapper operator_name
		{ $$ = hack_wrapper ($1, NULL_TREE, $2); }
	| wrapper scoped_declarator IDENTIFIER
		{ $$ = hack_wrapper ($1, $2, $3); }
	| wrapper scoped_declarator operator_name
		{ $$ = hack_wrapper ($1, $2, $3); }
	;

wrapper:  '(' ')'
		{ $$ = 0; }
	| '~' '(' ')'
		{ $$ = 1; }
	| '(' ')' '?'
		{ $$ = 2; }
	;

unop:     '-'
		{ $$ = NEGATE_EXPR; }
	| '+'
		{ $$ = CONVERT_EXPR; }
	| PLUSPLUS
		{ $$ = PREINCREMENT_EXPR; }
	| MINUSMINUS
		{ $$ = PREDECREMENT_EXPR; }
	| '!'
		{ $$ = TRUTH_NOT_EXPR; }
	;

expr:	nonnull_exprlist
		{ $$ = build_compound_expr ($1); }
	;

exprlist:
	  /* empty */
		{ $$ = NULL_TREE; }
	| nonnull_exprlist
	;

nonnull_exprlist:
	  expr_no_commas
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| nonnull_exprlist ',' expr_no_commas
		{ chainon ($1, build_tree_list (NULL_TREE, $3)); }
	| nonnull_exprlist ',' error
		{ chainon ($1, build_tree_list (NULL_TREE, error_mark_node)); }
	;

unary_expr:
	  primary %prec UNARY
		{
		  if (TREE_CODE ($1) == TYPE_EXPR)
		    $$ = build_component_type_expr (C_C_D, $1, NULL_TREE, 1);
		  else
		    $$ = $1;
		}
	| '*' cast_expr   %prec UNARY
		{ $$ = build_x_indirect_ref ($2, "unary *"); }
	| '&' cast_expr   %prec UNARY
		{ $$ = build_x_unary_op (ADDR_EXPR, $2); }
	| '~' cast_expr   %prec UNARY
		{ $$ = build_x_unary_op (BIT_NOT_EXPR, $2); }
	| unop cast_expr  %prec UNARY
		{ $$ = build_x_unary_op ($1, $2);
		  if ($1 == NEGATE_EXPR && TREE_CODE ($2) == INTEGER_CST)
		    TREE_NEGATED_INT ($$) = 1;
		}
		;

cast_expr:
	  unary_expr
	| '(' typename ')' expr_no_commas  %prec UNARY
		{ tree type = groktypename ($2);
		  $$ = build_c_cast (type, $4); }
	| '(' typename ')' '{' initlist maybecomma '}'  %prec UNARY
		{ tree type = groktypename ($2);
		  if (pedantic)
		    warning ("ANSI C forbids constructor-expressions");
		  $$ = digest_init (type, build_nt (CONSTRUCTOR, NULL_TREE, nreverse ($5)), 0);
		  if (TREE_CODE (type) == ARRAY_TYPE && TYPE_SIZE (type) == 0)
		    {
		      int failure = complete_array_type (type, $$, 1);
		      assert (!failure);
		    }
		}
		;

expr_no_commas:
	  cast_expr
	| SIZEOF expr_no_commas  %prec UNARY
		{ if (TREE_CODE ($2) == COMPONENT_REF
		      && TREE_PACKED (TREE_OPERAND ($2, 1)))
		    error ("sizeof applied to a bit-field");
		  $$ = cplus_sizeof (TREE_TYPE ($2)); }
	| SIZEOF '(' typename ')'  %prec HYPERUNARY
		{ $$ = cplus_sizeof (groktypename ($3)); }
	| ALIGNOF expr_no_commas  %prec UNARY
		{ if (TREE_CODE ($2) == COMPONENT_REF
		      && TREE_PACKED (TREE_OPERAND ($2, 1)))
		    error ("__alignof applied to a bit-field");
		  $$ = c_alignof (TREE_TYPE ($2)); }
	| ALIGNOF '(' typename ')'  %prec HYPERUNARY
		{ $$ = c_alignof (groktypename ($3)); }

	/* C++ extension.  */
	| DELETE expr_no_commas  %prec UNARY
		{ tree expr = convert_from_reference ($2);
		  tree type = TREE_TYPE (expr);

		  if (TREE_CODE (type) != POINTER_TYPE)
		    {
		      error ("non-pointer type to `delete'");
		      $$ = error_mark_node;
		    }
		  else
		    $$ = build_delete (type, expr, integer_one_node, 1);
		}
	| DELETE '[' expr ']' expr_no_commas  %prec UNARY
		{
		  tree maxindex = build_binary_op (MINUS_EXPR, $3, integer_one_node);
		  tree exp = convert_from_reference ($5);
		  tree elt_size = cplus_sizeof (TREE_TYPE (exp));

		  if (yychar == YYEMPTY)
		    yychar = YYLEX;

		  if (yychar == ';')
		    {
		      /* This is an optimization that allows vector
			 delete to be called as an inline function.  */
		      expand_vec_delete (exp, maxindex, elt_size, NULL_TREE,
					 integer_one_node, integer_zero_node);
		      $$ = error_mark_node;
		    }
		  else
		    {
		      $$ = build_vec_delete (exp, maxindex, elt_size, NULL_TREE,
					     integer_one_node, integer_zero_node);
		    }
		}

	| expr_no_commas '+' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '-' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '*' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '/' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '%' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas LSHIFT expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas RSHIFT expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas ARITHCOMPARE expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas EQCOMPARE expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas MIN_MAX expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '&' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '|' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas '^' expr_no_commas
		{ $$ = build_x_binary_op ($2, $1, $3); }
	| expr_no_commas ANDAND expr_no_commas
		{ $$ = build_x_binary_op (TRUTH_ANDIF_EXPR, $1, $3); }
	| expr_no_commas OROR expr_no_commas
		{ $$ = build_x_binary_op (TRUTH_ORIF_EXPR, $1, $3); }
	| expr_no_commas '?' xexpr ':' expr_no_commas
		{ $$ = build_x_conditional_expr ($1, $3, $5); }
	| expr_no_commas '=' expr_no_commas
		{ $$ = build_modify_expr ($1, NOP_EXPR, $3); }
	| expr_no_commas ASSIGN expr_no_commas
		{ register tree rval;
		  if (rval = build_opfncall (MODIFY_EXPR, $1, $3, $2))
		    $$ = rval;
		  else
		    $$ = build_modify_expr ($1, $2, $3); }

	/* Handle general members.  */
	| object '*' expr_no_commas   %prec UNARY
		{ $$ = build_m_component_ref ($1, build_x_indirect_ref ($3, "unary *")); }
	| object '&' expr_no_commas   %prec UNARY
		{ $$ = build_m_component_ref ($1, build_x_unary_op (ADDR_EXPR, $3)); }
	| object '~' expr_no_commas   %prec UNARY
		{ $$ = build_m_component_ref ($1, build_x_unary_op (BIT_NOT_EXPR, $3)); }
	| object unop expr_no_commas  %prec UNARY
		{ $$ = build_m_component_ref ($1, build_x_unary_op ($2, $3)); }
	| object '(' typename ')' expr_no_commas  %prec UNARY
		{ tree type = groktypename ($3);
		  $$ = build_m_component_ref ($1, build_c_cast (type, $5)); }
	| object primary_no_id  %prec UNARY
		{ $$ = build_m_component_ref ($1, $2); }
	;

primary:
	IDENTIFIER
		{ $$ = lastiddecl;
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  if (!$$ || $$ == error_mark_node)
		    {
		      if (yychar == '(')
			{
			  $$ = implicitly_declare ($1);
			  assemble_external ($$);
			  TREE_USED ($$) = 1;
			}
		      else if (current_function_decl == 0)
			{
			  error ("`%s' undeclared, outside of functions",
				 IDENTIFIER_POINTER ($1));
			  $$ = error_mark_node;
			}
		      else
			{
			  if (IDENTIFIER_GLOBAL_VALUE ($1) != error_mark_node
			      || IDENTIFIER_ERROR_LOCUS ($1) != current_function_decl)
			    {
			      error ("`%s' was not declared (first use this function)",
				     IDENTIFIER_POINTER ($1));
			      if (! undeclared_variable_notice)
				{
				  error ("(Each undeclared identifier is reported only once");
				  error ("for each function it appears in.)");
				  undeclared_variable_notice = 1;
				}
			    }
			  $$ = error_mark_node;
			  /* Prevent repeated error messages.  */
			  IDENTIFIER_GLOBAL_VALUE ($1) = error_mark_node;
			  IDENTIFIER_ERROR_LOCUS ($1) = current_function_decl;
			}
		    }
		  /* TREE_USED is set in `hack_identifier'.  */
		  if (TREE_CODE ($$) == CONST_DECL)
		    $$ = DECL_INITIAL ($$);
		  else $$ = hack_identifier ($$, $1, yychar);
		}
	| operator_name
		{ $$ = hack_operator ($1); }
	| CONSTANT
	| string
		{ $$ = combine_strings ($1); }
	| '(' expr ')'
		{ $$ = $2; }
	| '(' error ')'
		{ $$ = error_mark_node; }
	| '('
		{ if (current_function_decl == 0)
		    {
		      error ("braced-group within expression allowed only inside a function");
		      YYERROR;
		    }
		  $<ttype>$ = expand_start_stmt_expr (); }
	  compstmt ')'
		{ if (pedantic)
		    warning ("ANSI C forbids braced-groups within expressions");
		  $$ = expand_end_stmt_expr ($<ttype>2); }
	| primary '(' exprlist ')'
		{ $$ = build_x_function_call ($1, $3, current_class_decl); }
	| primary '[' expr ']'
		{
		do_array:
		  {
		    tree type = TREE_TYPE ($1);
		    if (type == error_mark_node)
		      $$ = error_mark_node;
		    else if (type == NULL_TREE)
		      {
			/* Something has gone very wrong.  Assume we
			   are mistakenly reducing an expression
			   instead of a declaration.  */
			error ("parser may be lost: is there a '{' missing somewhere?");
			$$ = NULL_TREE;
		      }
		    else
		      {
			if (TREE_CODE (type) == OFFSET_TYPE)
			  type = TREE_TYPE (type);
			if (TREE_CODE (type) == REFERENCE_TYPE)
			  type = TREE_TYPE (type);

			if (IS_AGGR_TYPE (type) &&
			    (TYPE_OVERLOADS_ARRAY_REF(TYPE_MAIN_VARIANT (type))))
			  $$ = build_opfncall (ARRAY_REF, $1, $3);
			else if (TREE_CODE(type) == POINTER_TYPE
				 || TREE_CODE(type) == ARRAY_TYPE)
			  $$ = build_array_ref ($1, $3);
			else
			  error("[] applied to non-pointer type");
		      }
		  }
		}
	| object identifier_or_opname  %prec UNARY
		{ $$ = build_component_ref ($1, $2, NULL_TREE, 1); }
	| object scoped_declarator identifier_or_opname %prec UNARY
		{
		do_scoped_member:
		  {
		    tree basetype = (TREE_CODE ($2) == SCOPE_REF) ? TREE_OPERAND ($2, 1) : $2;
		    if (is_aggr_typedef_or_else (basetype))
		      {
			basetype = TREE_TYPE (TREE_TYPE (basetype));

			if ($1 == error_mark_node)
			  $$ = error_mark_node;
			else if (basetype == TYPE_MAIN_VARIANT (TREE_TYPE ($1))
				 || get_base_type (basetype, TYPE_MAIN_VARIANT (TREE_TYPE ($1)), 0))
			  {
			    $$ = build_component_ref (build_scoped_ref ($1, $2), $3, NULL_TREE, 1);
			  }
			else
			  {
			    error_not_derived_type (basetype, TREE_TYPE ($1));
			    $$ = error_mark_node;
			  }
		      }
		    else $$ = error_mark_node;
		  }
		}
	| primary PLUSPLUS
		{ $$ = build_x_unary_op (POSTINCREMENT_EXPR, $1); }
	| primary MINUSMINUS
		{ $$ = build_x_unary_op (POSTDECREMENT_EXPR, $1); }

	/* C++ extensions */
	| THIS
		{ if (current_class_decl)
		    {
#ifdef WARNING_ABOUT_CCD
		      TREE_USED (current_class_decl) = 1;
#endif
		      $$ = current_class_decl;
		    }
		  else if (current_function_decl
			   && DECL_STATIC_FUNCTION_P (current_function_decl))
		    {
		      error ("`this' is unavailable for static member functions");
		      $$ = error_mark_node;
		    }
		  else
		    {
		      error ("request for `this' not in a class, struct, or union");
		      $$ = error_mark_node;
		    }
		}
	| dummy_decl TYPE_QUAL '(' exprlist ')'
		{
		do_functional_cast:
		  {
		    tree type;
		    tree id = $2;
		    int i;

		    /* This is a C cast in C++'s `functional' notation.  */
		    if ($4 == error_mark_node)
		      {
			$$ = error_mark_node;
			break;
		      }
		    if ($4 == NULL_TREE)
		      {
			error ("cannot cast null list to type `%s'",
			       IDENTIFIER_POINTER (TYPE_NAME ($2)));
			$$ = error_mark_node;
			break;
		      }
		    if (type == error_mark_node)
		      $$ = error_mark_node;
		    else
		      {
			if (id == ridpointers[(int) RID_CONST])
			  type = build_type_variant (integer_type_node, 1, 0);
			else if (id == ridpointers[(int) RID_VOLATILE])
			  type = build_type_variant (integer_type_node, 0, 1);
			else if (id == ridpointers[(int) RID_FRIEND])
			  {
			    error ("cannot cast expression to `friend' type");
			    $$ = error_mark_node;
			    break;
			  }
			else abort ();
			$$ = build_c_cast (type, build_compound_expr ($4));
		      }
		  }
		}
	| x_typespec '(' exprlist ')'
		{
		  do_functional_cast2:
		  {
		    /* This is either a call to a constructor,
		       or a C cast in C++'s `functional' notation.  */
		    tree tem, type;

		    if ($3 == error_mark_node)
		      {
			$$ = error_mark_node;
			break;
		      }
		    if (TREE_CODE ($1) == IDENTIFIER_NODE)
		      {
			if (! TREE_TYPE ($1))
			  {
			    type = lookup_name ($1);
			    if (!type || TREE_CODE (type) != TYPE_DECL)
			      {
				error ("`%s' fails to be a typedef or built-in type",
				       IDENTIFIER_POINTER ($1));
				$$ = error_mark_node;
				break;
			      }
			    type = TREE_TYPE (type);
			  }
			else
			  /* Either an enum or an aggregate type.  */
			  type = TREE_TYPE (TREE_TYPE ($1));
		      }
		    else type = $1;
		    if (! IS_AGGR_TYPE (type))
		      {
			/* this must build a C cast */
			if ($3 == NULL_TREE)
			  {
			    error ("cannot cast null list to type `%s'",
				   IDENTIFIER_POINTER ($1));
			    $$ = error_mark_node;
			    break;
			  }
			$$ = build_c_cast (type, build_compound_expr ($3));
		      }
		    else
		      {
			/* Call to a consructor.  If this expression
			   is actually used, for example,
			   
			   return X (arg1, arg2, ...);
			   
			   then the slot being initialized will be filled in.  */

			tree name = DECL_NAME (TYPE_NAME (type));
			if (! TYPE_NEEDS_CONSTRUCTOR (type))
			  {
			    error ("type `%s' does not have a constructor",
				   IDENTIFIER_POINTER (name));
			    $$ = error_mark_node;
			    break;
			  }
			else if (! TYPE_HAS_CONSTRUCTOR (type))
			  {
			    /* Look through this type until we find the
			       base type which has a constructor.  */
			    do
			      {
				int i, index = 0, win = 0;

				while (CLASSTYPE_N_BASECLASSES (type) == 1
				       && ! TYPE_HAS_CONSTRUCTOR (type))
				  type = CLASSTYPE_BASECLASS (type, 1);
				if (TYPE_HAS_CONSTRUCTOR (type))
				  break;
				/* Hack for MI.  */
				i = CLASSTYPE_N_BASECLASSES (type);
				while (i > 0)
				  {
				    if (TYPE_HAS_CONSTRUCTOR (CLASSTYPE_BASECLASS (type, i)))
				      {
					if (index == 0)
					  index = i;
					else
					  {
					    error ("multiple base classes with constructor, ambiguous");
					    type = 0;
					    break;
					  }
				      }
				    i -= 1;
				  }
				if (type == 0)
				  break;
			      } while (! TYPE_HAS_CONSTRUCTOR (type));
			    if (type == 0)
			      {
				$$ = error_mark_node;
				break;
			      }
			    name = DECL_NAME (TYPE_NAME (type));
			  }
			$$ = build_method_call (NULL_TREE, name, $3,
						NULL_TREE, LOOKUP_NORMAL);
			if ($$ != error_mark_node)
			  {
			    $$ = build (NEW_EXPR, type,
					TREE_OPERAND ($$, 0), TREE_OPERAND ($$, 1));
			    TREE_ADDRESSABLE ($$) = 1;
			  }
		      }
		  }
		}
	| SCOPE IDENTIFIER
		{
		do_scoped_identifier:
		  $$ = IDENTIFIER_GLOBAL_VALUE ($2);
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  if (! $$)
		    {
		      if (yychar == '(')
			{
			  $$ = implicitly_declare ($2);
			}
		      else
			{
			  if (IDENTIFIER_GLOBAL_VALUE ($2) != error_mark_node)
			    error ("undeclared variable `%s' (first use here)",
				   IDENTIFIER_POINTER ($2));
			  $$ = error_mark_node;
			  /* Prevent repeated error messages.  */
			  IDENTIFIER_GLOBAL_VALUE ($2) = error_mark_node;
			}
		    }
		  else if (TREE_CODE ($$) == CONST_DECL)
		    $$ = DECL_INITIAL ($$);
		}
	| scoped_declarator identifier_or_opname  %prec HYPERUNARY
		{ $$ = build_member_ref ($1, $2, 0); }
	| scoped_declarator identifier_or_opname '(' exprlist ')'
		{ $$ = build_member_call ($1, $2, $4, 0); }
	| scoped_declarator '~' identifier
		{ $$ = build_member_ref ($1, $3, 1); }
	| scoped_declarator '~' identifier '(' exprlist ')'
		{ $$ = build_member_call ($1, $3, $5, 1); }

	| object identifier_or_opname '(' exprlist ')'
		{ $$ = build_method_call ($1, $2, $4, NULL_TREE,
					  (LOOKUP_NORMAL|LOOKUP_AGGR)); }
	| object scoped_declarator identifier_or_opname '(' exprlist ')'
		{
		  /* Because this syntactic form does not allow
		     a pointer to a base class to be `stolen',
		     we need not protect the drived->base conversion
		     that happens here.

		     @@ But we do have to check visibility privileges later.  */
		  tree basename = (TREE_CODE ($2) == SCOPE_REF) ? TREE_OPERAND ($2, 1) : $2;
		  /* This is how virtual function calls are avoided.  */
		  if ($1 != error_mark_node
		      && is_aggr_typedef_or_else (basename))
		    {
		      tree basetype = TREE_TYPE (TREE_TYPE (basename));

		      if (basetype == TYPE_MAIN_VARIANT (TREE_TYPE ($1))
			  || (basetype = get_base_type (basetype, TYPE_MAIN_VARIANT (TREE_TYPE ($1)), 0)))
			{
			  tree decl;

			  if (basetype == error_mark_node)
			    {
			      $$ = error_mark_node;
			      break;
			    }
			  if (TREE_CODE ($1) == INDIRECT_REF)
			    decl = convert (TYPE_POINTER_TO (basetype),
					    build_unary_op (ADDR_EXPR, $1, 0));
			  else
			    decl = build_scoped_ref ($1, $2);
			  $$ = build_method_call (decl, $3, $5, NULL_TREE,
						  LOOKUP_NORMAL|LOOKUP_NONVIRTUAL);
			}
		      else
			{
			  error_not_derived_type (TREE_TYPE (TREE_TYPE (basename)),
						  TREE_TYPE ($1));
			  $$ = error_mark_node;
			}
		    }
		  else $$ = error_mark_node;
		}

	| new typename %prec '='
		{ $$ = build_new ($1, $2, NULL_TREE); }
	| new x_typespec '(' exprlist ')'
		{ $$ = build_new ($1, $2, $4); }
	| new typename '=' init %prec '='
		{ $$ = build_new ($1, $2, $4); }
	| new '(' typename ')'
		{ $$ = build_new ($1, $3, NULL_TREE); }
	;

primary_no_id:
	  '(' expr ')'
		{ $$ = $2; }
	| '(' error ')'
		{ $$ = error_mark_node; }
	| '('
		{ if (current_function_decl == 0)
		    {
		      error ("braced-group within expression allowed only inside a function");
		      YYERROR;
		    }
		  $<ttype>$ = expand_start_stmt_expr (); }
	  compstmt ')'
		{ if (pedantic)
		    warning ("ANSI C forbids braced-groups within expressions");
		  $$ = expand_end_stmt_expr ($<ttype>2); }
	| primary_no_id '(' exprlist ')'
		{ $$ = build_x_function_call ($1, $3, current_class_decl); }
	| primary_no_id '[' expr ']'
		{ goto do_array; }
	| primary_no_id PLUSPLUS
		{ $$ = build_x_unary_op (POSTINCREMENT_EXPR, $1); }
	| primary_no_id MINUSMINUS
		{ $$ = build_x_unary_op (POSTDECREMENT_EXPR, $1); }
	| SCOPE IDENTIFIER
		{ goto do_scoped_identifier; }
	;

new:	  NEW
		{ $$ = NULL_TREE; }
	| NEW '{' nonnull_exprlist '}'
		{ $$ = $3; }
	| NEW DYNAMIC  %prec EMPTY
		{ $$ = void_type_node; }
	| NEW DYNAMIC '(' string ')'
		{ $$ = combine_strings ($4); }
	;

/* Produces a STRING_CST with perhaps more STRING_CSTs chained onto it.  */
string:
	  STRING
	| string STRING
		{ $$ = chainon ($1, $2); }
	;

nodecls:
		{ goto do_xdecls; }
	;
xdecls:
	  /* empty */
		{
		do_xdecls:
		  if (! current_function_parms_stored)
		    store_parm_decls ();
		  setup_vtbl_ptr ();
		}
	| decls
		{ warning ("Old style parameter specification frowned upon");
		  goto do_xdecls; }
	;

object:	  primary '.'
		{
		  if ($1 == error_mark_node)
		    $$ = error_mark_node;
		  else
		    {
		      tree type = TREE_TYPE ($1);

		      if (IS_AGGR_TYPE (type)
			  || (TREE_CODE (type) == REFERENCE_TYPE
			      && IS_AGGR_TYPE (TREE_TYPE (type))))
			$$ = $1;
		      else
			{
			  error ("object in '.' expression is not of aggregate type");
			  $$ = error_mark_node;
			}
		    }
		}
	| primary POINTSAT
		{
		  $$ = build_x_arrow ($1);
		}
	;

decls:
	  decl
	| errstmt
	| decls decl
	| decl errstmt
	;

decl:
	  typed_declspecs initdecls ';'
		{ resume_momentary ($2); }
	| declmods notype_initdecls ';'
		{ resume_momentary ($2); }
	| typed_declspecs ';'
		{ shadow_tag ($1); }
	| declmods ';'
		{ warning ("empty declaration"); }
	;

/* Any kind of declarator (thus, all declarators allowed
   after an explicit typespec).  */

declarator:
	  after_type_declarator
	| notype_declarator
	;

/* Declspecs which contain at least one type specifier or typedef name.
   (Just `const' or `volatile' is not enough.)
   A typedef'd name following these is taken as a name to be declared.  */

typed_declspecs:
	  x_typespec
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| declmods typespec
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	| x_typespec reserved_declspecs
		{ $$ = tree_cons (NULL_TREE, $1, $2); }
	| declmods typespec reserved_declspecs
		{ $$ = tree_cons (NULL_TREE, $2, chainon ($3, $1)); }
	;

reserved_declspecs:  /* empty
		{ $$ = NULL_TREE; } */
	  typespecqual_reserved
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| SCSPEC
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| reserved_declspecs typespecqual_reserved
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	| reserved_declspecs SCSPEC
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	;

/* List of just storage classes and type modifiers.
   A declaration can start with just this, but then it cannot be used
   to redeclare a typedef-name.  */

declmods:
	  dummy_decl TYPE_QUAL
		{ $$ = build_tree_list (NULL_TREE, $2); }
	| dummy_decl SCSPEC
		{ $$ = build_tree_list (NULL_TREE, $2); }
	| declmods TYPE_QUAL
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	| declmods SCSPEC
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	;


/* Used instead of declspecs where storage classes are not allowed
   (that is, for typenames and structure components).

   C++ can takes storage classes for structure components.
   Don't accept a typedef-name if anything but a modifier precedes it.  */

typed_typespecs:
	  x_typespec  %prec EMPTY
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| nonempty_type_quals typespec
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	| x_typespec reserved_typespecquals
		{ $$ = tree_cons (NULL_TREE, $1, $2); }
	| nonempty_type_quals typespec reserved_typespecquals
		{ $$ = tree_cons (NULL_TREE, $2, chainon ($3, $1)); }
	;

reserved_typespecquals:
	  typespecqual_reserved
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| reserved_typespecquals typespecqual_reserved
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	;

/* A typespec (but not a type qualifier).
   Once we have seen one of these in a declaration,
   if a typedef name appears then it is being redeclared.  */

typespec: TYPESPEC
	| structsp
	| TYPENAME
	| TYPEOF '(' expr ')'
		{ $$ = TREE_TYPE ($3);
		  if (pedantic)
		    warning ("ANSI C forbids `typeof'"); }
	| TYPEOF '(' typename ')'
		{ $$ = groktypename ($3);
		  if (pedantic)
		    warning ("ANSI C forbids `typeof'"); }
	;

/* A typespec that is a reserved word, or a type qualifier.  */

typespecqual_reserved: TYPESPEC
	| TYPE_QUAL
	| structsp
	;

x_typespec:
	  dummy_decl TYPESPEC
		{ $$ = $2; }
	| dummy_decl structsp
		{ $$ = $2; }
	| dummy_decl TYPENAME
		{ $$ = $2; }
	| dummy_decl TYPEOF '(' expr ')'
		{ $$ = TREE_TYPE ($4);
		  if (pedantic)
		    warning ("ANSI C forbids `typeof'") }
	| dummy_decl TYPEOF '(' typename ')'
		{ $$ = groktypename ($4);
		  if (pedantic)
		    warning ("ANSI C forbids `typeof'") }
	;

initdecls:
	  initdcl0
	| initdecls ',' initdcl
	;

notype_initdecls:
	  notype_initdcl0
	| notype_initdecls ',' initdcl
	;

maybeasm:
	  /* empty */
		{ $$ = NULL_TREE; }
	| ASM '(' string ')'
		{ if (TREE_CHAIN ($3)) $3 = combine_strings ($3);
		  $$ = $3;
		  if (pedantic)
		    warning ("ANSI C forbids use of `asm' keyword");
		}
	;

initdcl0:
	  declarator maybeasm '='
		{ current_declspecs = $<ttype>0;
		  $<itype>3 = suspend_momentary ();
		  $<ttype>$ = start_decl ($1, current_declspecs, 1); }
	  init
/* Note how the declaration of the variable is in effect while its init is parsed! */
		{ finish_decl ($<ttype>4, $5, $2 ? TREE_STRING_POINTER ($2) : 0);
		  $$ = $<itype>3; }
	| declarator maybeasm
		{ tree d;
		  current_declspecs = $<ttype>0;
		  $$ = suspend_momentary ();
		  d = start_decl ($1, current_declspecs, 0);
		  finish_decl (d, NULL_TREE, $2 ? TREE_STRING_POINTER ($2) : 0);
		}
	;

initdcl:
	  declarator maybeasm '='
		{ $<ttype>$ = start_decl ($1, current_declspecs, 1); }
	  init
/* Note how the declaration of the variable is in effect while its init is parsed! */
		{ finish_decl ($<ttype>4, $5, $2 ? TREE_STRING_POINTER ($2) : 0); }
	| declarator maybeasm
		{ tree d = start_decl ($1, current_declspecs, 0);
		  finish_decl (d, NULL_TREE, $2 ? TREE_STRING_POINTER ($2) : 0);
		}
	;

notype_initdcl0:
	  notype_declarator maybeasm '='
		{ current_declspecs = $<ttype>0;
		  $<itype>3 = suspend_momentary ();
		  $<ttype>$ = start_decl ($1, current_declspecs, 1); }
	  init
/* Note how the declaration of the variable is in effect while its init is parsed! */
		{ finish_decl ($<ttype>4, $5, $2 ? TREE_STRING_POINTER ($2) : 0);
		  $$ = $<itype>3; }
	| notype_declarator maybeasm
		{ tree d;
		  current_declspecs = $<ttype>0;
		  $$ = suspend_momentary ();
		  d = start_decl ($1, current_declspecs, 0);
		  finish_decl (d, NULL_TREE, $2 ? TREE_STRING_POINTER ($2) : 0);
		}
	;

init:
	  expr_no_commas %prec '='
	| '{' '}'
		{ $$ = build_nt (CONSTRUCTOR, NULL_TREE, NULL_TREE);
		  if (pedantic)
		    warning ("ANSI C forbids empty initializer braces"); }
	| '{' initlist '}'
		{ $$ = build_nt (CONSTRUCTOR, NULL_TREE, nreverse ($2)); }
	| '{' initlist ',' '}'
		{ $$ = build_nt (CONSTRUCTOR, NULL_TREE, nreverse ($2)); }
	| error
		{ $$ = NULL_TREE; }
	;

/* This chain is built in reverse order,
   and put in forward order where initlist is used.  */
initlist:
	  init
		{ $$ = build_tree_list (NULL_TREE, $1); }
	| initlist ',' init
		{ $$ = tree_cons (NULL_TREE, $3, $1); }
	;

structsp:
	  ENUM identifier '{'
		{ $<itype>3 = suspend_momentary ();
		  $$ = start_enum ($2); }
	  enumlist maybecomma_warn '}'
		{ $$ = finish_enum ($<ttype>4, $5);
		  resume_momentary ($<itype>1);
		  check_for_missing_semicolon ($<ttype>4); }
	| ENUM '{'
		{ $<itype>2 = suspend_momentary ();
		  $$ = start_enum (make_anon_name ()); }
	  enumlist maybecomma_warn '}'
		{ $$ = finish_enum ($<ttype>3, $4);
		  resume_momentary ($<itype>1);
		  check_for_missing_semicolon ($<ttype>3); }
	| ENUM identifier
		{ $$ = xref_tag (enum_type_node, $2, NULL_TREE); }

	/* C++ extensions, merged with C to avoid shift/reduce conflicts */
	| class_head LC opt.component_decl_list '}'
		{
		  if (TREE_CODE ($1) == ENUMERAL_TYPE)
		    $$ = $1;
		  else
		    $$ = finish_struct ($1, $3, 0, 0);
		    
		  if ($2 & 1)
		    resume_temporary_allocation ();
		  if ($2 & 2)
		    resume_momentary (1);
		  check_for_missing_semicolon ($$); }
	| class_head LC opt.component_decl_list '}' ';'
		{ extern FILE *finput;

		  if (TREE_CODE ($1) == ENUMERAL_TYPE)
		    $$ = $1;
		  else
		    $$ = finish_struct ($1, $3, 1, 1);
		  if ($2 & 1)
		    resume_temporary_allocation ();
		  if ($2 & 2)
		    resume_momentary (1);
		  ungetc (';', finput); }
	| class_head  %prec EMPTY
		{ $$ = $1; }
	;

maybecomma:
	  /* empty */
	| ','
	;

maybecomma_warn:
	  /* empty */
	| ','
		{ if (pedantic) warning ("comma at end of enumerator list"); }
	;

aggr:	  AGGR
		{ $$ = $1; }
	| DYNAMIC AGGR
		{ $$ = build_tree_list (NULL_TREE, $2); }
	| DYNAMIC '(' string ')' AGGR
		{ $$ = build_tree_list ($3, $5); }
	| aggr SCSPEC
		{ error ("storage class specifier `%s' not allow after struct or class", IDENTIFIER_POINTER ($2));
		}
	| aggr TYPESPEC
		{ error ("storage class specifier `%s' not allow after struct or class", IDENTIFIER_POINTER ($2));
		}
	| aggr TYPE_QUAL
		{ error ("storage class specifier `%s' not allow after struct or class", IDENTIFIER_POINTER ($2));
		}
	| aggr AGGR
		{ error ("no body nor ';' separates two class, struct or union declarations");
		}

class_head:
	  aggr  %prec EMPTY
		{ $$ = xref_tag ($1, make_anon_name (), NULL_TREE); }
	| aggr identifier  %prec EMPTY
		{ $$ = xref_tag ($1, $2, NULL_TREE); }
	| aggr IDENTIFIER ':' base_class_list %prec EMPTY
		{ $$ = xref_tag ($1, $2, $4); }
	| aggr TYPENAME_COLON base_class_list %prec EMPTY
		{ $$ = xref_tag ($1, $2, $3); }
	;

base_class_list:
	  identifier
		{ if (! is_aggr_typedef_or_else ($1))
		    $$ = NULL_TREE;
		  else $$ = build_tree_list ((tree)visibility_default, $1); }
	| base_class_visibility_list identifier
		{ if (! is_aggr_typedef_or_else ($2))
		    $$ = NULL_TREE;
		  else $$ = build_tree_list ($1, $2); }
	| base_class_list ',' identifier
		{ if (! is_aggr_typedef_or_else ($3))
		    $$ = NULL_TREE;
		  else $$ = chainon ($1, build_tree_list ((tree)visibility_default, $3)); }
	| base_class_list ',' base_class_visibility_list identifier
		{ if (! is_aggr_typedef_or_else ($4))
		    $$ = NULL_TREE;
		  else $$ = chainon ($1, build_tree_list ($3, $4)); }
	;

base_class_visibility_list:
	  PUBLIC
		{ $$ = (tree)visibility_public; }
	| PRIVATE
		{ $$ = (tree)visibility_private; }
	| SCSPEC
		{ if ($1 != ridpointers[(int)RID_VIRTUAL])
		    sorry ("non-virtual visibility");
		  $$ = (tree)visibility_default_virtual; }
	| base_class_visibility_list PUBLIC
		{ if ($1 == (tree)visibility_private)
		    error ("base class cannot be public and private");
		  else if ($1 == (tree)visibility_default_virtual)
		    $$ = (tree)visibility_public_virtual; }
	| base_class_visibility_list PRIVATE
		{ if ($1 == (tree)visibility_public)
		    error ("base class cannot be private and public");
		  else if ($1 == (tree)visibility_default_virtual)
		    $$ = (tree)visibility_private_virtual; }
	| base_class_visibility_list SCSPEC
		{ if ($2 != ridpointers[(int)RID_VIRTUAL])
		    sorry ("non-virtual visibility");
		  if ($1 == (tree)visibility_public)
		    $$ = (tree)visibility_public_virtual;
		  else if ($1 == (tree)visibility_private)
		    $$ = (tree)visibility_private_virtual; }
	;

LC: '{'
		{ int temp = allocation_temporary_p ();
		  int momentary = suspend_momentary ();
		  if (temp)
		    end_temporary_allocation ();
		  $$ = (momentary << 1) | temp;
		  pushclass ($<ttype>0, 0); }

opt.component_decl_list:
	/* empty */
		{ $$ = NULL_TREE; }
	| component_decl_list
		{ $$ = build_tree_list ((tree)visibility_default, $1); }
	| opt.component_decl_list PUBLIC ':' component_decl_list
		{ $$ = chainon ($1, build_tree_list ((tree)visibility_public, $4)); }
	| opt.component_decl_list PRIVATE ':' component_decl_list
		{ $$ = chainon ($1, build_tree_list ((tree)visibility_private, $4)); }
	| opt.component_decl_list PROTECTED ':' component_decl_list
		{ $$ = chainon ($1, build_tree_list ((tree)visibility_protected, $4)); }
	| opt.component_decl_list PUBLIC ':'
	| opt.component_decl_list PRIVATE ':'
	| opt.component_decl_list PROTECTED ':'
	;

component_decl_list:
	  component_decl
		{ if ($1 == void_type_node) $$ = NULL_TREE; }
	| component_decl_list component_decl
		{ if ($2 != NULL_TREE && $2 != void_type_node)
		    $$ = chainon ($1, $2); }
	| component_decl_list ';'
		{ if (pedantic)
		    warning ("extra semicolon in struct or union specified"); }
	;

component_decl:
	  typed_declspecs components ';'
		{
		do_components:
		  if ($2 == void_type_node)
		    /* We just got some friends.
		       They have been recorded elsewhere.  */
		    $$ = NULL_TREE;
		  else if ($2 == NULL_TREE)
		    {
		      tree t = groktypename (build_tree_list ($1, NULL_TREE));
		      if (t == NULL_TREE)
			{
			  yylineerror (@1.first_line,
				       "error in component specification");
			  $$ = NULL_TREE;
			}
		      else if (TREE_CODE (t) == UNION_TYPE)
			{
			  /* handle anonymous unions */
			  if (CLASSTYPE_FN_FIELDS (t))
			    sorry ("methods in anonymous unions");
			  $$ = build_lang_decl (FIELD_DECL, NULL_TREE, t);
			  DECL_ANON_UNION_ELEM ($$) = 1;
			}
		      else if (TREE_CODE (t) == ENUMERAL_TYPE)
			$$ = grok_enum_decls (t, NULL_TREE);
		      else
			$$ = NULL_TREE;
		    }
		  else
		    {
		      tree t = TREE_TYPE ($2);
		      if (TREE_CODE (t) == ENUMERAL_TYPE && TREE_NONLOCAL (t))
			$$ = grok_enum_decls (t, $2);
		      else
			$$ = $2;
		    }
		}
	| typed_declspecs '(' parmlist ')' ';'
		{
		do_type_component:
		  $$ = groktypefield ($1, $3);
		}
	| declmods components ';'
		{ goto do_components; }
	| declmods '(' parmlist ')' ';'
		{ goto do_type_component; }
	| ':' expr_no_commas ';'
		{ $$ = grokfield (NULL_TREE, NULL_TREE, $2, NULL_TREE, NULL_TREE); }
	| error
		{ $$ = NULL_TREE; }

	/* C++: handle constructors, destructors and inline functions */
	/* note that INLINE is like a TYPESPEC */
	| fn.def2 ':' /* base_init compstmt */
		{ $$ = finish_method ($1); }
	| fn.def2 '{' /* xdecls compstmt */
		{ $$ = finish_method ($1); }
	| dummy_decl notype_declarator ';'
		{ $$ = grokfield ($2, NULL_TREE, NULL_TREE, NULL_TREE, NULL_TREE); }
/*
	| dummy_decl TYPENAME '(' parmlist ')' ';'
		{
		  tree decl = build_nt (CALL_EXPR, $2, $4);
		  $$ = grokfield (decl, NULL_TREE, NULL_TREE, NULL_TREE, NULL_TREE);
		}
*/
	;

components:
	  /* empty: possibly anonymous */
		{ $$ = NULL_TREE; }
	| component_declarator0
	| components ',' component_declarator
		{
		  /* In this context, void_type_node encodes
		     friends.  They have been recorded elsewhere.  */
		  if ($1 == void_type_node)
		    $$ = $3;
		  else
		    $$ = chainon ($1, $3);
		}
	;

component_declarator0:
	  declarator maybeasm opt.init
		{ current_declspecs = $<ttype>0;
		  $$ = grokfield ($1, current_declspecs, NULL_TREE, $3,
				  $2 ? TREE_STRING_POINTER ($2) : 0); }
	| IDENTIFIER ':' expr_no_commas
		{ current_declspecs = $<ttype>0;
		  $$ = grokfield ($1, current_declspecs, $3, NULL_TREE, NULL_TREE); }
	| TYPENAME_COLON expr_no_commas
		{ current_declspecs = $<ttype>0;
		  $$ = grokfield ($1, current_declspecs, $2, NULL_TREE, NULL_TREE); }
	| ':' expr_no_commas
		{ $$ = grokfield (NULL_TREE, NULL_TREE, $2, NULL_TREE, NULL_TREE); }
	;

component_declarator:
	  declarator maybeasm opt.init
		{ $$ = grokfield ($1, current_declspecs, NULL_TREE, $3,
				  $2 ? TREE_STRING_POINTER ($2) : 0); }
	| IDENTIFIER ':' expr_no_commas
		{ $$ = grokfield ($1, current_declspecs, $3, NULL_TREE, NULL_TREE); }
	| TYPENAME_COLON expr_no_commas
		{ $$ = grokfield ($1, current_declspecs, $2, NULL_TREE, NULL_TREE); }
	| ':' expr_no_commas
		{ $$ = grokfield (NULL_TREE, NULL_TREE, $2, NULL_TREE, NULL_TREE); }
	;

/* We chain the enumerators in reverse order.
   Because of the way enums are built, the order is
   insignificant.  Take advantage of this fact.  */

enumlist:
	  enumerator
	| enumlist ',' enumerator
		{ TREE_CHAIN ($3) = $1; $$ = $3; }
	;


enumerator:
	  identifier
		{ $$ = build_enumerator ($1, NULL_TREE); }
	| identifier '=' expr_no_commas
		{ $$ = build_enumerator ($1, $3); }
	;

typename:
	  typed_typespecs absdcl
		{ $$ = build_tree_list ($1, $2); }
	| nonempty_type_quals absdcl
		{ $$ = build_tree_list ($1, $2); }
	;

absdcl:   /* an abstract declarator */
	/* empty */ %prec EMPTY
		{ $$ = NULL_TREE; }
	| absdcl1  %prec EMPTY
	;

nonempty_type_quals:
	  dummy_decl TYPE_QUAL
		{ $$ = build_tree_list (NULL_TREE, $2); }
	| nonempty_type_quals TYPE_QUAL
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	;

type_quals:
	  /* empty */
		{ $$ = NULL_TREE; }
	| type_quals TYPE_QUAL
		{ $$ = tree_cons (NULL_TREE, $2, $1); }
	;

/* These rules must follow the rules for function declarations
   and component declarations.  That way, longer rules are prefered.  */

/* A declarator that is allowed only after an explicit typespec.  */
/* may all be followed by prec '.' */
after_type_declarator:
	  after_type_declarator '(' nonnull_exprlist ')'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| after_type_declarator '(' parmlist ')'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| after_type_declarator '(' error ')'
		{ $$ = build_nt (CALL_EXPR, $1, NULL_TREE); }
	| after_type_declarator '[' expr ']'
		{ $$ = build_nt (ARRAY_REF, $1, $3); }
	| after_type_declarator '[' ']'
		{ $$ = build_nt (ARRAY_REF, $1, NULL_TREE); }
	| '(' dummy_decl after_type_declarator_no_typename ')'
		{ $$ = $3; }
	| '(' '*' type_quals after_type_declarator ')'
		{ $$ = make_pointer_declarator ($3, $4); }
	| PAREN_STAR_PAREN
		{ $$ = $1;
		  see_typename (); }
	| PAREN_X_SCOPE_STAR_PAREN
		{ $$ = $1;
		  see_typename (); }
	| PAREN_X_SCOPE_REF_PAREN
		{ $$ = $1;
		  see_typename (); }
	| '(' '&' type_quals after_type_declarator ')'
		{ $$ = make_reference_declarator ($3, $4); }
	| '*' type_quals after_type_declarator  %prec UNARY
		{ $$ = make_pointer_declarator ($2, $3); }
	| '&' type_quals after_type_declarator  %prec UNARY
		{ $$ = make_reference_declarator ($2, $3); }
	| TYPENAME
	;

after_type_declarator_no_typename:
	  after_type_declarator_no_typename '(' nonnull_exprlist ')'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| after_type_declarator_no_typename '(' parmlist ')'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| after_type_declarator_no_typename '(' error ')'
		{ $$ = build_nt (CALL_EXPR, $1, NULL_TREE); }
	| after_type_declarator_no_typename '[' expr ']'
		{ $$ = build_nt (ARRAY_REF, $1, $3); }
	| after_type_declarator_no_typename '[' ']'
		{ $$ = build_nt (ARRAY_REF, $1, NULL_TREE); }
	| '(' dummy_decl after_type_declarator_no_typename ')'
		{ $$ = $3; }
	| PAREN_STAR_PAREN
		{ $$ = $1;
		  see_typename (); }
	| PAREN_X_SCOPE_STAR_PAREN
		{ $$ = $1;
		  see_typename (); }
	| PAREN_X_SCOPE_REF_PAREN
		{ $$ = $1;
		  see_typename (); }
	| '*' type_quals after_type_declarator  %prec UNARY
		{ $$ = make_pointer_declarator ($2, $3); }
	| '&' type_quals after_type_declarator  %prec UNARY
		{ $$ = make_reference_declarator ($2, $3); }
	;

/* A declarator allowed whether or not there has been
   an explicit typespec.  These cannot redeclare a typedef-name.  */

notype_declarator:
	  notype_declarator '(' nonnull_exprlist ')'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| notype_declarator '(' parmlist ')'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| notype_declarator '(' error ')'
		{ $$ = build_nt (CALL_EXPR, $1, NULL_TREE); }
	| '(' notype_declarator ')'
		{ $$ = $2; }
	| '*' type_quals notype_declarator  %prec UNARY
		{ $$ = make_pointer_declarator ($2, $3); }
	| '&' type_quals notype_declarator  %prec UNARY
		{ $$ = make_reference_declarator ($2, $3); }
	| notype_declarator '[' expr ']'
		{ $$ = build_nt (ARRAY_REF, $1, $3); }
	| notype_declarator '[' ']'
		{ $$ = build_nt (ARRAY_REF, $1, NULL_TREE); }
	| IDENTIFIER
		{ see_typename (); }

	/* C++ extensions.  */
	| operator_name
		{ see_typename (); }

	| '~' TYPENAME
		{ see_typename ();
		  $$ = build_nt (BIT_NOT_EXPR, $2); }
	| '~' IDENTIFIER
		{ see_typename ();
		  $$ = build_nt (BIT_NOT_EXPR, $2); }
	| '(' parmlist ')' TYPENAME
		{
		  see_typename ();
		  if ($2 && (TREE_CHAIN ($2) || TREE_VALUE ($2) != void_type_node))
		    error ("junk in wrapper declaration");
		  $$ = build_nt (WRAPPER_EXPR, $4);
		}
	| '(' parmlist ')' IDENTIFIER
		{
		  see_typename ();
		  if ($2 && (TREE_CHAIN ($2) || TREE_VALUE ($2) != void_type_node))
		    error ("junk in wrapper declaration");
		  $$ = build_nt (WRAPPER_EXPR, $4);
		}
	| '(' parmlist ')' '?' TYPENAME
		{
		  see_typename ();
		  if ($2 && (TREE_CHAIN ($2) || TREE_VALUE ($2) != void_type_node))
		    error ("junk in wrapper predicate declaration");
		  $$ = build_nt (WRAPPER_EXPR,
				 build_nt (COND_EXPR, $5, NULL_TREE, NULL_TREE));
		}
	| '(' parmlist ')' '?' IDENTIFIER
		{
		  see_typename ();
		  if ($2 && (TREE_CHAIN ($2) || TREE_VALUE ($2) != void_type_node))
		    error ("junk in wrapper predicate declaration");
		  $$ = build_nt (WRAPPER_EXPR,
				 build_nt (COND_EXPR, $5, NULL_TREE, NULL_TREE));
		}
	| '~' '(' ')' TYPENAME
		{ see_typename ();
		  $$ = build_nt (ANTI_WRAPPER_EXPR, $4); }
	| '~' '(' ')' IDENTIFIER
		{ see_typename ();
		  $$ = build_nt (ANTI_WRAPPER_EXPR, $4); }
	| TYPENAME_SCOPE type_quals notype_declarator  %prec '('
		{ see_typename ();
		  $$ = build_nt (SCOPE_REF, $1, $3); }
	| IDENTIFIER SCOPE type_quals notype_declarator  %prec '('
		{ see_typename ();
		  $$ = build_nt (SCOPE_REF, $1, $4); }
	| TYPENAME_SCOPE TYPENAME  %prec '('
		{ $$ = build_nt (SCOPE_REF, $1, $2); }
	| IDENTIFIER SCOPE TYPENAME  %prec '('
		{ $$ = build_nt (SCOPE_REF, $1, $3); }
	| TYPENAME_SCOPE type_quals see_typename TYPENAME '(' nonnull_exprlist ')'
		{ $$ = build_nt (SCOPE_REF, $1,
				 build_nt (CALL_EXPR, $4, $6)); }
	| IDENTIFIER SCOPE type_quals see_typename TYPENAME '(' nonnull_exprlist ')'
		{ $$ = build_nt (SCOPE_REF, $1,
				 build_nt (CALL_EXPR, $5, $7)); }
	| TYPENAME_SCOPE type_quals see_typename TYPENAME '(' parmlist ')'
		{ $$ = build_nt (SCOPE_REF, $1,
				 build_nt (CALL_EXPR, $4, $6)); }
	| IDENTIFIER SCOPE type_quals see_typename TYPENAME '(' parmlist ')'
		{ $$ = build_nt (SCOPE_REF, $1,
				 build_nt (CALL_EXPR, $5, $7)); }
	| TYPENAME_SCOPE type_quals see_typename TYPENAME '(' error ')'
		{ $$ = build_nt (SCOPE_REF, $1, build_nt (CALL_EXPR, $4, NULL_TREE)); }
	| IDENTIFIER SCOPE type_quals see_typename TYPENAME '(' error ')'
		{ $$ = build_nt (SCOPE_REF, $1, build_nt (CALL_EXPR, $5, NULL_TREE)); }
	;

scoped_declarator:
	  TYPENAME_SCOPE
	| IDENTIFIER SCOPE
	| scoped_declarator TYPENAME_SCOPE
		{ $$ = build_nt (SCOPE_REF, $1, $2); }
	;

absdcl1:  /* a nonempty abstract declarator */
	  '(' absdcl1 ')'
		{ see_typename ();
		  $$ = $2; }
	  /* `(typedef)1' is `int'.  */
	| '*' type_quals absdcl1  %prec EMPTY
		{ $$ = make_pointer_declarator ($2, $3); }
	| '*' type_quals  %prec EMPTY
		{ $$ = make_pointer_declarator ($2, NULL_TREE); }
	| PAREN_STAR_PAREN
		{ $$ = $1;
		  see_typename (); }
	| PAREN_X_SCOPE_STAR_PAREN
		{ $$ = $1;
		  see_typename (); }
	| PAREN_X_SCOPE_REF_PAREN
		{ $$ = $1;
		  see_typename (); }
	| '&' type_quals absdcl1 %prec EMPTY
		{ $$ = make_reference_declarator ($2, $3); }
	| '&' type_quals %prec EMPTY
		{ $$ = make_reference_declarator ($2, NULL_TREE); }
	| absdcl1 '(' parmlist ')'  %prec '.'
		{ $$ = build_nt (CALL_EXPR, $1, $3); }
	| absdcl1 '[' expr ']'  %prec '.'
		{ $$ = build_nt (ARRAY_REF, $1, $3); }
	| absdcl1 '[' ']'  %prec '.'
		{ $$ = build_nt (ARRAY_REF, $1, NULL_TREE); }
	| '(' parmlist ')'  %prec '.'
		{ $$ = build_nt (CALL_EXPR, NULL_TREE, $2); }
	| '[' expr ']'  %prec '.'
		{ $$ = build_nt (ARRAY_REF, NULL_TREE, $2); }
	| '[' ']'  %prec '.'
		{ $$ = build_nt (ARRAY_REF, NULL_TREE, NULL_TREE); }
	| TYPENAME_SCOPE type_quals absdcl1  %prec EMPTY
		{ $$ = build_nt (SCOPE_REF, $1, $3); }
	| IDENTIFIER SCOPE type_quals absdcl1  %prec EMPTY
		{ $$ = build_nt (SCOPE_REF, $1, $4); }
	| TYPENAME_SCOPE type_quals %prec EMPTY
		{ $$ = build_nt (SCOPE_REF, $1, 0); }
	| IDENTIFIER SCOPE type_quals %prec EMPTY
		{ $$ = build_nt (SCOPE_REF, $1, 0); }
	;

/* at least one statement, the first of which parses without error.  */
/* stmts is used only after decls, so an invalid first statement
   is actually regarded as an invalid decl and part of the decls.  */

stmts:
	  stmt
	| stmts stmt
	;

errstmt:  error ';'
	;

/* build the LET_STMT node before parsing its contents,
  so that any LET_STMTs within the context can have their display pointers
  set up to point at this one.  */

.pushlevel:  /* empty */
		{ pushlevel (0);
		  clear_last_expr ();
		  push_momentary ();
		  expand_start_bindings (0);
		  stmt_decl_ok = 1;
		}
	;

/* This is the body of a function definition.
   It causes syntax errors to ignore to the next openbrace.  */
compstmt_or_error:
	  compstmt
	| error compstmt
	;

compstmt: '{' '}'
	| '{' .pushlevel stmts '}'
		{ expand_end_bindings (getdecls (), 1, 1);
		  poplevel (1, 1, 0);
		  pop_momentary (); }
	| '{' .pushlevel error '}'
		{ expand_end_bindings (getdecls (), 0, 1);
		  poplevel (0, 0, 0);
		  pop_momentary (); }
	;

simple_if:
	  IF '(' expr ')'
		{ emit_line_note (input_filename, lineno);
		  expand_start_cond (truthvalue_conversion ($3), 0);
		  stmt_decl_ok = 0; }
	  stmt
	;

stmt:
	  compstmt
		{ finish_stmt (); }
	| decl
		{ if (stmt_decl_ok == 0)
		    error ("declaration(s) in conditional statement invalid");
		  finish_stmt (); }
	| expr ';'
		{ emit_line_note (input_filename, lineno);
		  /* Do default conversion if safe and possibly important,
		     in case within ({...}).  */
		  if ((TREE_CODE (TREE_TYPE ($1)) == ARRAY_TYPE
		       && lvalue_p ($1))
		      || TREE_CODE (TREE_TYPE ($1)) == FUNCTION_TYPE)
		    $1 = default_conversion ($1);
		  expand_cplus_expr_stmt ($1);
		  clear_momentary ();
		  finish_stmt (); }
	| simple_if ELSE
		{ expand_start_else ();
		  stmt_decl_ok = 0; }
	  stmt
		{ expand_end_else ();
		  stmt_decl_ok = 1;
		  finish_stmt (); }
	| simple_if %prec IF
		{ expand_end_cond ();
		  stmt_decl_ok = 1;
		  finish_stmt (); }
	| WHILE
		{ emit_nop ();
		  emit_line_note (input_filename, lineno);
		  expand_start_loop (1); }
	  '(' expr ')'
		{ expand_exit_loop_if_false (truthvalue_conversion ($4)); }
	  stmt
		{ expand_end_loop ();
		  finish_stmt (); }
	| DO
		{ emit_nop ();
		  emit_line_note (input_filename, lineno);
		  expand_start_loop_continue_elsewhere (1); }
	  stmt WHILE
		{ expand_loop_continue_here (); }
	  '(' expr ')' ';'
		{ emit_line_note (input_filename, lineno);
		  expand_exit_loop_if_false (truthvalue_conversion ($7));
		  expand_end_loop ();
		  clear_momentary ();
		  finish_stmt (); }
	| forhead.1
		{ emit_nop ();
		  emit_line_note (input_filename, lineno);
		  if ($1) expand_cplus_expr_stmt ($1);
		  expand_start_loop_continue_elsewhere (1); }
	  xexpr ';'
		{ emit_line_note (input_filename, lineno);
		  if ($3) expand_exit_loop_if_false (truthvalue_conversion ($3)); }
	  xexpr ')'
		/* Don't let the tree nodes for $6 be discarded
		   by clear_momentary during the parsing of the next stmt.  */
		{ push_momentary (); }
	  stmt
		{ emit_line_note (input_filename, lineno);
		  expand_loop_continue_here ();
		  if ($6) expand_cplus_expr_stmt ($6);
		  pop_momentary ();
		  expand_end_loop ();
		  finish_stmt (); }
	| forhead.2
		{ emit_nop ();
		  emit_line_note (input_filename, lineno);
		  expand_start_loop_continue_elsewhere (1); }
	  xexpr ';'
		{ emit_line_note (input_filename, lineno);
		  if ($3) expand_exit_loop_if_false (truthvalue_conversion ($3)); }
	  xexpr ')'
		/* Don't let the tree nodes for $6 be discarded
		   by clear_momentary during the parsing of the next stmt.  */
		{ push_momentary ();
		  $<itype>7 = lineno; }
	  stmt
		{ emit_line_note (input_filename, $<itype>7);
		  expand_loop_continue_here ();
		  if ($6) expand_cplus_expr_stmt ($6);
		  pop_momentary ();
		  expand_end_loop ();
		  if ($1)
		    {
		      register keep = $1 > 0;
		      if (keep) expand_end_bindings (0, keep, 1);
		      poplevel (keep, 1, 0);
		      pop_momentary ();
		    }
		  finish_stmt ();
		}
	| SWITCH '(' expr ')'
		{ emit_line_note (input_filename, lineno);
		  c_expand_start_case ($3);
		  /* Don't let the tree nodes for $3 be discarded by
		     clear_momentary during the parsing of the next stmt.  */
		  push_momentary (); }
	  stmt
		{ expand_end_case ($3);
		  pop_momentary ();
		  finish_stmt (); }
	| CASE expr ':'
		{ register tree value = $2;
		  register tree label
		    = build_decl (LABEL_DECL, NULL_TREE, NULL_TREE);

		  /* build_c_cast puts on a NOP_EXPR to make a non-lvalue.
		     Strip such NOP_EXPRs.  */
		  if (TREE_CODE (value) == NOP_EXPR
		      && TREE_TYPE (value) == TREE_TYPE (TREE_OPERAND (value, 0)))
		    value = TREE_OPERAND (value, 0);

		  if (TREE_READONLY (value) && TREE_CODE (value) == VAR_DECL)
		    {
		      value = decl_constant_value (value);
		      /* build_c_cast puts on a NOP_EXPR to make a non-lvalue.
			 Strip such NOP_EXPRs.  */
		      if (TREE_CODE (value) == NOP_EXPR
			  && TREE_TYPE (value) == TREE_TYPE (TREE_OPERAND (value, 0)))
			value = TREE_OPERAND (value, 0);
		    }
		  value = fold (value);

		  if (TREE_CODE (value) != INTEGER_CST
		      && value != error_mark_node)
		    {
		      error ("case label does not reduce to an integer constant");
		      value = error_mark_node;
		    }
		  else
		    /* Promote char or short to int.  */
		    value = default_conversion (value);
		  if (value != error_mark_node)
		    {
		      int success = pushcase (value, label);
		      if (success == 1)
			error ("case label not within a switch statement");
		      else if (success == 2)
			error ("duplicate case value");
		      else if (success == 3)
			warning ("case value out of range");
		    }
		}
	  stmt
	| CASE expr RANGE expr ':'
		{ register tree value1 = $2;
		  register tree value2 = $4;
		  register tree label
		    = build_decl (LABEL_DECL, NULL_TREE, NULL_TREE);

		  if (pedantic)
		    {
		      error ("ANSI C does not allow range expressions in switch statement");
		      value1 = error_mark_node;
		      value2 = error_mark_node;
		      break;
		    }
		  /* build_c_cast puts on a NOP_EXPR to make a non-lvalue.
		     Strip such NOP_EXPRs.  */
		  if (TREE_CODE (value1) == NOP_EXPR
		      && TREE_TYPE (value1) == TREE_TYPE (TREE_OPERAND (value1, 0)))
		    value1 = TREE_OPERAND (value1, 0);

		  if (TREE_READONLY (value1) && TREE_CODE (value1) == VAR_DECL)
		    {
		      value1 = decl_constant_value (value1);
		      /* build_c_cast puts on a NOP_EXPR to make a non-lvalue.
			 Strip such NOP_EXPRs.  */
		      if (TREE_CODE (value1) == NOP_EXPR
			  && TREE_TYPE (value1) == TREE_TYPE (TREE_OPERAND (value1, 0)))
			value1 = TREE_OPERAND (value1, 0);
		    }
		  value1 = fold (value1);

		  /* build_c_cast puts on a NOP_EXPR to make a non-lvalue.
		     Strip such NOP_EXPRs.  */
		  if (TREE_CODE (value2) == NOP_EXPR
		      && TREE_TYPE (value2) == TREE_TYPE (TREE_OPERAND (value2, 0)))
		    value2 = TREE_OPERAND (value2, 0);

		  if (TREE_READONLY (value2) && TREE_CODE (value2) == VAR_DECL)
		    {
		      value2 = decl_constant_value (value2);
		      /* build_c_cast puts on a NOP_EXPR to make a non-lvalue.
			 Strip such NOP_EXPRs.  */
		      if (TREE_CODE (value2) == NOP_EXPR
			  && TREE_TYPE (value2) == TREE_TYPE (TREE_OPERAND (value2, 0)))
			value2 = TREE_OPERAND (value2, 0);
		    }
		  value2 = fold (value2);


		  if (TREE_CODE (value1) != INTEGER_CST
		      && value1 != error_mark_node)
		    {
		      error ("case label does not reduce to an integer constant");
		      value1 = error_mark_node;
		    }
		  if (TREE_CODE (value2) != INTEGER_CST
		      && value2 != error_mark_node)
		    {
		      error ("case label does not reduce to an integer constant");
		      value2 = error_mark_node;
		    }
		  if (value1 != error_mark_node
		      && value2 != error_mark_node)
		    {
		      int success = pushcase_range (value1, value2, label);
		      if (success == 1)
			error ("case label not within a switch statement");
		      else if (success == 2)
			error ("duplicate (or overlapping) case value");
		      else if (success == 3)
			warning ("case value out of range");
		      else if (success == 4)
			warning ("empty range specified");
		    }
		}
	  stmt
	| DEFAULT ':'
		{
		  register tree label
		    = build_decl (LABEL_DECL, NULL_TREE, NULL_TREE);
		  int success = pushcase (NULL_TREE, label);
		  if (success == 1)
		    error ("default label not within a switch statement");
		  else if (success == 2)
		    error ("multiple default labels in one switch");
		}
	  stmt
	| BREAK ';'
		{ emit_line_note (input_filename, lineno);
		  if ( ! expand_exit_something ())
		    error ("break statement not within loop or switch"); }
	| CONTINUE ';'
		{ emit_line_note (input_filename, lineno);
		  if (! expand_continue_loop ())
		    error ("continue statement not within a loop"); }
	| RETURN ';'
		{ emit_line_note (input_filename, lineno);
		  c_expand_return (NULL_TREE); }
	| RETURN expr ';'
		{ emit_line_note (input_filename, lineno);
		  c_expand_return ($2);
		  finish_stmt ();
		}
	| ASM maybe_type_qual '(' string ')' ';'
		{ if (TREE_CHAIN ($4)) $4 = combine_strings ($4);
		  emit_line_note (input_filename, lineno);
		  expand_asm ($4);
		  finish_stmt ();
		}
	/* This is the case with just output operands.  */
	| ASM maybe_type_qual '(' string ':' asm_operands ')' ';'
		{ if (TREE_CHAIN ($4)) $4 = combine_strings ($4);
		  c_expand_asm_operands ($4, $6, NULL_TREE, NULL_TREE,
					 $2 == ridpointers[(int)RID_VOLATILE],
					 input_filename, lineno);
		  finish_stmt ();
		}
	/* This is the case with input operands as well.  */
	| ASM maybe_type_qual '(' string ':' asm_operands ':' asm_operands ')' ';'
		{ if (TREE_CHAIN ($4)) $4 = combine_strings ($4);
		  emit_line_note (input_filename, lineno);
		  c_expand_asm_operands ($4, $6, $8, NULL_TREE,
					 $2 == ridpointers[(int)RID_VOLATILE],
					 input_filename, lineno);
		  finish_stmt ();
		}
	/* This is the case with clobbered registers as well.  */
	| ASM maybe_type_qual '(' string ':' asm_operands ':'
  	  asm_operands ':' asm_clobbers ')' ';'
		{ if (TREE_CHAIN ($4)) $4 = combine_strings ($4);
		  emit_line_note (input_filename, lineno);
		  c_expand_asm_operands ($4, $6, $8, $10,
					 $2 == ridpointers[(int)RID_VOLATILE],
					 input_filename, lineno);
		  finish_stmt ();
		}
	| GOTO identifier ';'
		{ tree decl;
		  emit_line_note (input_filename, lineno);
		  decl = lookup_label ($2);
		  TREE_USED (decl) = 1;
		  expand_goto (decl); }
	| IDENTIFIER ':'
		{ tree label = define_label (input_filename, lineno, $1);
		  emit_nop ();
		  if (label)
		    expand_label (label); }
	  stmt
		{ finish_stmt (); }
	| TYPENAME_COLON
		{ tree label = define_label (input_filename, lineno, $1);
		  if (label)
		    expand_label (label); }
	  stmt
		{ finish_stmt (); }
	| ';'
		{ finish_stmt (); }
	;

forhead.1:
	  FOR '(' ';'
		{ $$ = NULL_TREE; }
	| FOR '(' expr ';'
		{ $$ = $3; }
	| FOR '(' '{' '}'
		{ $$ = NULL_TREE; }
	;

forhead.2:
	  FOR '(' decl
		{ $$ = 0; }
	| FOR '(' '{' .pushlevel stmts '}'
		{ $$ = 1; }
	| FOR '(' '{' .pushlevel error '}'
		{ $$ = -1; }
	;

/* Either a type-qualifier or nothing.  First thing in an `asm' statement.  */

maybe_type_qual:
	/* empty */
		{ if (pedantic)
		    warning ("ANSI C forbids use of `asm' keyword");
		  /* We use emit_note rather than emit_line_note
		     so that this note is emitted even if line #s
		     are not generally wanted.  */
		  emit_note (input_filename, lineno); }
	| TYPE_QUAL
		{ if (pedantic)
		    warning ("ANSI C forbids use of `asm' keyword");
		  emit_note (input_filename, lineno); }
	;

xexpr:
	/* empty */
		{ $$ = NULL_TREE; }
	| expr
	| error
		{ $$ = NULL_TREE; }
	;

/* These are the operands other than the first string and colon
   in  asm ("addextend %2,%1": "=dm" (x), "0" (y), "g" (*x))  */
asm_operands: /* empty */
		{ $$ = NULL_TREE; }
	| nonnull_asm_operands
	;

nonnull_asm_operands:
	  asm_operand
	| nonnull_asm_operands ',' asm_operand
		{ $$ = chainon ($1, $3); }
	;

asm_operand:
	  STRING '(' expr ')'
		{ $$ = build_tree_list ($1, $3); }
	;

asm_clobbers:
	  STRING
		{ $$ = tree_cons (NULL_TREE, $1, NULL_TREE); }
	| asm_clobbers ',' STRING
		{ $$ = tree_cons (NULL_TREE, $3, $1); }
	;

/* This is what appears inside the parens in a function declarator.
   Its value is represented in the format that grokdeclarator expects.

   In C++, declaring a function with no parameters
   means that that function takes *no* parameters.  */
parmlist:  /* empty */
		{
		  if (strict_prototype)
		    {
		      $$ = build_tree_list (NULL_TREE, void_type_node);
		      TREE_PARMLIST($$) = 1;
		    }
		  else $$ = NULL_TREE;
		}
	| parms
  		{
		  $$ = chainon ($1, build_tree_list (NULL_TREE,
						     void_type_node));
		  TREE_PARMLIST ($$) = 1;
		}
	| parms ',' ELLIPSIS
		{
		  $$ = $1;
		  TREE_PARMLIST ($$) = 1;
		}
	/* C++ allows an ellipsis without a separating ',' */
	| parms ELLIPSIS
		{
		  $$ = $1;
		  TREE_PARMLIST ($$) = 1;
		}
	| ELLIPSIS
		{
		  warning ("lazy prototyping frowned upon");
		  $$ = NULL_TREE;
		}
	| TYPENAME_ELLIPSIS
		{
		  $$ = $1;
		  TREE_PARMLIST ($$) = 1;
		}
	| parms TYPENAME_ELLIPSIS
		{
		  $$ = $1;
		  TREE_PARMLIST ($$) = 1;
		}
	| parms ':'
		{
		  /* This helps us recover from really nasty
		     parse errors, for example, a missing right
		     parenthesis.  */
		  extern FILE *finput;

		  yyerror ("possibly missing ')'");
		  $$ = chainon ($1, build_tree_list (NULL_TREE,
						     void_type_node));
		  TREE_PARMLIST ($$) = 1;
		  ungetc (':', finput);
		  yychar = ')';
		}
	;

/* A nonempty list of parameter declarations or type names.  */
parms:
	  parm opt.init
		{ $$ = build_tree_list ($2, $1); }
	| parms ',' parm opt.init
		{ $$ = chainon ($1, build_tree_list ($4, $3)); }
	| parms ',' bad_parm opt.init
		{ $$ = chainon ($1, build_tree_list ($4, $3)); }
	;

/* A single parameter declaration or parameter type name,
   as found in a parmlist.  */
parm:
	  typed_declspecs dont_see_typename notype_declarator
		{ $$ = build_tree_list ($1, $3);
		  see_typename (); }
	| typed_declspecs dont_see_typename absdcl
		{ $$ = build_tree_list ($1, $3);
		  see_typename (); }
	| declmods dont_see_typename notype_declarator
		{ $$ = build_tree_list ($1, $3);
		  see_typename (); }
	| declmods dont_see_typename absdcl
		{ $$ = build_tree_list ($1, $3);
		  see_typename (); }
	;

see_typename: /* empty */
	{
	  see_typename ();
	}
dont_see_typename: /* empty */
	{
	  dont_see_typename ();
	}

bad_parm:
	  dummy_decl notype_declarator
		{
		  warning ("type specifier omitted for parameter");
		  $$ = build_tree_list (TREE_PURPOSE (TREE_VALUE ($<ttype>-1)), $2);
		}
	| dummy_decl absdcl
		{
		  warning ("type specifier omitted for parameter");
		  $$ = build_tree_list (TREE_PURPOSE (TREE_VALUE ($<ttype>-1)), $2);
		}
	;

	/* C++ extension: allow for initialization */
opt.init:
	  /* empty */
		{ $$ = NULL_TREE; }
	| '=' init
		{ $$ = $2; }
	;

operator_name:
	  OPERATOR '*'
		{ $$ = build_op_identifier (0, make_node (MULT_EXPR)); }
	| OPERATOR '/'
		{ $$ = build_op_identifier (0, make_node (TRUNC_DIV_EXPR)); }
	| OPERATOR '%'
		{ $$ = build_op_identifier (0, make_node (TRUNC_MOD_EXPR)); }
	| OPERATOR '+'
		{ $$ = build_op_identifier (0, make_node (PLUS_EXPR)); }
	| OPERATOR '-'
		{ $$ = build_op_identifier (0, make_node (MINUS_EXPR)); }
	| OPERATOR '&'
		{ $$ = build_op_identifier (0, make_node (BIT_AND_EXPR)); }
	| OPERATOR '|'
		{ $$ = build_op_identifier (0, make_node (BIT_IOR_EXPR)); }
	| OPERATOR '^'
		{ $$ = build_op_identifier (0, make_node (BIT_XOR_EXPR)); }
	| OPERATOR '~'
		{ $$ = build_op_identifier (0, make_node (BIT_NOT_EXPR)); }
	| OPERATOR ARITHCOMPARE
		{ $$ = build_op_identifier (0, make_node ($2)); }
	| OPERATOR EQCOMPARE
		{ $$ = build_op_identifier (0, make_node ($2)); }
	| OPERATOR ASSIGN
		{ $$ = build_op_identifier (make_node (MODIFY_EXPR),
					    make_node ($2)); }
	| OPERATOR '='
		{
		  $$ = build_op_identifier (make_node (MODIFY_EXPR),
					    make_node (NOP_EXPR));
		  if (current_class_type)
		    {
		      TREE_HAS_ASSIGNMENT (current_class_type) = 1;
		      TREE_GETS_ASSIGNMENT (current_class_type) = 1;
		    }
		}
	| OPERATOR LSHIFT
		{ $$ = build_op_identifier (0, make_node ($2)); }
	| OPERATOR RSHIFT
		{ $$ = build_op_identifier (0, make_node ($2)); }
	| OPERATOR PLUSPLUS
		{ $$ = build_op_identifier (0, make_node (POSTINCREMENT_EXPR)); }
	| OPERATOR MINUSMINUS
		{ $$ = build_op_identifier (0, make_node (PREDECREMENT_EXPR)); }
	| OPERATOR ANDAND
		{ $$ = build_op_identifier (0, make_node (TRUTH_ANDIF_EXPR)); }
	| OPERATOR OROR
		{ $$ = build_op_identifier (0, make_node (TRUTH_ORIF_EXPR)); }
	| OPERATOR '!'
		{ $$ = build_op_identifier (0, make_node (TRUTH_NOT_EXPR)); }
	| OPERATOR '?' ':'
		{ $$ = build_op_identifier (0, make_node (COND_EXPR)); }
	| OPERATOR MIN_MAX
		{ $$ = build_op_identifier (0, make_node ($2)); }
	| OPERATOR POINTSAT  %prec EMPTY
		{ $$ = build_op_identifier (0, make_node (COMPONENT_REF)); }
	| OPERATOR POINTSAT_LEFT_RIGHT
		{
		  if (yychar == YYEMPTY)
		    yychar = YYLEX;
		  if (yychar == '(')
		    {
		      $$ = build_op_identifier (0, make_node (METHOD_CALL_EXPR));
		      if (current_class_type)
			TYPE_OVERLOADS_METHOD_CALL_EXPR (current_class_type) = 1;
		    }
		  else
		    $$ = build_nt (CALL_EXPR, build_op_identifier (0, make_node (COMPONENT_REF)), NULL_TREE);
		}
	| OPERATOR '(' ')'
		{ $$ = build_op_identifier (0, make_node (CALL_EXPR));
		  if (current_class_type)
		    TYPE_OVERLOADS_CALL_EXPR (current_class_type) = 1;
		}
	| OPERATOR '[' ']'
		{ $$ = build_op_identifier (0, make_node (ARRAY_REF));
		  if (current_class_type)
		    TYPE_OVERLOADS_ARRAY_REF (current_class_type) = 1;
		}
	| OPERATOR NEW
		{
		  $$ = build_op_identifier (0, make_node (NEW_EXPR));
		  if (current_class_type)
		    TREE_GETS_NEW (current_class_type) = 1;
		}
	| OPERATOR DELETE
		{
		  $$ = build_op_identifier (0, make_node (DELETE_EXPR));
		  if (current_class_type)
		    TREE_GETS_DELETE (current_class_type) = 1;
		}
	| OPERATOR_PUSH
		{
		  $$ = build_op_identifier (0, make_node (PUSH_EXPR));
		}
	| OPERATOR_POP
		{
		  $$ = build_op_identifier (0, make_node (POP_EXPR));
		}

	/* These should do `groktypename' and set up TREE_HAS_X_CONVERSION
	   here, rather than doing it in class.c .  */
	| OPERATOR typed_typespecs absdcl
		{
		  $$ = build (TYPE_EXPR, $2, $3);
		}
	| OPERATOR error
		{ $$ = NULL_TREE; }
	;

%%
db_yyerror (s, yyps, yychar)
     char *s;
     short *yyps;
     int yychar;
{
  FILE *yyout;
  char buf[1024];
  int st;

  yyerror (s);
  printf ("State is %d, input token number is %d.\n", *yyps, yychar);

#ifdef PARSE_OUTPUT
  if (*yyps < 1) fatal ("Cannot start from here");
  else if ((yyout = fopen (PARSE_OUTPUT, "r")) == NULL)
    error ("cannot open file parse.output");
  else
    {
      printf ("That is to say,\n\n");
      while (fgets(buf, sizeof (buf)-1, yyout))
	{
	  if (buf[0] != 's') continue;
	  st = atoi (buf+6);
	  if (st != *yyps) continue;
	  printf ("%s", buf);
	  while (fgets (buf, sizeof (buf)-1, yyout))
	    {
	      if (buf[0] == 's') break;
	      printf ("%s", buf);
	    }
	  break;
	}
      printf ("With the token %s\n", yytname[YYTRANSLATE (yychar)]);
      fclose (yyout);
    }
#endif
}

void
yyerror (string)
     char *string;
{
  extern FILE *finput;
  extern char *token_buffer;
  char buf[200];

  strcpy (buf, string);

  /* We can't print string and character constants well
     because the token_buffer contains the result of processing escapes.  */
  if (end_of_file || feof (finput))
    strcat (buf, " at end of input");
  else if (token_buffer[0] == 0)
    strcat (buf, " at null character");
  else if (token_buffer[0] == '"')
    strcat (buf, " before string constant");
  else if (token_buffer[0] == '\'')
    strcat (buf, " before character constant");
  else
    strcat (buf, " before `%s'");

  error (buf, token_buffer);
}
