/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libepubgen/libepubgen.h>

#include <libwpd/libwpd.h>

#include "EpubPackage.hxx"

using namespace libwpd;

class EpubOutputFileHelper
{
public:
	EpubOutputFileHelper(const char *const outFileName) :
		m_fileName(outFileName)
	{
	}

	bool convertDocument(librevenge::RVNGInputStream &input)
	{
		EpubPackage package(m_fileName);
		libepubgen::EPUBTextGenerator collector(&package);

		return WPD_OK == WPDocument::parse(&input, &collector, "");
	}

	bool isSupportedFormat(librevenge::RVNGInputStream &input, const char *password = "")
	{
		WPDConfidence confidence = WPDocument::isFileFormatSupported(&input);
		if (WPD_CONFIDENCE_EXCELLENT != confidence && WPD_CONFIDENCE_SUPPORTED_ENCRYPTION != confidence)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid WordPerfect document.\n");
			return false;
		}
		if (WPD_CONFIDENCE_SUPPORTED_ENCRYPTION == confidence && !password)
		{
			fprintf(stderr, "ERROR: The WordPerfect document is encrypted and you did not give us a password.\n");
			return false;
		}
		if (confidence == WPD_CONFIDENCE_SUPPORTED_ENCRYPTION && password &&
		        (WPD_PASSWORD_MATCH_OK != WPDocument::verifyPassword(&input, password)))
		{
			fprintf(stderr, "ERROR: The WordPerfect document is encrypted and we either\n");
			fprintf(stderr, "ERROR: don't know how to decrypt it or the given password is wrong.\n");
			return false;
		}

		return true;
	}

private:
	std::string m_fileName;
};

int printUsage(char *name)
{
	fprintf(stderr, "USAGE : %s [--stdout] <infile> [outfile]\n", name);
	fprintf(stderr, "USAGE : Where <infile> is the WordPerfect source document\n");
	fprintf(stderr, "USAGE : and [outfile] is the ePub target document.\n");
	return 1;
}

int main(int argc, char *argv[])
{
	if ((argc < 2) || (argc > 3))
		return printUsage(argv[0]);

	char *szInputFile = argv[1];
	char *szOutFile = 0;

	if (argc > 2)
		szOutFile = argv[2];

	if (!szInputFile)
		return printUsage(argv[0]);

	librevenge::RVNGFileStream input(szInputFile);
	EpubOutputFileHelper helper(szOutFile);

	if (!helper.isSupportedFormat(input))
	{
		fprintf(stderr, "ERROR: the document is not in any supported format.\n");
		return 1;
	}

	if (!helper.convertDocument(input))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
