//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file polarisation/Pol01/src/DetectorConstruction.cc
/// \brief Implementation of the DetectorConstruction class
//
//
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "DetectorConstruction.hh"

#include "DetectorMessenger.hh"

#include "G4Box.hh"
#include "G4GeometryManager.hh"
#include "G4LogicalVolume.hh"
#include "G4LogicalVolumeStore.hh"
#include "G4Material.hh"
#include "G4NistManager.hh"
#include "G4PVPlacement.hh"
#include "G4PhysicalVolumeStore.hh"
#include "G4PolarizationManager.hh"
#include "G4RunManager.hh"
#include "G4SolidStore.hh"
#include "G4SystemOfUnits.hh"
#include "G4UnitsTable.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

DetectorConstruction::DetectorConstruction()
  : G4VUserDetectorConstruction(), fWorld(0), fBox(0), fTargetMaterial(0), fWorldMaterial(0)
{
  fBoxSizeXY = 50 * mm;
  fBoxSizeZ = 5 * mm;
  fWorldSize = 1. * m;
  ConstructMaterials();
  fMessenger = new DetectorMessenger(this);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

DetectorConstruction::~DetectorConstruction()
{
  delete fMessenger;
}
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorConstruction::ConstructMaterials()
{
  auto nistManager = G4NistManager::Instance();

  fTargetMaterial = nistManager->FindOrBuildMaterial("G4_Fe");
  if (fTargetMaterial == nullptr) {
    G4cerr << "### ERROR - Material: <"
           << "G4_Fe"
           << "> not found" << G4endl;
  }

  fWorldMaterial = nistManager->FindOrBuildMaterial("G4_Galactic");
  if (fWorldMaterial == nullptr) {
    G4cerr << "### ERROR -  Material: <"
           << "G4_Galactic"
           << "> not found" << G4endl;
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4LogicalVolume* lBox;

G4VPhysicalVolume* DetectorConstruction::Construct()
{
  // G4PolarizationManager::GetInstance()->Clean();

  // World
  //
  G4Box* sWorld = new G4Box("World",  // name
                            fWorldSize / 2, fWorldSize / 2, fWorldSize / 2);  // dimensions

  G4LogicalVolume* lWorld = new G4LogicalVolume(sWorld,  // shape
                                                fWorldMaterial,  // material
                                                "World");  // name

  fWorld = new G4PVPlacement(0,  // no rotation
                             G4ThreeVector(),  // at (0,0,0)
                             lWorld,  // logical volume
                             "World",  // name
                             0,  // mother volume
                             false,  // no boolean operation
                             0);  // copy number

  // Box
  //
  G4Box* sBox = new G4Box("Container",  // its name
                          fBoxSizeXY / 2., fBoxSizeXY / 2., fBoxSizeZ / 2.);  // its dimensions

  // G4LogicalVolume*
  lBox = new G4LogicalVolume(sBox,  // its shape
                             fTargetMaterial,  // its material
                             "theBox");  // its name

  fBox = new G4PVPlacement(0,  // no rotation
                           G4ThreeVector(),  // at (0,0,0)
                           lBox,  // its logical volume
                           fTargetMaterial->GetName(),  // its name
                           lWorld,  // its mother  volume
                           false,  // no boolean operation
                           0);  // copy number

  PrintParameters();

  // always return the root volume
  //
  return fWorld;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorConstruction::ConstructSDandField()
{
  // register logical Volume in PolarizationManager with zero polarization
  G4PolarizationManager* polMgr = G4PolarizationManager::GetInstance();
  polMgr->SetVolumePolarization(lBox, G4ThreeVector(0., 0., 0.));
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorConstruction::PrintParameters()
{
  G4cout << "\n The Box is " << G4BestUnit(fBoxSizeXY, "Length") << " x "
         << G4BestUnit(fBoxSizeXY, "Length") << " x " << G4BestUnit(fBoxSizeZ, "Length") << " of "
         << fTargetMaterial->GetName() << G4endl;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorConstruction::SetTargetMaterial(G4String materialChoice)
{
  // search the material by its name
  G4Material* mat = G4NistManager::Instance()->FindOrBuildMaterial(materialChoice);
  if (mat != fTargetMaterial) {
    if (mat) {
      fTargetMaterial = mat;
      G4RunManager::GetRunManager()->PhysicsHasBeenModified();
      UpdateGeometry();
    }
    else {
      G4cout << "### Warning!  Target material: <" << materialChoice << "> not found" << G4endl;
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorConstruction::SetWorldMaterial(G4String materialChoice)
{
  // search the material by its name
  G4Material* mat = G4NistManager::Instance()->FindOrBuildMaterial(materialChoice);
  if (mat != fWorldMaterial) {
    if (mat) {
      fWorldMaterial = mat;
      G4RunManager::GetRunManager()->PhysicsHasBeenModified();
      UpdateGeometry();
    }
    else {
      G4cout << "### Warning! World material: <" << materialChoice << "> not found" << G4endl;
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorConstruction::SetSizeXY(G4double value)
{
  fBoxSizeXY = value;
  if (fWorldSize < fBoxSizeXY) fWorldSize = 1.2 * fBoxSizeXY;
  UpdateGeometry();
}

void DetectorConstruction::SetSizeZ(G4double value)
{
  fBoxSizeZ = value;
  if (fWorldSize < fBoxSizeZ) fWorldSize = 1.2 * fBoxSizeZ;
  UpdateGeometry();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4RunManager.hh"

void DetectorConstruction::UpdateGeometry()
{
  // if (fWorld)
  //   G4RunManager::GetRunManager()->DefineWorldVolume(Construct());
  G4RunManager::GetRunManager()->GeometryHasBeenModified();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
