/* $OpenBSD: exportertest.c,v 1.3 2023/04/14 14:23:05 tb Exp $ */
/*
 * Copyright (c) 2022 Joel Sing <jsing@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <openssl/err.h>
#include <openssl/ssl.h>

#include <err.h>
#include <stdio.h>
#include <string.h>

#include "ssl_local.h"

static void
hexdump(const unsigned char *buf, size_t len)
{
	size_t i;

	for (i = 1; i <= len; i++)
		fprintf(stderr, " 0x%02hhx,%s", buf[i - 1], i % 8 ? "" : "\n");

	fprintf(stderr, "\n");
}

struct exporter_test {
	uint16_t tls_version;
	unsigned int cipher_id;
	const uint8_t *label;
	size_t label_len;
	const uint8_t context_value[64];
	size_t context_value_len;
	int use_context;
	const uint8_t client_random[SSL3_RANDOM_SIZE];
	const uint8_t server_random[SSL3_RANDOM_SIZE];
	const uint8_t master_key[SSL_MAX_MASTER_KEY_LENGTH];
	const uint8_t shared_key[64];
	size_t shared_key_len;
	const uint8_t export[64];
	size_t export_len;
	int want_error;
};

static const struct exporter_test exporter_tests[] = {
	{
		/* Valid export, no context - 32 bytes. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.use_context = 0,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export = {
			0x14, 0x08, 0x00, 0x9e, 0x6a, 0x67, 0x75, 0x4c,
			0xc4, 0xf3, 0x51, 0x57, 0x2f, 0x75, 0x0b, 0xf8,
			0x16, 0xfa, 0x61, 0x74, 0xd2, 0x12, 0x8f, 0x78,
			0x77, 0xf9, 0x8a, 0x3e, 0x58, 0x70, 0xf3, 0xd8,
		},
		.export_len = 32,
	},
	{
		/* Valid export, no context - 32 bytes. */
		.tls_version = TLS1_3_VERSION,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.use_context = 0,
		.shared_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.shared_key_len = 32,
		.export = {
			0x69, 0xf4, 0xac, 0xec, 0x80, 0x67, 0xac, 0x5c,
			0xa6, 0x24, 0x47, 0xb1, 0x0f, 0xc8, 0xa1, 0x13,
			0x3b, 0x91, 0x33, 0x82, 0x97, 0x0a, 0xc0, 0xbf,
			0xac, 0x6d, 0x6b, 0x34, 0x20, 0xd3, 0x3a, 0x02,
		},
		.export_len = 32,
	},
	{
		/* Valid export, no context - 64 bytes. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.use_context = 0,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export = {
			0x14, 0x08, 0x00, 0x9e, 0x6a, 0x67, 0x75, 0x4c,
			0xc4, 0xf3, 0x51, 0x57, 0x2f, 0x75, 0x0b, 0xf8,
			0x16, 0xfa, 0x61, 0x74, 0xd2, 0x12, 0x8f, 0x78,
			0x77, 0xf9, 0x8a, 0x3e, 0x58, 0x70, 0xf3, 0xd8,
			0xe8, 0xd2, 0xb7, 0xcd, 0xbc, 0x37, 0xdf, 0x16,
			0x12, 0xf1, 0xe8, 0xb2, 0x62, 0x79, 0x91, 0x45,
			0x77, 0xe0, 0x68, 0x6d, 0xd5, 0x31, 0x54, 0x55,
			0x22, 0x63, 0xc0, 0x36, 0x31, 0x07, 0xda, 0x33,
		},
		.export_len = 64,
	},
	{
		/* Valid export, no context - 64 bytes. */
		.tls_version = TLS1_3_VERSION,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.use_context = 0,
		.shared_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.shared_key_len = 32,
		.export = {
			0x77, 0x15, 0xe2, 0x07, 0x65, 0x64, 0x3b, 0x14,
			0x38, 0xcb, 0x73, 0x93, 0xda, 0x70, 0xfa, 0x86,
			0x2c, 0x34, 0xcc, 0x94, 0x52, 0xc2, 0xd3, 0xb4,
			0x59, 0x2c, 0xc8, 0x05, 0x70, 0xfe, 0x48, 0x61,
			0xd3, 0xea, 0x57, 0x66, 0xa9, 0x66, 0x2f, 0x4a,
			0x35, 0xc9, 0x88, 0x86, 0x28, 0x52, 0xe3, 0x64,
			0x5e, 0xf9, 0x28, 0x53, 0x8a, 0x3a, 0x92, 0x92,
			0x40, 0x8c, 0x89, 0x17, 0x59, 0xd0, 0xd0, 0x82,
		},
		.export_len = 64,
	},
	{
		/* Valid export, zero length context - 32 bytes. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.context_value_len = 0,
		.use_context = 1,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export = {
			0xdb, 0xc9, 0xdf, 0x7c, 0x04, 0x39, 0xdd, 0x23,
			0xc3, 0x68, 0xdc, 0xf3, 0x04, 0xcf, 0x4c, 0x4d,
			0x86, 0x5b, 0xe6, 0x48, 0xc5, 0x6d, 0xe5, 0x1e,
			0xea, 0xc5, 0xe4, 0x00, 0x27, 0x72, 0xda, 0xb6,
		},
		.export_len = 32,
	},
	{
		/* Valid export, zero length context - 32 bytes. */
		.tls_version = TLS1_3_VERSION,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.context_value_len = 0,
		.use_context = 1,
		.shared_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.shared_key_len = 32,
		.export = {
			0x69, 0xf4, 0xac, 0xec, 0x80, 0x67, 0xac, 0x5c,
			0xa6, 0x24, 0x47, 0xb1, 0x0f, 0xc8, 0xa1, 0x13,
			0x3b, 0x91, 0x33, 0x82, 0x97, 0x0a, 0xc0, 0xbf,
			0xac, 0x6d, 0x6b, 0x34, 0x20, 0xd3, 0x3a, 0x02,
		},
		.export_len = 32,
	},
	{
		/* Valid export, with context value - 32 bytes. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.context_value = {
			0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
			0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
		},
		.context_value_len = 16,
		.use_context = 1,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export = {
			0x0e, 0xb4, 0xd1, 0x3a, 0x0e, 0x24, 0xab, 0x0d,
			0x4c, 0x48, 0x35, 0x25, 0xf6, 0x4d, 0xa2, 0x9b,
			0xaa, 0x1d, 0xbc, 0x54, 0x7e, 0xb0, 0x3c, 0x4b,
			0x07, 0x04, 0x9c, 0x7c, 0x06, 0xa7, 0xea, 0x70,
		},
		.export_len = 32,
	},
	{
		/* Valid export, with context value - 32 bytes. */
		.tls_version = TLS1_3_VERSION,
		.label = "EXPERIMENTAL testing",
		.label_len = 20,
		.context_value = {
			0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
			0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
		},
		.context_value_len = 16,
		.use_context = 1,
		.shared_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.shared_key_len = 32,
		.export = {
			0x34, 0xb8, 0x00, 0x6a, 0xb2, 0x62, 0xab, 0xea,
			0xc7, 0x2b, 0x15, 0xa0, 0x85, 0xda, 0xaa, 0xa5,
			0x12, 0x85, 0xbf, 0x4a, 0xa4, 0x71, 0x42, 0xc8,
			0xd4, 0xa6, 0x66, 0x18, 0xc6, 0xc9, 0x26, 0x6f,
		},
		.export_len = 32,
	},
	{
		/* Valid export, with different label - 32 bytes. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = "EXPERIMENTAL more testing",
		.label_len = 20,
		.context_value = {
			0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
			0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
		},
		.context_value_len = 16,
		.use_context = 1,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export = {
			0xb0, 0xb6, 0x45, 0xdd, 0x30, 0x76, 0xf0, 0x57,
			0x22, 0x31, 0xbb, 0x8d, 0xe1, 0xf9, 0xe3, 0xed,
			0xae, 0x74, 0x6f, 0x40, 0x94, 0xf6, 0xc2, 0xfc,
			0x21, 0xff, 0xf7, 0x00, 0x86, 0x54, 0xb6, 0x06,
		},
		.export_len = 32,
	},
	{
		/* Valid export, with different label - 32 bytes. */
		.tls_version = TLS1_3_VERSION,
		.label = "EXPERIMENTAL more testing",
		.label_len = 20,
		.context_value = {
			0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
			0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
		},
		.context_value_len = 16,
		.use_context = 1,
		.shared_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.shared_key_len = 32,
		.export = {
			0x18, 0x4e, 0x65, 0x3c, 0x91, 0x5d, 0x6a, 0xc3,
			0x25, 0x38, 0xbe, 0x6e, 0xca, 0x12, 0x54, 0x76,
			0x5a, 0x84, 0xf7, 0x19, 0x44, 0x78, 0xec, 0xc0,
			0x83, 0xf6, 0x22, 0xb8, 0x86, 0x31, 0xe9, 0x2e,
		},
		.export_len = 32,
	},
	{
		/* Invalid - illegal label. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = TLS_MD_CLIENT_FINISH_CONST,
		.label_len = TLS_MD_CLIENT_FINISH_CONST_SIZE,
		.use_context = 0,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export_len = 32,
		.want_error = SSL_R_TLS_ILLEGAL_EXPORTER_LABEL,
	},
	{
		/* Invalid - illegal label. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = TLS_MD_SERVER_FINISH_CONST,
		.label_len = TLS_MD_SERVER_FINISH_CONST_SIZE,
		.use_context = 0,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export_len = 32,
		.want_error = SSL_R_TLS_ILLEGAL_EXPORTER_LABEL,
	},
	{
		/* Invalid - illegal label. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = TLS_MD_KEY_EXPANSION_CONST,
		.label_len = TLS_MD_KEY_EXPANSION_CONST_SIZE,
		.use_context = 0,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export_len = 32,
		.want_error = SSL_R_TLS_ILLEGAL_EXPORTER_LABEL,
	},
	{
		/* Invalid - illegal label. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = TLS_MD_MASTER_SECRET_CONST,
		.label_len = TLS_MD_MASTER_SECRET_CONST_SIZE,
		.use_context = 0,
		.client_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export_len = 32,
		.want_error = SSL_R_TLS_ILLEGAL_EXPORTER_LABEL,
	},
	{
		/* Invalid - illegal label, split over label and seed. */
		.tls_version = TLS1_2_VERSION,
		.cipher_id = TLS1_CK_ECDHE_RSA_CHACHA20_POLY1305,
		.label = "master ",
		.label_len = 7,
		.use_context = 0,
		.client_random = {
			 's',  'e',  'c',  'r',  'e',  't', 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.server_random = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.master_key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		},
		.export = {
			0x40, 0x70, 0xba, 0xfa, 0xba, 0x44, 0x74, 0x93,
			0xa2, 0x43, 0x18, 0x07, 0xa4, 0x4f, 0x3f, 0xda,
			0x88, 0x7b, 0x0e, 0x79, 0x70, 0xcf, 0xdb, 0x91,
			0xfc, 0x3f, 0x96, 0x78, 0x6b, 0x50, 0xe3, 0xa6,
		},
		.export_len = 32,
		.want_error = SSL_R_TLS_ILLEGAL_EXPORTER_LABEL,
	},
};

#define N_EXPORTER_TESTS (sizeof(exporter_tests) / sizeof(exporter_tests[0]))

static int
exporter_test(size_t test_no, const struct exporter_test *et)
{
	struct tls13_secret tls13_context = { .data = "", .len = 0 };
	struct tls13_ctx *tls13_ctx;
	struct tls13_secrets *tls13_secrets;
	SSL_SESSION *ssl_session = NULL;
	SSL_CTX *ssl_ctx = NULL;
	SSL *ssl = NULL;
	uint8_t export[256];
	int err, ret;
	int failed = 1;

	memset(export, 0, sizeof(export));

	if ((ssl_ctx = SSL_CTX_new(TLS_method())) == NULL) {
		fprintf(stderr, "FAIL: SSL_CTX_new\n");
		goto failure;
	}
	if ((ssl = SSL_new(ssl_ctx)) == NULL) {
		fprintf(stderr, "FAIL: SSL_new\n");
		goto failure;
	}
	if ((ssl_session = SSL_SESSION_new()) == NULL) {
		fprintf(stderr, "FAIL: SSL_SESSION_new\n");
		goto failure;
	}

	ssl_session->ssl_version = et->tls_version;

	if (!SSL_set_session(ssl, ssl_session)) {
		fprintf(stderr, "FAIL: SSL_set_session\n");
		goto failure;
	}

	memcpy(ssl_session->master_key, et->master_key,
	    sizeof(ssl_session->master_key));
	memcpy(ssl->s3->client_random, et->client_random,
	    sizeof(ssl->s3->client_random));
	memcpy(ssl->s3->server_random, et->server_random,
	    sizeof(ssl->s3->server_random));

	if (et->tls_version >= TLS1_3_VERSION) {
		if ((tls13_ctx = tls13_ctx_new(TLS13_HS_CLIENT, ssl)) == NULL) {
			fprintf(stderr, "FAIL: tls13_ctx_new\n");
			goto failure;
		}
		ssl->tls13 = tls13_ctx;

		if ((tls13_secrets = tls13_secrets_create(EVP_sha384(),
		    0)) == NULL) {
			fprintf(stderr, "FAIL: tls13_secrets_create\n");
			goto failure;
		}
		ssl->s3->hs.tls13.secrets = tls13_secrets;

		if (!tls13_derive_early_secrets(tls13_secrets,
		    tls13_secrets->zeros.data, tls13_secrets->zeros.len,
		    &tls13_context)) {
			fprintf(stderr, "FAIL: tls13_derive_early_secrets\n");
			goto failure;
		}
		if (!tls13_derive_handshake_secrets(tls13_secrets, et->shared_key,
		    et->shared_key_len, &tls13_context)) {
			fprintf(stderr, "FAIL: tls13_derive_handshake_secrets\n");
			goto failure;
		}
		if (!tls13_derive_application_secrets(tls13_secrets,
		    &tls13_context)) {
			fprintf(stderr, "FAIL: tls13_derive_early_secrets\n");
			goto failure;
		}

		tls13_ctx->handshake_completed = 1;
	}

	ssl->s3->hs.state = SSL_ST_OK;
	ssl->s3->hs.negotiated_tls_version = et->tls_version;
	ssl->s3->hs.cipher = SSL_CIPHER_get_by_id(et->cipher_id);

	ret = SSL_export_keying_material(ssl, export, et->export_len, et->label,
	    et->label_len, et->context_value, et->context_value_len,
	    et->use_context);

	if (et->want_error != 0) {
		if (ret) {
			fprintf(stderr, "FAIL: test %zu - "
			    "SSL_export_keying_material() succeeded, want "
			    "error\n", test_no);
			goto failure;
		}

		err = ERR_peek_error();
		if (ERR_GET_REASON(err) != et->want_error) {
			fprintf(stderr, "FAIL: %zu - got error reason %d, "
			    "want %d\n", test_no, ERR_GET_REASON(err),
			    et->want_error);
			goto failure;
		}
	} else {
		if (!ret) {
			fprintf(stderr, "FAIL: test %zu - "
			    "SSL_export_keying_material() failed\n", test_no);
			ERR_print_errors_fp(stderr);
			goto failure;
		}

		if (memcmp(et->export, export, et->export_len) != 0) {
			fprintf(stderr, "FAIL: test %zu\n", test_no);
			fprintf(stderr, "Got export:\n");
			hexdump(export, et->export_len);
			fprintf(stderr, "Want export:\n");
			hexdump(et->export, et->export_len);
			goto failure;
		}
	}

	failed = 0;

 failure:
	SSL_SESSION_free(ssl_session);
	SSL_CTX_free(ssl_ctx);
	SSL_free(ssl);

	return failed;
}

int
main(int argc, char **argv)
{
	int failed = 0;
	size_t i;

	for (i = 0; i < N_EXPORTER_TESTS; i++)
		failed |= exporter_test(i, &exporter_tests[i]);

	return (failed);
}
