//
// Syd: rock-solid application kernel
// src/kernel/uname.rs: uname(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::mem::MaybeUninit;

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{config::MMAP_MIN_ADDR, hook::UNotifyEventRequest};

pub(crate) fn sys_uname(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        // SAFETY: Check pointer against mmap_min_addr.
        if req.data.args[0] < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        let mut name = MaybeUninit::<libc::utsname>::uninit();

        // SAFETY: In libc we trust.
        Errno::result(unsafe { libc::uname(name.as_mut_ptr()) })?;

        // SAFETY: uname() has initialized `name` if it succeeded.
        let mut name = unsafe { name.assume_init() };

        // Wipe the version fields with zeros.
        // SAFETY: Unsafe is needed because we are directly manipulating
        // C structure fields. Here we trust the return value of
        // uname(2).
        unsafe {
            std::ptr::write_bytes(name.version.as_mut_ptr(), 0, 65);
        }

        // SAFETY: The use of `from_raw_parts` here is safe because
        // `name` is fully initialized at this point by the preceding
        // `uname` system call, ensuring that the memory region from
        // `&name` up to the size of `libc::utsname` is valid. The
        // conversion to a byte slice is done to facilitate copying the
        // structure to another memory location without altering its
        // contents. This operation does not extend the lifetime of
        // `name` beyond this function, nor does it modify the content
        // of `name`, adhering to Rust's safety and borrowing rules.
        let name = unsafe {
            std::slice::from_raw_parts(
                std::ptr::addr_of!(name) as *const u8,
                std::mem::size_of_val(&name),
            )
        };

        request.write_mem(name, req.data.args[0])?;
        Ok(request.return_syscall(0))
    })
}
