// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief e+e- > pi+ pi-
  class CMD3_2023_I2634277 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CMD3_2023_I2634277);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      for (size_t ix=0; ix<3; ++ix) {
        book(_sigma[ix], 1, 1+ix, 1);
        for (const string& en : _sigma[ix].binning().edges<0>()) {
          const double eval = stod(en)*MeV;
          if (isCompatibleWithSqrtS(eval)) {
            _sqs[ix] = en; break;
          }
        }
      }
      raiseBeamErrorIf(_sqs[0].empty() && _sqs[1].empty() && _sqs[2].empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      if (fs.particles().size() != 2) vetoEvent;
      for (const Particle& p : fs.particles()) {
        if (p.abspid() != PID::PIPLUS) vetoEvent;
      }
      for (size_t ix=0; ix<3; ++ix) {
        if (!_sqs[ix].empty()) _sigma[ix]->fill(_sqs[ix]);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma, crossSection()/sumOfWeights()/nanobarn);
      const double gev2nb = 0.389e6;
      const double alpha = 1./137.035999084;
      for (size_t ix=0; ix<3; ++ix) {
        for (auto& b : _sigma[ix]->bins()) {
          const double eval = stod(b.xEdge())*MeV;
          const double beta = sqrt(1.-4.*sqr(0.13957039/eval));
          const double sigma0 = gev2nb*M_PI*sqr(alpha)*beta*sqr(beta)/3/sqr(eval);
          b.scaleW(1.0/sigma0);
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigma[3];
    string _sqs[3];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(CMD3_2023_I2634277);
}
