/*
 * aurun - start a prorgram with auditing enabled.
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <ctype.h>
#include <sys/param.h>

#include <laus.h>

static const char *	opt_user;
static int		opt_session = 1;

static void		usage(int);

int
main(int argc, char **argv)
{
	uid_t	uid = getuid();
	int	c, attached = 0;

	while ((c = getopt(argc, argv, "+Nu:")) != EOF) {
		switch (c) {
		case 'N':
			opt_session = 0;
			break;
		case 'u':
			opt_user = optarg;
			break;
		default:
			usage(1);
		}
	}

	if (optind >= argc)
		usage(1);

	if (opt_user != NULL) {
		struct passwd *pw;

		if (!(pw = getpwnam(opt_user))) {
			fprintf(stderr, "Unknown user name %s\n", opt_user);
			return 1;
		}
		if (initgroups(opt_user, pw->pw_gid) < 0) {
			perror("initgroups");
			return 1;
		}
		uid = pw->pw_uid;
	}

	if (laus_open(NULL) < 0)
		goto failed;

	/* If we're no attached, do so now, else the laus_log
	 * call below will fail */
	if (laus_attach() >= 0) {
		if (laus_setauditid() < 0)
			goto failed;

		if (opt_session
		 && laus_setsession(uid, NULL, NULL, ttyname(0)) < 0)
			goto failed;
		attached = 1;
	}

	/* write log message with command and args */
	{
		char out[ARG_MAX], *p, *end;
		int i;

		end = out + sizeof(out) - 5;
		for (i = optind, p = out; i < argc; ++i) {
			const char *s = argv[i];
			int	n, needquote = 0;

			for (n = 0; s[n]; n++) {
				if (!isprint(s[n]) || isspace(s[n]))
					needquote = 1;
			}

			if (p + n + 2 * needquote + 1 >= end) {
				strcpy(p, " ...");
				p += 4;
				break;
			}

			*p++ = ' ';
			if (needquote)
				*p++ = '"';
			memcpy(p, s, n);
			p += n;
			if (needquote)
				*p++ = '"';
		}
		*p = '\0';

		if (laus_log(PROC_execute, "aurun: execute%s\n", out) < 0) {
			fprintf(stderr, "%s\n", laus_strerror(errno));
			return 1;
		}
	}

	/* If our caller was being audited already, detach and
	 * start new session */
	if (!attached) {
		if (laus_detach() < 0
		 || laus_attach() < 0
		 || laus_setauditid() < 0)
		 	goto failed;

		if (opt_session
		 && laus_setsession(uid, NULL, NULL, ttyname(0)) < 0)
			goto failed;
	}

	laus_close();

	if (uid != getuid())
		setuid(uid);

	execvp(argv[optind], argv + optind);
	perror(argv[optind]);
	return 1;

failed:
	fprintf(stderr, "Cannot start auditing: %s\n",
		laus_strerror(errno));
	return 1;
}

void
usage(int exval)
{
	fprintf(stderr,
		"usage: aurun [-N] [-u user] command ...\n");
	exit(exval);
}
