/*
 * Audit record output functions
 *
 * Copyright (C) 2003 SuSE Linux AG
 * Written by okir@suse.de
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>
#include <stdlib.h>
#include <sys/uio.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <laus.h>
#include <laussrv.h>

#include "auditd.h"
#include "config.h"

enum { TYPE_BIN, TYPE_FILE, TYPE_COMMAND };

typedef struct output {
	struct output *	next;
	int		type;
	const char *	path;
	const char *	notify;
	const char *	current;
	size_t		size;
	int		sync;
        int             nsync;
	void *		data;
	action_t *	actions;
	void		(*write_record)(struct output *, void *, size_t);
	int		(*reopen)(struct output *);
	void		(*process_done)(struct output *, pid_t, int);
} output_t;

typedef struct binfile {
	struct binfile *next;
	const char *	path;
	int		fd;
	unsigned int	index;
	caddr_t		mem;
	size_t		offset;
	pid_t		pid;
} binfile_t;

typedef struct streamfile {
	FILE *		pipe;
	int		fd;
} streamfile_t;

static output_t *	destinations;
static size_t		pagesize;

static pid_t		output_notify(output_t *, const char *);
static void		output_failed(output_t *out);
static int		mkdirp(const char *filename);

/*
 * Write output to all destinations
 */
void
output_write_record(void *data, size_t len)
{
	sigset_t	set;
	output_t	*out;

	if (opt_debug) {
		struct aud_message *msg = (struct aud_message *) data;

		log_dbg("write_record: msg_type=%u len=%u", msg->msg_type, len);
	}

	/* Block SIGHUP and SIGCHLD delivery while we're here */
	sigfillset(&set);
	sigprocmask(SIG_SETMASK, &set, &set);

	for (out = destinations; out; out = out->next)
		out->write_record(out, data, len);
	
	/* Put back the previous sigprocmask */
	sigprocmask(SIG_SETMASK, &set, NULL);
}

/*
 * A child process died. Check all output processes.
 */
void
output_child_complete(pid_t pid, int status)
{
	output_t	*out;

	for (out = destinations; out; out = out->next)
		out->process_done(out, pid, status);
}

/*
 * Write a record to a file
 */
static int
common_record_write(int fd, caddr_t space, const void *msgdata, size_t len, int sync)
{
	int		res;
	struct laus_record_header header;

	/* Fill in the header */
	header.r_time = time(NULL);
	header.r_size = len;

	if (space) {
		memcpy(space, &header, sizeof(header));
		memcpy(space + sizeof(header), msgdata, len);
		res = sizeof(header) + len;
		if (sync) {
			size_t adjust = (size_t)space & (pagesize - 1);

			if (msync(space - adjust, adjust + sizeof(header) + len, MS_SYNC) < 0)
				res = -1;
		}
	}
	else {
		sigset_t	sigset, oldset;
		int		oerrno;
		struct iovec	iov[2];

		/* Fill in iovec */
		iov[0].iov_base = &header;
		iov[0].iov_len  = sizeof(header);
		iov[1].iov_base = (void *)msgdata;
		iov[1].iov_len  = len;

		/* Prevent SIGTERM etc while we're writing the record */
		sigfillset(&sigset);
		sigprocmask(SIG_SETMASK, &sigset, &oldset);

		res = writev(fd, iov, 2);

		oerrno = errno;
		sigprocmask(SIG_SETMASK, &oldset, NULL);
		errno = oerrno;

		if (sync && res > 0 && fsync(fd) < 0)
			res = -1;
	}

	return res;
}

/*
 * Write the file header
 */
static int
common_header_write(int fd, struct laus_file_header *hdr, int sync)
{
	struct laus_file_header header;
	int res;

	if (!hdr)
		memset(hdr = &header, 0, sizeof(header));

	hdr->h_version = laus_version();
	hdr->h_msgversion = laus_api_version();
	hdr->h_count = -1;
	gethostname(hdr->h_hostname, sizeof(hdr->h_hostname));

	if (hdr == &header) {
		res = write(fd, &header, sizeof(header));
		if (sync && res > 0 && fsync(fd) < 0)
			res = -1;
	}
	else
		res = sync ? msync(hdr, sizeof(*hdr), MS_SYNC) : 0;
	if (res < 0)
		log_dbg("unable to write audit file header: %m");
	else
		res = 0;

	return res;
}

static int
common_header_check(int fd, const struct laus_file_header *hdr)
{
	struct laus_file_header header;

	if (hdr == NULL) {
		int res = read(fd, &header, sizeof(header));

		if (res < 0 || (size_t)res < sizeof(header))
			return 0;
		hdr = &header;
	}
	return hdr->h_version == laus_version()
	    && hdr->h_msgversion == laus_api_version();
}

/*
 * Set the binfile to write to
 */
static int
binfile_set(output_t *out, binfile_t *bfp0)
{
	binfile_t *bfp;
	int force = 0;

	for (bfp = bfp0; ; bfp = bfp->next) {
		struct laus_file_header *hdr;

		/* Inspect file header to see if this is a "live" bin.
		 * This happens when restarting the daemon, for instance.
		 */
		hdr = (struct laus_file_header *) bfp->mem;

		bfp->offset = sizeof(*hdr);
		if (force || hdr->h_count == 0) {
			/* Rewind to beginning of file and write
			 * header. */
			common_header_write(bfp->fd, hdr, out->sync);
			hdr->h_count = 0;
			break;
		}
		if (common_header_check(bfp->fd, hdr)) {
			bfp->offset += hdr->h_count;
			lseek(bfp->fd, bfp->offset, SEEK_SET);
			break;
		}
		log_err(LOG_ERR, "Skipping binfile %s - different previous writer\n", bfp->path);
		if (bfp->next == bfp0) {
			log_err(LOG_CRIT, "No clean audit log file found, forcing cleanup of %s\n", bfp0->path);
			force = 1;
		}
	}

	if (out->current) {
		if (opt_debug > 1) {
			log_dbg("setting symlink %s -> %s",
				out->current, bfp->path);
		}
		unlink(out->current);
		symlink(bfp->path, out->current);
	}
	out->data = bfp;
	return 0;
}

/*
 * Notify command is done. Check status
 */
static int
binfile_notify_done(output_t *out, binfile_t *bfp, int status)
{
	if (opt_debug) {
		log_dbg("Notify command(%s) completed, status %d",
				bfp->path, status);
	}

	bfp->pid = 0;
	if (status == -1) {
		/* waitpid() failed */
	} else
	if (!WIFEXITED(status)) {
		log_err(LOG_ERR, "Notify command %s terminated abnormally",
			out->notify);
	} else if (WEXITSTATUS(status) != 0) {
		log_err(LOG_ERR, "Notify command %s exited with status %d",
			out->notify, WEXITSTATUS(status));
	} else {
		/* Convenience function - make sure bin file was
		 * cleaned. Raise a huge ruckus if it's not
		 */
		if (((struct laus_file_header *) bfp->mem)->h_count) {
			log_err(LOG_CRIT,
				"Audit log file wasn't cleaned by notify command!");
			log_err(LOG_CRIT,
				"Cleaning now, but please make sure your notify script "
				"uses audbin -C");
			memset(bfp->mem, 0, sizeof(struct laus_file_header));
		}

		return 0;
	}

	/* Notify admin, and wait for him to correct the problem */
	output_failed(out);

	/* Re-run the command. If called from binfile_process_exit,
	 * this will cause auditd to wait for the command to complete. */
	log_err(LOG_ERR, "Retrying failed notify command");
	bfp->pid = output_notify(out, bfp->path);

	return -1;
}

static void
binfile_process_exit(output_t *out, pid_t pid, int status)
{
	binfile_t	*bfp, *head;

	bfp = head = (binfile_t *) out->data;
	do {
		if (bfp->pid == pid) {
			binfile_notify_done(out, bfp, status);
			return;
		}
		bfp = bfp->next;
	} while (bfp != head);
}

/*
 * Switch to next binfile
 */
static binfile_t *
binfile_switch(output_t *out)
{
	int		status;
	binfile_t	*bfp;
	sigset_t	sigset, oldset;

	bfp = (binfile_t *) out->data;
	bfp->pid = output_notify(out, bfp->path);
	log_dbg("started notify command, pid=%d", bfp->pid);

	bfp = bfp->next;

	log_dbg("switching output to binfile %u", bfp->index);

	sigemptyset(&sigset);
	sigaddset(&sigset, SIGCHLD);
	sigprocmask(SIG_BLOCK, &sigset, &oldset);

	while (bfp->pid) {
		log_dbg("Waiting for notify command to complete (%s)", bfp->path);

		if (waitpid(bfp->pid, &status, 0) < 0) {
			log_err(LOG_ERR,
				"Failed to collect exit status of notify command: %m");
			status = -1;
		}
		binfile_notify_done(out, bfp, status);
	}

	sigprocmask(SIG_SETMASK, &oldset, NULL);

	binfile_set(out, bfp);
	return bfp;
}

/*
 * Write record to binfile
 */
static void
binfile_write_record(output_t *out, void *msgdata, size_t len)
{
	const size_t	hdrlen = sizeof(struct laus_record_header);
	binfile_t	*bfp;
	size_t		left;
	int		n;
	static int      written = 0;

again:
	bfp = (binfile_t *) out->data;

	left = out->size - (bfp->offset + hdrlen);
	if ((long) left < 0 || len > left) {
		bfp = binfile_switch(out);
		left = out->size - (bfp->offset + hdrlen);
		if (len > left)
			len = left;
	}

	if ((n = common_record_write(bfp->fd, bfp->mem + bfp->offset, msgdata, len, out->sync)) < 0) {
		log_err(LOG_CRIT,
			"Failed to write to audit log file (binfile #%u): %m",
			bfp->index);
		pause();
		goto again;
	}
	((struct laus_file_header *) bfp->mem)->h_count += n;

	++written;

	if (out->sync) {
		if( written >= out->nsync ) {
			if( opt_debug ) {
				log_dbg( "calling msync(), written = %d", written );
			}
			msync(bfp->mem + bfp->offset, len, MS_SYNC);
			written = 0;
		}
	}

	bfp->offset += n;
}

/*
 * Find out which binfile we were writing to when we exited.
 * Return head by default.
 */
static binfile_t *
binfile_get_current(binfile_t *head, const char *linkname)
{
	binfile_t	*bfp;
	char		current[PATH_MAX];
	int		n;

	if (linkname == NULL)
		return head;

	if ((n = readlink(linkname, current, sizeof(current))) < 0)
		return head;
	current[n] = '\0';

	bfp = head;
	do {
		if (!strcmp(bfp->path, current))
			break;
		bfp = bfp->next;
	} while (bfp != head);

	if (opt_debug > 1)
		log_dbg("current bin file %s -> %s", linkname, bfp->path);
	return bfp;
}

int
binfile_reopen(output_t *out)
{
	binfile_t	*head = (binfile_t *) out->data;

	/* Restart where we left off last time */
	return binfile_set(out, binfile_get_current(head, out->current));
}

/*
 * Set up logging to binfiles
 */
static void
binfile_init(output_t *out, unsigned int numfiles)
{
	binfile_t	*head = NULL, *bfp;

	if (numfiles < 2)
		numfiles = 2;

	if (out->path == NULL)
		out->path = PATH_LOGFILE;

	if (out->notify == NULL)
		log_fatal("No notify command given for binfile output!");

	/* Create directory containing bin files */
	mkdirp(out->path);

	while (numfiles--) {
		char		path[PATH_MAX];
		int		fd;
		void		*p;
		struct stat	stb;

		snprintf(path, sizeof(path), "%s.%u", out->path, numfiles);

		if ((fd = open(path, O_CREAT|O_RDWR, 0600)) < 0)
			log_fatal("Unable to open %s: %m", path);

		/* Get original file size */
		if (fstat(fd, &stb) < 0)
			log_fatal("Unable to stat %s: %m", path);

		/* Resize file if necessary. Note that we never shrink
		 * a file, because there may still be data in it that
		 * hasn't been archived yet.
		 */
		if (stb.st_size < out->size) {
			if (ftruncate(fd, out->size) < 0)
				log_fatal("Failed to resize %s: %m", path);
		}

		/* Map the bin file into memory */
		p = mmap(NULL, out->size, PROT_WRITE|PROT_READ, MAP_SHARED, fd, 0);
		if (p == MAP_FAILED)
			log_fatal("Failed to mmap %s: %m", path);

		/* Force allocation of disk blocks */
		if (stb.st_size < out->size)
			memset(p + stb.st_size, 0, out->size - stb.st_size);

		/* Now set up the binfile structure in memory */
		bfp = (binfile_t *) calloc(1, sizeof(*bfp));
		bfp->path = strdup(path);
		bfp->index = numfiles;
		bfp->mem = p;
		bfp->fd = fd;

		/* Insert at head of list */
		bfp->next = head;
		head = bfp;
	}

	/* Make the list circular */
	for (bfp = head; bfp->next; bfp = bfp->next)
		;
	bfp->next = head;

	out->write_record = binfile_write_record;
	out->reopen = binfile_reopen;
	out->process_done = binfile_process_exit;

	/* Restart where we left off last time */
	binfile_set(out, binfile_get_current(head, out->current));
}

/*
 * Close the log file
 */
static void
stream_close(output_t *out)
{
	streamfile_t *streamfile = (streamfile_t *) out->data;

	if (streamfile == NULL)
		return;

	if (opt_debug)
		log_dbg("Closing audit log");

	if (streamfile->pipe)
		pclose(streamfile->pipe);
	else
		close(streamfile->fd);

	free(streamfile);
	out->data = NULL;
}

/*
 * Reopen the log file
 */
static int
stream_reopen(output_t *out)
{
	streamfile_t *streamfile;
	int	write_header = 0;

	stream_close(out);

	streamfile = (streamfile_t *) calloc(1, sizeof(*streamfile));
	if (out->type == TYPE_COMMAND) {
		FILE	*fp;

		/* Streaming to an external command */
		if ((fp = popen(out->path, "w")) == NULL)
			return -1;
		streamfile->pipe = fp;
		streamfile->fd = fileno(fp);
		write_header = 1;
	} else {
		struct stat stb;
		int	fd;

		fd = open(out->path, O_CREAT|O_RDWR|O_APPEND, 0600);
		if (fd < 0)
			return -1;
		streamfile->fd = fd;
		if (fstat(fd, &stb) < 0)
			return -1;
		write_header = (stb.st_size == 0);
	}

	if (write_header)
		common_header_write(streamfile->fd, NULL, out->sync);
	else if (!common_header_check(streamfile->fd, NULL)) {
		log_err(LOG_CRIT, "Different previous writer of %s\n", out->path);
		return -1;
	}

	if (out->type != TYPE_COMMAND) /* makes no sense to check, if command */
		recheck_disk_thresholds(out->path);
	out->data = streamfile;
	return 0;
}

/*
 * Write a record to the audit file, taking care of disk space
 * problems.
 *
 * Note: if the log file is not available, we will hang in this
 * function until someone fixes the situation and sends us a SIGHUP.
 * While we hang here, we won't be able to collect any audit records
 * piling up in kernel space, effectively blocking all applications
 * being audited.
 */
void
stream_write_record(output_t *out, void *msgdata, size_t len)
{
	streamfile_t	*streamfile;
	static int      written = 0;
again:
	while (out->data == NULL)
		output_failed(out);

	/* Check disk thresholds and warn if necessary */
	if (out->type != TYPE_COMMAND) /* makes no sense to check, if command */
		check_disk_thresholds(out->path);

	/* Write audit record to log file. If writing fails
	 * for _any_ reason we assume it's a disk full problem.
	 * Close the log and wait for it to become available
	 * again.
	 * All tasks being audited will block while we're waiting.
	 */
	streamfile = (streamfile_t *) out->data;
	if (common_record_write(streamfile->fd, NULL, msgdata, len, out->sync) < 0) {
		if (errno == EINTR)
			goto again;
		log_err(LOG_CRIT, "Failed to write to log file: %m");
		stream_close(out);
		goto again;
	}
}

static void
stream_init(output_t *out)
{
	if (out->path == NULL)
		out->path = PATH_LOGFILE;

	if (stream_reopen(out) < 0)
		log_fatal("Failed to open stream \"%s\": %m", out->path);

	out->write_record = stream_write_record;
	out->reopen = stream_reopen;
}

/*
 * Configure audit destinations
 */
int
configure_output(void)
{
	cf_node_t	*np;
	const char	*complain;

	pagesize = sysconf(_SC_PAGESIZE);
	np = cf_node_find(cf_root, "output");
	while (np != NULL) {
		const char	*mode, *dest = NULL;
		output_t	*out;

		out = (output_t *) calloc(1, sizeof(*out));

		if (!(mode = cf_node_value(np, "mode"))) {
			complain = "no output mode defined";
			goto fail;
		}

		out->path = cf_node_value(np, "file-name");
		out->size = cf_node_atofs(np, "file-size", 0);
		out->sync = cf_node_atob(np, "sync", 1);
		out->nsync = cf_node_atoi(np, "sync-after", 0);
		if( out->nsync > 0 && opt_debug ) {
		  log_dbg( "sync-after set to %d", out->nsync );
		}
		out->notify = cf_node_value(np, "notify");
		out->current = cf_node_value(np, "current");

		if (!strcmp(mode, "stream") || !strcmp(mode, "append")) {
			dest = cf_node_value(np, "command");
			if (dest != NULL) {
				out->path = dest;
				out->type = TYPE_COMMAND;
			} else {
				out->type = TYPE_FILE;
				if (out->path && out->path[0] == '|') {
					out->type = TYPE_COMMAND;
					out->path++;
				}
			}
			if (out->type == TYPE_COMMAND) {
				out->sync = 0; /* fsync(2) fails for commands (e.g. pipe) */
			}
			stream_init(out);
		} else if (!strcmp(mode, "bin")) {
			unsigned int	count;

			if (out->size == 0) {
				complain = "no bin file size given";
				goto fail;
			}
			if ((count = cf_node_atoi(np, "num-files", 2)) < 2)
				count = 2;
			binfile_init(out, count);
		} else {
			complain = "unknown output mode";
			goto fail;
		}

		out->next = destinations;
		out->actions = configure_actions(cf_node_find(np, "error"));

		destinations = out;

		np = cf_node_find_next(np, "output");
	}

	/* The default is to write an append log */
	if (destinations == NULL) {
		destinations = (output_t *) calloc(1, sizeof(output_t));
		stream_init(destinations);
	}

	return 0;

fail:	log_err(LOG_ERR, "failed to configure audit output: %s", complain);
	exit(1);
}

void
output_reopen(void)
{
	output_t	*out;

	log_dbg("Reopening output file(s)");
	for (out = destinations; out; out = out->next) {
		if (out->reopen)
			out->reopen(out);
	}
}

/*
 * Notification commands (binfile only)
 */
pid_t
output_notify(output_t *out, const char *path)
{
	char	command[2048], *s;
	char	*argv[64];
	int	fd, argc, attempts;
	pid_t	pid;

	log_dbg("run notify command: %s %s", out->notify, path);

	if ((fd = open("/dev/null", O_RDWR)) < 0)
		log_fatal("Unable to open /dev/null");

	snprintf(command, sizeof(command), "%s %s", out->notify, path);

	/* Split up command; simplistic version (no quoting) */
	s = strtok(command, " \t");
	for (argc = 0; s && argc < 63; argc++) {
		argv[argc] = s;
		s = strtok(NULL, " \t");
	}
	argv[argc] = NULL;

	attempts = 0;
	while ((pid = fork()) < 0) {
		if ((pid = fork()) >= 0)
			break;
		if (attempts++ > 5) {
			log_err(LOG_ERR, "Cannot run %s, fork: %m", out->notify);
			output_failed(out);
			attempts = 0;
		}
		sleep(2 * attempts);
	}

	if (pid != 0) {
		close(fd);
		return pid;
	}

	/* lower priority - gzip doesn't have to run with realtime
	 * priority */
	audit_set_priority(0);

	signal(SIGINT, SIG_IGN);
	signal(SIGHUP, SIG_IGN);
	dup2(fd, 0);
	dup2(fd, 1);
	dup2(fd, 2);

	execv(argv[0], argv);
	log_err(LOG_ERR, "Failed to execute %s, execv: %m", out->notify);
	exit(127);
}

/*
 * Handle output errors.
 *
 * By default, we stall until a SIGHUP arrives and tells us we
 * should continue
 */
void
output_failed(output_t *out)
{
	action_t	*ac, dummy;

	if ((ac = out->actions) == NULL) {
		memset(&dummy, 0, sizeof(dummy));
		dummy.ac_type = ACTION_SUSPEND;
		ac = &dummy;
	}
	perform_actions(ac, "output error", "output error");
}

/*
 * We should probably have a utils.c for stuff like this
 */
int
mkdirp(const char *filename)
{
	char	path[PATH_MAX], *p;

	strncpy(path, filename, sizeof(path));
	path[sizeof(path)-1] = '\0';

	for (p = path; *p == '/'; p++)
		;
	while ((p = strchr(p, '/')) != NULL) {
		*p = '\0';
		if (mkdir(path, 0755) < 0 && errno != EEXIST)
			return -1;
		*p++ = '/';
	}

	return 0;
}
