/*
 * Photo Image Print System
 * Copyright (C) 2000-2005 EPSON AVASYS Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2000-2005.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON AVASYS Corporation gives permission to
 * link the code of this program with libraries which are covered by
 * the EPSON AVASYS Public License and distribute their linked
 * combinations.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the libraries which
 * are covered by EPSON AVASYS Public License.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <dlfcn.h>
#include "pips.h"
#include "pipsError.h"
#include "setup.h"
#include "ekpcom.h"
#include "unistd.h"

extern int resource_ready;
static int graphicsConverter (UCHAR*, POINT, SEP_INIT_PARAM*, FILE*);

/***************************************************
 *                    main
 ***************************************************/

#define STD_IN_NAME	"STDIN"
#define RETRY_COUNT 2

int main( int argc, char *argv[] )
{
   int error;
   char *filename;
   SEP_INIT_PARAM *sepip;
   PIPS_INIT_PARAM *pip;
   char printer_id[256];
   int i;

  filename = NULL;
  sepip = NULL;
  pip = NULL;
  error = 0;
  printer_id[0] = 0;

  sepip = (SEP_INIT_PARAM *)malloc (sizeof(SEP_INIT_PARAM));
  if ( !sepip )
  {
    error = -1;
    goto bail;
  }
  pip = (PIPS_INIT_PARAM *)calloc (1, sizeof(PIPS_INIT_PARAM));
  if ( !pip )
  {
    error = -1;
    goto bail;
  }

  fprintf(stderr, "\nPhoto Image Print System %s Version %s\n", PRINTER_MODEL, PACKAGE_VERSION);

  sock_open ();
  load_printer_info(printer_id);
  for (i = 0; printer_id[0] == 0 && i < RETRY_COUNT; i++)
  {
	  /* retry */
	  sleep (1);
	  load_printer_info(printer_id);
  }

  /* Not connected */
  if (printer_id[0] == 0)
  {
	  /* Check the resource file */
	  FILE* rscfile;

	  if (((rscfile = fopen(PRTOPT_PATH,"r")) == NULL) || 
	      (load_default_printer_info(printer_id) != 0))
	  {
		  resource_ready = 0;
	  }
	  else 
	  {
		  resource_ready = 1;

		  /* get default printer */
		  if (load_default_printer_info(printer_id))
			  goto bail;
	  }		
  }
  else 
  {
	  resource_ready = 1;
  }
  
  switch( getRsc( sepip, printer_id) )
  {
    case -1:
	    if (resource_ready)
		    fprintf(stderr, _("Resource file not found or resource not set.\n"));
    case 0:
      break;
    default:
      error = -1;
      goto bail;
  }

  if ( (error = getOpts( argc, argv, &filename, sepip, pip )) )
  {
    goto bail;
  }
  
  if( !filename )
  {
    filename = (char *)malloc( strlen( STD_IN_NAME ) + 1 );
    strcpy( filename, STD_IN_NAME );
  }

  if ( pip->action_mode == ACTION_MODE_GET_PRINTER_ID )
  {
	  if (printer_id[0] != 0)
	  {
		  fprintf (stdout, "%s", printer_id);
	  }
	  else
	  {
		  goto bail;
	  }
  }
#if USE_GTK
  else if ( pip->action_mode == ACTION_MODE_GUI )
  {
    if (init_prt_name ())
      {
	set_prt_name (SPOOL_NAME);
      }
    
    if ((error = xMode(argc, argv, filename, sepip, pip)))
      goto bail;
  }
#endif
  else
  {
    if((error = cMode(filename, sepip, pip)))
      goto bail;
  }

bail:
  free_resource();
  sock_close();

  if ( sepip )
    free( sepip );
  if ( pip )
    free( pip );
  if ( filename )
    free( filename );
  if ( error )
    pipsError( NULL, ABNORMAL_END );

  return error ? 1 : 0;
}


#undef STD_IN_NAME

/* ################################################## */
/*           printLoop                                */
/* ################################################## */

int printLoop (char* filename, FILE *outfp, SEP_INIT_PARAM *sepip)
{
  FILE *inStrm;
  FILE *outStrm;
  POINT size;
  UCHAR *bgr_data;
  int error_code = 0;
  int loop = 1;

  if(!(outStrm = outfp)) return UNEXPECTED_ERROR;
  if ( strcmp( filename, "STDIN" ) )
    {
      if ( !(inStrm = fopen( filename, "rb" )) )
	{
	  return pipsError( filename, NO_FILE_ERROR );
	}
    }
  else
    {
      inStrm = stdin;
    }
  
  while( !feof( inStrm ) || !ferror( inStrm ))
    {
#if DEBUG
      fprintf( stderr, "begin fortune wheel %d\n",loop );
#endif
      loop++;
      bgr_data = NULL;
      if ( (error_code = fileLoad( inStrm, outStrm, &size, &bgr_data)) )
	{
	  if (error_code == -1)
	    error_code = 0;
	  else if ( error_code != INPUT_ESCP_WARNING )
	    pipsError( filename, error_code );
	  break;
	}
      
      if ( bgr_data )
	{
	  error_code = graphicsConverter( bgr_data, size, sepip, outfp );
	  free( bgr_data );
	  bgr_data = NULL;
	}
      if (error_code) goto error;
    }
  
  if ( inStrm && (inStrm != stdin) )
    fclose( inStrm );
#ifdef DEBUG
  fprintf( stderr, "END %d\n", loop );
#endif
  return NO_ERROR;
error:
  if ( inStrm && (inStrm != stdin) )
    fclose( inStrm );

  if (bgr_data) free (bgr_data);
  return error_code;
}

/* -------------------------------------------------- */
/*           graphicsConverter                        */
/* -------------------------------------------------- */

static int graphicsConverter(UCHAR *bgr_data, POINT size, SEP_INIT_PARAM* sepip, FILE *fp)
{
  void *handle;
  func_t funptr;
  int error_code;

  error_code = NO_ERROR;
  handle = NULL;
  funptr = NULL;

  if( !(handle = dlopen(LIBPATH, RTLD_LAZY)) )
  {
    pipsError((char *)dlerror(), NO_SUCH_LIBRARY);
    return NO_SUCH_LIBRARY;
  }
  funptr = (func_t)dlsym(handle, "SEPStart");
  if ( !funptr )
  {
	pipsError( NULL, CONNECTION_ERROR );
	return CONNECTION_ERROR;
  }
  error_code = (*funptr)(bgr_data, (void *)&size, (void *)sepip, (void *)fp);
  if ( handle )
   dlclose(handle);
  if(error_code)
  {
    if(error_code == 1)
    {
      pipsError((char *)NULL, CONNECTION_ERROR);
      return CONNECTION_ERROR;
    }
    else if(error_code == 2)
    {
      pipsError((char *)NULL, LIBRARY_PARAM_ERROR);
      return LIBRARY_PARAM_ERROR;
    }
  }
  return NO_ERROR;
}
