/**
 * Copyright 2014 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rx.internal.operators;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.*;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.*;

import org.junit.*;
import org.mockito.InOrder;

import rx.Observable;
import rx.Observer;
import rx.Scheduler.Worker;
import rx.Subscriber;
import rx.functions.*;
import rx.internal.util.*;
import rx.observers.TestSubscriber;
import rx.schedulers.Schedulers;
import rx.subjects.PublishSubject;

public class OperatorTakeLastTest {

    @Test
    public void testTakeLastEmpty() {
        Observable<String> w = Observable.empty();
        Observable<String> take = w.takeLast(2);

        @SuppressWarnings("unchecked")
        Observer<String> observer = mock(Observer.class);
        take.subscribe(observer);
        verify(observer, never()).onNext(any(String.class));
        verify(observer, never()).onError(any(Throwable.class));
        verify(observer, times(1)).onCompleted();
    }

    @Test
    public void testTakeLast1() {
        Observable<String> w = Observable.just("one", "two", "three");
        Observable<String> take = w.takeLast(2);

        @SuppressWarnings("unchecked")
        Observer<String> observer = mock(Observer.class);
        InOrder inOrder = inOrder(observer);
        take.subscribe(observer);
        inOrder.verify(observer, times(1)).onNext("two");
        inOrder.verify(observer, times(1)).onNext("three");
        verify(observer, never()).onNext("one");
        verify(observer, never()).onError(any(Throwable.class));
        verify(observer, times(1)).onCompleted();
    }

    @Test
    public void testTakeLast2() {
        Observable<String> w = Observable.just("one");
        Observable<String> take = w.takeLast(10);

        @SuppressWarnings("unchecked")
        Observer<String> observer = mock(Observer.class);
        take.subscribe(observer);
        verify(observer, times(1)).onNext("one");
        verify(observer, never()).onError(any(Throwable.class));
        verify(observer, times(1)).onCompleted();
    }

    @Test
    public void testTakeLastWithZeroCount() {
        Observable<String> w = Observable.just("one");
        Observable<String> take = w.takeLast(0);

        @SuppressWarnings("unchecked")
        Observer<String> observer = mock(Observer.class);
        take.subscribe(observer);
        verify(observer, never()).onNext("one");
        verify(observer, never()).onError(any(Throwable.class));
        verify(observer, times(1)).onCompleted();
    }

    @Test
    public void testTakeLastWithNull() {
        Observable<String> w = Observable.just("one", null, "three");
        Observable<String> take = w.takeLast(2);

        @SuppressWarnings("unchecked")
        Observer<String> observer = mock(Observer.class);
        take.subscribe(observer);
        verify(observer, never()).onNext("one");
        verify(observer, times(1)).onNext(null);
        verify(observer, times(1)).onNext("three");
        verify(observer, never()).onError(any(Throwable.class));
        verify(observer, times(1)).onCompleted();
    }

    @Test(expected = IndexOutOfBoundsException.class)
    public void testTakeLastWithNegativeCount() {
        Observable.just("one").takeLast(-1);
    }

    @Test
    public void testBackpressure1() {
        TestSubscriber<Integer> ts = new TestSubscriber<Integer>();
        Observable.range(1, 100000).takeLast(1).observeOn(Schedulers.newThread()).map(newSlowProcessor()).subscribe(ts);
        ts.awaitTerminalEvent();
        ts.assertNoErrors();
        ts.assertReceivedOnNext(Arrays.asList(100000));
    }

    @Test
    public void testBackpressure2() {
        TestSubscriber<Integer> ts = new TestSubscriber<Integer>();
        Observable.range(1, 100000).takeLast(RxRingBuffer.SIZE * 4).observeOn(Schedulers.newThread()).map(newSlowProcessor()).subscribe(ts);
        ts.awaitTerminalEvent();
        ts.assertNoErrors();
        assertEquals(RxRingBuffer.SIZE * 4, ts.getOnNextEvents().size());
    }

    private Func1<Integer, Integer> newSlowProcessor() {
        return new Func1<Integer, Integer>() {
            int c;

            @Override
            public Integer call(Integer i) {
                if (c++ < 100) {
                    try {
                        Thread.sleep(1);
                    } catch (InterruptedException e) {
                    }
                }
                return i;
            }

        };
    }

    @Test
    public void testIssue1522() {
        // https://github.com/ReactiveX/RxJava/issues/1522
        assertEquals(0, Observable
                .empty()
                .count()
                .filter(UtilityFunctions.alwaysFalse())
                .toList()
                .toBlocking().single().size());
    }

    @Test
    public void testIgnoreRequest1() {
        // If `takeLast` does not ignore `request` properly, StackOverflowError will be thrown.
        Observable.range(0, 100000).takeLast(100000).subscribe(new Subscriber<Integer>() {

            @Override
            public void onStart() {
                request(Long.MAX_VALUE);
            }

            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
            }

            @Override
            public void onNext(Integer integer) {
                request(Long.MAX_VALUE);
            }
        });
    }

    @Test
    public void testIgnoreRequest2() {
        // If `takeLast` does not ignore `request` properly, StackOverflowError will be thrown.
        Observable.range(0, 100000).takeLast(100000).subscribe(new Subscriber<Integer>() {

            @Override
            public void onStart() {
                request(1);
            }

            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
            }

            @Override
            public void onNext(Integer integer) {
                request(1);
            }
        });
    }

    @Test(timeout = 30000)
    public void testIgnoreRequest3() {
        // If `takeLast` does not ignore `request` properly, it will enter an infinite loop.
        Observable.range(0, 100000).takeLast(100000).subscribe(new Subscriber<Integer>() {

            @Override
            public void onStart() {
                request(1);
            }

            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
            }

            @Override
            public void onNext(Integer integer) {
                request(Long.MAX_VALUE);
            }
        });
    }


    @Test
    public void testIgnoreRequest4() {
        // If `takeLast` does not ignore `request` properly, StackOverflowError will be thrown.
        Observable.range(0, 100000).takeLast(100000).subscribe(new Subscriber<Integer>() {

            @Override
            public void onStart() {
                request(Long.MAX_VALUE);
            }

            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
            }

            @Override
            public void onNext(Integer integer) {
                request(1);
            }
        });
    }

    @Test
    public void testUnsubscribeTakesEffectEarlyOnFastPath() {
        final AtomicInteger count = new AtomicInteger();
        Observable.range(0, 100000).takeLast(100000).subscribe(new Subscriber<Integer>() {

            @Override
            public void onStart() {
                request(Long.MAX_VALUE);
            }

            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
            }

            @Override
            public void onNext(Integer integer) {
                count.incrementAndGet();
                unsubscribe();
            }
        });
        assertEquals(1,count.get());
    }

    @Test(timeout = 10000)
    public void testRequestOverflow() {
        final List<Integer> list = new ArrayList<Integer>();
        Observable.range(1, 100).takeLast(50).subscribe(new Subscriber<Integer>() {

            @Override
            public void onStart() {
                request(2);
            }

            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(Integer t) {
                list.add(t);
                request(Long.MAX_VALUE - 1);
            }});
        assertEquals(50, list.size());
    }

    @Test(timeout = 30000) // original could get into an infinite loop
    public void completionRequestRace() {
        Worker w = Schedulers.computation().createWorker();
        try {
            final int n = 1000;
            for (int i = 0; i < 25000; i++) {
                if (i % 1000 == 0) {
                    System.out.println("completionRequestRace >> " + i);
                }
                PublishSubject<Integer> ps = PublishSubject.create();
                final TestSubscriber<Integer> ts = new TestSubscriber<Integer>(0);

                ps.takeLast(n).subscribe(ts);

                for (int j = 0; j < n; j++) {
                    ps.onNext(j);
                }

                final AtomicBoolean go = new AtomicBoolean();

                w.schedule(new Action0() {
                    @Override
                    public void call() {
                        while (!go.get()) { }
                        ts.requestMore(n + 1);
                    }
                });

                go.set(true);
                ps.onCompleted();

                ts.awaitTerminalEvent(1, TimeUnit.SECONDS);

                ts.assertValueCount(n);
                ts.assertNoErrors();
                ts.assertCompleted();

                List<Integer> list = ts.getOnNextEvents();
                for (int j = 0; j < n; j++) {
                    Assert.assertEquals(j, list.get(j).intValue());
                }
            }
        } finally {
            w.unsubscribe();
        }
    }

    @Test
    public void nullElements() {
        TestSubscriber<Integer> ts = new TestSubscriber<Integer>(0);

        Observable.from(new Integer[] { 1, null, 2}).takeLast(4)
        .subscribe(ts);

        ts.assertNoValues();
        ts.assertNoErrors();
        ts.assertNotCompleted();

        ts.requestMore(1);

        ts.assertValue(1);
        ts.assertNoErrors();
        ts.assertNotCompleted();

        ts.requestMore(2);

        ts.assertValues(1, null, 2);
        ts.assertCompleted();
        ts.assertNoErrors();
    }

    @Test
    public void takeLastBuffer() {
        TestSubscriber<List<Integer>> ts = TestSubscriber.create();

        Observable.range(1, 5).takeLastBuffer(1).subscribe(ts);

        ts.assertValue(Arrays.asList(5));
        ts.assertNoErrors();
        ts.assertCompleted();
    }

    @Test
    public void takeLastBufferTimed() {
        TestSubscriber<List<Integer>> ts = TestSubscriber.create();

        Observable.range(1, 5).takeLastBuffer(5, TimeUnit.SECONDS).subscribe(ts);

        ts.awaitTerminalEventAndUnsubscribeOnTimeout(5, TimeUnit.SECONDS);
        ts.assertValue(Arrays.asList(1, 2, 3, 4, 5));
        ts.assertNoErrors();
        ts.assertCompleted();
    }

    @Test
    public void takeLastBufferTimedSized() {
        TestSubscriber<List<Integer>> ts = TestSubscriber.create();

        Observable.range(1, 5).takeLastBuffer(1, 5, TimeUnit.SECONDS).subscribe(ts);

        ts.awaitTerminalEventAndUnsubscribeOnTimeout(5, TimeUnit.SECONDS);
        ts.assertValue(Arrays.asList(5));
        ts.assertNoErrors();
        ts.assertCompleted();
    }

    @Test
    public void takeLastBufferTimedIO() {
        TestSubscriber<List<Integer>> ts = TestSubscriber.create();

        Observable.range(1, 5).takeLastBuffer(5, TimeUnit.SECONDS, Schedulers.io()).subscribe(ts);

        ts.awaitTerminalEventAndUnsubscribeOnTimeout(5, TimeUnit.SECONDS);
        ts.assertValue(Arrays.asList(1, 2, 3, 4, 5));
        ts.assertNoErrors();
        ts.assertCompleted();
    }

    @Test
    public void takeLastBufferTimedSizedIO() {
        TestSubscriber<List<Integer>> ts = TestSubscriber.create();

        Observable.range(1, 5).takeLastBuffer(1, 5, TimeUnit.SECONDS, Schedulers.io()).subscribe(ts);

        ts.awaitTerminalEventAndUnsubscribeOnTimeout(5, TimeUnit.SECONDS);
        ts.assertValue(Arrays.asList(5));
        ts.assertNoErrors();
        ts.assertCompleted();
    }

}
