/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.camel.kafkaconnector.sink.slack;

import org.apache.camel.kafkaconnector.EndpointUrlBuilder;
import org.apache.camel.kafkaconnector.SinkConnectorPropertyFactory;


/**
 * Creates the set of properties used by a Camel Slack Sink Connector
 */
final class CamelSlackPropertyFactory extends SinkConnectorPropertyFactory<CamelSlackPropertyFactory> {
    private CamelSlackPropertyFactory() {

    }

    public CamelSlackPropertyFactory withChannel(String channel) {
        return setProperty("camel.sink.path.channel", channel);
    }

    public CamelSlackPropertyFactory withWebhookUrl(String webhookUrl) {
        return setProperty("camel.sink.endpoint.webhookUrl", webhookUrl);
    }

    public EndpointUrlBuilder<CamelSlackPropertyFactory> withUrl(String channel) {
        String queueUrl = String.format("slack:%s", channel);

        return new EndpointUrlBuilder<>(this::withSinkUrl, queueUrl);
    }

    public static CamelSlackPropertyFactory basic() {
        return new CamelSlackPropertyFactory()
                .withName("CamelSlackSinkConnector")
                .withTasksMax(1)
                .withConnectorClass("org.apache.camel.kafkaconnector.slack.CamelSlackSinkConnector")
                .withKeyConverterClass("org.apache.kafka.connect.storage.StringConverter")
                .withValueConverterClass("org.apache.kafka.connect.storage.StringConverter");
    }
}
