package com.redhat.installer.asconfiguration.ascontroller;

import com.izforge.izpack.installer.AutomatedInstallData;
import org.jboss.as.cli.*;
import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.controller.client.helpers.Operations;
import org.jboss.dmr.ModelNode;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;
import java.util.logging.Logger;

public class ServerCommands {


    /**
     * Useful commands made final, change in one place if needed
     */
    private static final String RELOAD_CMD = "reload --admin-only=true";
    private static final String DOMAIN_CMD_PREFIX = "/host=%s";
    private static final String SHUTDOWN_CMD = ":shutdown";

    /**
     * Command Context related variables *
     */
    private CommandContext context;
    private boolean isDomain; // commands need to be modified if we're a domain
    private String[] domainProfiles; // the profile the domain commands should be
    // for use with DOMAIN_CMD_PREFIX

    /** This class needs to know about these. */
    //TODO: use JBossJDBCConstants?

    /**
     * These fields are logger related. For our uses, we simply use a List of
     * the commands we have entered on the context, unless we're in Batch Mode;
     * then we log the batch commands
     */
    private List<String> naiveLogger;
    private Logger logger;

    /**
     * Create an instance to connect using the given username / password If the
     * host parameter is null, the context will connect to the localhost.
     */
    private ServerCommands(String user, char[] pwd, String host, int port, boolean slave, String... profile) throws CliInitializationException {
        if (profile != null) {
            domainProfiles = profile;
            isDomain = true;
        }

        Map<String, String> newEnv = new HashMap<String, String>();
        newEnv.put("JBOSS_HOME", AutomatedInstallData.getInstance().getInstallPath() + File.separator);
        addEnv(newEnv);

        context = CommandContextFactory.getInstance().newCommandContext(host, user, pwd);
        context = ((CommandContext) AutomatedInstallData.getInstance().getAttribute("embedded.standalone"));
        context.setSilent(false);
        context.setResolveParameterValues(false);
        naiveLogger = new ArrayList<String>(1);
    }

    /**
     * Method to return an instance of ServerCommands that connects to a domain
     * ascontroller using a username and password
     * @param user    The username for authenticating with the management interface
     *                of the server in question
     * @param pwd     The password associated with the given username
     * @param port    The port upon which the management interface is listening
     * @param slave
     * @param profile The profile(s) that the Context should connect to on the host
     */

    public static ServerCommands createLocalDomainUsernameSession(String user, char[] pwd, int port, boolean slave, String... profile) throws CliInitializationException {
        return new ServerCommands(user, pwd, null, port, slave, profile);
    }

    /**
     * Disconnects the context from the host. The hostname and port are not
     * retained.
     */
    public void terminateSession() {
        if (!context.isTerminated()) {
            context.terminateSession();
        }
    }

    /**
     * Connects the ascontroller to the server that the commandContext was
     * instantiated with. also loads in all commands that were accumulated in
     * disconnected batch mode.
     * TODO: fix to use DMR composite operations
     */
    public void connectContext() throws CommandLineException {
        if (context.getControllerHost() == null && context.isBatchMode()) {
            // we were in disconnected batch mode.
            context.connectController();
            // add all of the commands in the log to the real batch.
            // these should all be safe already
            for (String command : naiveLogger) {
                try {
                    handle(command);
                } catch (CommandLineException | IOException e) {
                    e.printStackTrace();
                }
            }
        } else {
            // nothing fancy needed
            context.connectController();
        }
    }

    /**
     * A refactoring of functionality that used to exist within submitCommand. This method allows other methods to see what the command
     * actually being run is, rather than being stuck with a default command
     *
     * @param cmd jboss-cli.sh format command to process
     * @return the modified jboss-cli.sh command to fit the current server mode (domain, profiles, etc)
     */
    private String prepareCommand(String cmd) {
        if (isDomain) {
            if (cmd.contains("subsystem")) {
                cmd = "/profile=%s" + cmd;
                return cmd;
            } else if (cmd.contains(RELOAD_CMD)) {
                cmd = RELOAD_CMD + " --host=" + getDomainHostname();
                return cmd;
            } else if (cmd.contains("core-service") || cmd.contains(SHUTDOWN_CMD)) {
                cmd =  String.format(DOMAIN_CMD_PREFIX, getDomainHostname()) + cmd;
                return cmd;
            } else {
                // the logic is destroyed because of the new "apply domain commands to every profile in domainProfiles"
                return cmd;
            }
        } else {
            return cmd;
        }
    }

    /**
     * Helper method to submit a command to the context. This method exists to
     * help with logging all the commands, instead of explicitly adding log
     * statements to each command.
     *
     * @param cmd The jboss-cli.sh command to execute.
     */
    public ModelNode submitCommand(String cmd) {

        ModelNode result = null;
        cmd = prepareCommand(cmd);
        if (isDomain) {
            if (cmd.contains("subsystem")) {
                for (String profile : domainProfiles) {
                    String formattedCmd = String.format(cmd, profile);
                    // run the command for each profile
                    result = addToLoggerAndHandle(formattedCmd);
                }
            } else {
                result = addToLoggerAndHandle(cmd);
            }

        } else {
            result = addToLoggerAndHandle(cmd);
        }
        return result;
    }

    /**
     * Helper method to make the submitCommand method less cluttered.
     * Adds the given command to the logger, and also calls handle with it,
     * as long as it is safe to do so.
     * <p/>
     * Note: can return null if the context is not connect AND is in Batch mode
     *
     * @param command the jboss-cli.sh command to execute
     * @return a ModelNode describing the operation's success or failure
     */
    private ModelNode addToLoggerAndHandle(String command) {
        ModelNode result = null;
        naiveLogger.add(command);
        // if we are NOT in disconnected batch mode
        if (!(context.getControllerHost() == null && context.isBatchMode())) {
            try {
                result = handle(command);
            } catch (CommandLineException | IOException e) {
                result = getFailureResult(command, e.getMessage());
                e.printStackTrace();
            }
        }
        if (Operations.isSuccessfulOutcome(result)){
            ServerCommandsHelper.setCommand(result,command);
        }
        return result;
    }

    /**
     * Issues a shutdown command to the currently connected host.
     */

    public ModelNode shutdownHost() {
        return submitCommand(SHUTDOWN_CMD);
    }

    /**
     * Utility method for use when the ServerCommands instance is connected to a domain host, but the
     * ServerCommands instance may not be aware of this (not created through the *DomainSession* factory methods. <br/>
     * Explicitly calls shutdown on a domain host. This will result in failure if used on a standalone host.
     *
     * @return A ModelNode denoting the success or failure of the shutdown operation
     */
    public ModelNode shutdownDomainHost() {
        return submitCommand(String.format(DOMAIN_CMD_PREFIX, getDomainHostname()) + SHUTDOWN_CMD);
    }


    public void setLogger(Logger logger) {
        this.logger = logger;
        //TODO: is this string appropriate here? Also, possible for more efficiency?
        logger.info("Running commands from: " + logger.getName());
    }

    /**
     * Set environment variables into memory
     */
    /**
     * Append or overwrite environment variables to system variables in memory.
     *
     * @param newenv Map of additional variables you want to add
     */
    private static void addEnv(Map<String, String> newenv) {
        setEnv(newenv, true);
    }

    private static void setEnv(Map<String, String> newenv, final boolean append) {

        Map<String, String> env = System.getenv();
        if (append) {
            for (String key : env.keySet())
                if (newenv.get(key) == null) newenv.put(key, env.get(key));
        }

        Class[] classes = Collections.class.getDeclaredClasses();
        try {
            for (Class cl : classes) {
                if ("java.util.Collections$UnmodifiableMap".equals(cl.getName())) {
                    Field field = cl.getDeclaredField("m");
                    field.setAccessible(true);
                    Object obj = field.get(env);
                    Map<String, String> map = (Map<String, String>) obj;
                    map.clear();
                    map.putAll(newenv);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Replacement for the .handle method in the CommandContext, because it will print to console regardless of our settings.
     */
    private ModelNode handle(String command) throws CommandFormatException, IOException {
        ModelNode returnValue;
        if (logger != null) {
            logger.info(command);
        }
        ModelControllerClient mcc = context.getModelControllerClient();
        ModelNode request = context.buildRequest(command);
        if (mcc != null){
            returnValue = mcc.execute(request);
        } else {
            returnValue = getFailureResult(command, "No ModelControllerClient was available to execute the request.");
        }
        return returnValue;
    }

    /**
     * returns a ModelNode containing a failure. used for situations which fail-fast
     */
    private ModelNode getFailureResult(String cmd, String failureMsg) {
        ModelNode failedNode = new ModelNode();
        failedNode.get("outcome").set("failed");
        failedNode.get("failure-description").set(failureMsg);
        ServerCommandsHelper.setCommand(failedNode, cmd);
        return failedNode;
    }

    /**
     * gets the domainHostname.
     */
    public static String getDomainHostname() {
        return "master";
    }

    public boolean isDomain(){
        return isDomain;
    }

}
