package org.jboss.brmsbpmsuite.patching.client;

import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.Test;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsIn.isIn;

public class PatchFilesReaderTest {

    private PatchFilesReader reader;
    private Matcher<String> pathsPrefixMatcher;

    @Before
    public void setUp() {
        pathsPrefixMatcher = anyOf(startsWith("path/to/files"), startsWith("another/path"));

        File patchDir = new File(this.getClass().getClassLoader().getResource("patch-files-reader-test").getFile());
        reader = new PatchFilesReader(patchDir, DistributionType.RHPAM_EAP7X_BC);
    }

    @Test
    public void whenUpdateTypeIsNotMapped_checksumsShouldNotBeFiltered() {
        Map<String, List<Checksum>> checksums = reader.checksumsFor("unmapped-update");
        assertThat(checksums.keySet().size(), is(11));
    }

    @Test
    public void whenUpdateTypeIsNotMapped_newContentShouldNotBeFiltered() {
        List<PatchEntry> patchEntries = reader.newContentFor("unmapped-update");
        assertThat(patchEntries.size(), is(9));
    }

    @Test
    public void whenUpdateTypeIsNotMapped_removalsShouldNotBeFiltered() {
        List<String> removals = reader.removeListFor("unmapped-update");
        assertThat(removals.size(), is(2));
    }

    @Test
    public void whenUpdateTypeIsNotMapped_blacklistShouldNotBeFiltered() {
        List<String> blacklisted = reader.blacklist();
        assertThat(blacklisted.size(), is(2));
    }

    @Test
    public void whenUpdateTypeIsMapped_checksumsShouldBeFiltered() {
        Map<String, List<Checksum>> checksums = reader.checksumsFor("update1");
        assertThat(checksums.keySet().size(), is(5));
    }

    @Test
    public void whenUpdateTypeIsMapped_newContentShouldBeFiltered() {
        List<PatchEntry> patchEntries = reader.newContentFor("update1");
        assertThat(patchEntries.size(), is(5));
    }

    @Test
    public void whenUpdateTypeIsMapped_removalsShouldBeFiltered() {
        List<String> removals = reader.removeListFor("update1");
        assertThat(removals.size(), is(0));
    }

    @Test
    public void whenUpdateTypeIsNotMapped_checksumsPathsShouldNotBeTrimmed() {
        Map<String, List<Checksum>> checksums = reader.checksumsFor("unmapped-update");
        checksums.keySet().stream().forEach(path -> assertThat(path, pathsPrefixMatcher));
    }

    @Test
    public void whenUpdateTypeIsNotMapped_newContentPathsShouldNotBeTrimmed() {
        List<PatchEntry> patchEntries = reader.newContentFor("unmapped-update");
        patchEntries.stream().forEach(pathEntry -> assertThat(pathEntry.getRelativePath(), pathsPrefixMatcher));
    }

    @Test
    public void whenUpdateTypeIsNotMapped_removalsPathsShouldNotBeTrimmed() {
        List<String> removals = reader.removeListFor("unmapped-update");
        removals.stream().forEach(path -> assertThat(path, pathsPrefixMatcher));
    }

    @Test
    public void whenUpdateTypeIsMapped_checksumsPathsShouldBeTrimmed() {
        Map<String, List<Checksum>> checksums = reader.checksumsFor("update2");
        checksums.keySet().stream().forEach(path -> assertThat(path, not(pathsPrefixMatcher)));
    }

    @Test
    public void whenUpdateTypeIsMapped_newContentPathsShouldBeTrimmed() {
        List<PatchEntry> patchEntries = reader.newContentFor("update2");
        patchEntries.stream().forEach(pathEntry -> assertThat(pathEntry.getRelativePath(), not(pathsPrefixMatcher)));
    }

    @Test
    public void whenUpdateTypeIsMapped_removalsPathsShouldBeTrimmed() {
        List<String> removals = reader.removeListFor("update2");
        removals.stream().forEach(path -> assertThat(path, not(pathsPrefixMatcher)));
    }

    @Test
    public void whenAFileHasASingleVersion_itShouldBeTheOnlyEntryForItsMapKey() {
        Map<String, List<Checksum>> updateChecksums = reader.checksumsFor("update1");

        List<Checksum> duplicatedFileChecksums = updateChecksums.get("file4.txt");
        assertThat(duplicatedFileChecksums.size(), is(1));
    }

    @Test
    public void whenAFileHasMultipleVersions_itShouldBeAddedToTheSameMapKey() {
        Map<String, List<Checksum>> updateChecksums = reader.checksumsFor("update1");

        List<Checksum> duplicatedFileChecksums = updateChecksums.get("file-duplicated.txt");
        assertThat(duplicatedFileChecksums.size(), is(2));

        List<Object> fileChecksums = Arrays.asList("1111", "2222");
        assertThat(duplicatedFileChecksums.get(0).getHexValue(), isIn(fileChecksums));
        assertThat(duplicatedFileChecksums.get(1).getHexValue(), isIn(fileChecksums));
    }

}