package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.collect.Lists;
import org.apache.commons.io.FileUtils;
import org.junit.Test;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.FileTime;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.number.OrderingComparison.greaterThan;
import static org.junit.Assert.assertThat;

public class ClientPatcherRunnerTest extends BaseClientPatcherTest {

    @Test
    public void shouldSucceedWhenCorrectDistributionPathAndTypeSpecified() throws Exception {
        FileUtils.copyDirectory(
                getCPResourceAsFile("/client-patcher-runner-test/simple-distro"),
                tmpDir);

        File distroRoot = new File(tmpDir, "standalone/deployments/kie-server-ee8");
        File patchDir = getCPResourceAsFile("/client-patcher-runner-test/simple-patch");

        ClientPatcherConfig config = new ClientPatcherConfig();
        config.setPatchBasedir(patchDir);
        config.setDistributionRoot(distroRoot);
        config.setProduct(TargetProduct.RHPAM);
        config.setSupportedDistroTypes(Lists.newArrayList(DistributionType.RHPAM_EE8_KIE));
        config.setDistributionType(DistributionType.RHPAM_EE8_KIE);
        config.setBackupBaseDir(tmpDir);
        ClientPatcherRunner runner = new ClientPatcherRunner(config);
        runner.run();
        // just a dummy check that the patcher was correctly executed
        assertFileNotExists(new File(distroRoot, "drools-compiler-6.2.0.Final-redhat-3.jar"));
        assertFileExists(new File(distroRoot, "drools-compiler-6.2.0.Final-redhat-4.jar"));
    }

    @Test(expected = ClientPatcherException.class)
    public void shouldReportErrorWhenBackupFails() {
        ClientPatcherConfig config = ClientPatcherConfig.empty();
        ClientPatcherRunner runner = new ClientPatcherRunner(config);
        DistributionPatcher dummyPatcher = new DistributionPatcher() {
            @Override
            public void checkDistro() {
            }

            @Override
            public void checkVersion() {

            }

            @Override
            public void backup(File backupDir) throws IOException {
                throw new IOException("Backup failed - expected testing exception!");
            }

            @Override
            public void apply() {
            }

            @Override
            public void explodeWar() {

            }

            @Override
            public void zipWar() {

            }
        };
        runner.runDistributionPatcher(dummyPatcher, config.getBackupDir());
    }

    @Test(expected = ClientPatcherException.class)
    public void shouldReportErrorWhenApplyFails() {
        ClientPatcherConfig config = ClientPatcherConfig.empty();
        ClientPatcherRunner runner = new ClientPatcherRunner(config);
        DistributionPatcher dummyPatcher = new DistributionPatcher() {
            @Override
            public void checkDistro() {
            }

            @Override
            public void checkVersion() {

            }

            @Override
            public void backup(File backupDir) {
            }

            @Override
            public void apply() throws IOException {
                throw new IOException("Apply failed - expected testing exception!");
            }

            @Override
            public void explodeWar() {

            }

            @Override
            public void zipWar() {

            }
        };
        runner.runDistributionPatcher(dummyPatcher, config.getBackupDir());
    }

    @Test
    public void shouldNotPatchConfigurationWhenNotTargetingEAP7() throws Exception {
        FileUtils.copyDirectory(
                getCPResourceAsFile("/client-patcher-runner-test/simple-distro"),
                tmpDir);

        File distroRoot = new File(tmpDir, "standalone/deployments/kie-server-ee8");
        File patchDir = getCPResourceAsFile("/client-patcher-runner-test/simple-patch");

        File updatedFile = new File(tmpDir, "standalone/configuration/standalone.xml");
        long beforePatchFileSize = Files.size(updatedFile.toPath());
        FileTime beforePatchLastModifiedTime = Files.getLastModifiedTime(updatedFile.toPath());

        ClientPatcherConfig config = new ClientPatcherConfig();
        config.setPatchBasedir(patchDir);
        config.setDistributionRoot(distroRoot);
        config.setProduct(TargetProduct.RHPAM);
        config.setSupportedDistroTypes(Lists.newArrayList(DistributionType.RHPAM_EE8_KIE));
        config.setDistributionType(DistributionType.RHPAM_EE8_KIE);
        config.setBackupBaseDir(tmpDir);
        ClientPatcherRunner runner = new ClientPatcherRunner(config);
        runner.run();

        File backupFile = new File(config.getBackupDir(), "configuration/standalone.xml");
        assertThat("The backup file for standalone.xml has been generated!", backupFile.exists(), is(false));
        assertThat("There is not a standalone.xml file in the configuration!", updatedFile.exists(), is(true));
        assertThat("The standalone.xml file size has changed!", Files.size(updatedFile.toPath()), is(beforePatchFileSize));
        assertThat("The standalone.xml has been updated!", Files.getLastModifiedTime(updatedFile.toPath()), is(beforePatchLastModifiedTime));
    }

    @Test
    public void shouldPatchConfigurationWhenTargetingEAP7() throws Exception {
        FileUtils.copyDirectory(
                getCPResourceAsFile("/client-patcher-runner-test/simple-distro"),
                tmpDir);

        File distroRoot = new File(tmpDir, "standalone/deployments/decision-central-eap7-deployable");
        File patchDir = getCPResourceAsFile("/client-patcher-runner-test/simple-patch");

        File updatedFile = new File(tmpDir, "standalone/configuration/standalone.xml");
        long originalFileSize = Files.size(updatedFile.toPath());
        FileTime originalFileLastModifiedTime = Files.getLastModifiedTime(updatedFile.toPath());

        ClientPatcherConfig config = new ClientPatcherConfig();
        config.setPatchBasedir(patchDir);
        config.setDistributionRoot(distroRoot);
        config.setProduct(TargetProduct.RHDM);
        config.setSupportedDistroTypes(Lists.newArrayList(DistributionType.RHDM_EAP7X_DC));
        config.setDistributionType(DistributionType.RHDM_EAP7X_DC);
        config.setBackupBaseDir(tmpDir);
        ClientPatcherRunner runner = new ClientPatcherRunner(config);
        runner.run();

        File backupFile = new File(config.getBackupDir(), config.getDistributionType().getName() + "/configuration/standalone.xml");
        assertThat("The backup file for standalone.xml has not been generated!", backupFile.exists(), is(true));
        assertThat("The file standalone.xml is not in the the updated configuration!", updatedFile.exists(), is(true));
        assertThat("The standalone.xml file size has not changed!", Files.size(updatedFile.toPath()), is(greaterThan(originalFileSize)));
        assertThat("The standalone.xml file size has not been updated!", Files.getLastModifiedTime(updatedFile.toPath()), is(greaterThan(originalFileLastModifiedTime)));
    }

}
