/*
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 * 
 * http://izpack.org/
 * http://izpack.codehaus.org/
 * 
 * Copyright 2008 Piotr Skowronek
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.izforge.izpack.panels;

import com.izforge.izpack.gui.*;
import com.izforge.izpack.util.Debug;

import javax.swing.*;
import javax.swing.plaf.ColorUIResource;
import javax.swing.text.JTextComponent;
import javax.swing.text.NumberFormatter;

import java.awt.*;
import java.awt.event.KeyEvent;
import java.text.NumberFormat;
import java.util.Map;
import java.util.List;

/*---------------------------------------------------------------------------*/
/**
 * This class is a wrapper for JTextField to allow field validation.
 * Based on RuleInputField.
 *
 * @author Piotr Skowronek
 */
/*---------------------------------------------------------------------------*/
public class TextInputField extends JComponent implements ProcessingClient
{

    /**
     *
     */
    private static final long serialVersionUID = 8611515659787697087L;

    /**
     * Validator parameters.
     */
    private Map<String, String> validatorParams;

    /**
     * Holds an instance of the <code>Validator</code> if one was specified and available
     */
    private Validator validationService;

    /**
     * This composite can only contain one component ie JTextField
     */
    private JTextComponent field;

    /**
     * Do we have parameters for validator?
     */
    private boolean hasParams = false;

    private Map<String, String> currentValidatorParams = null;
    private List<ValidatorContainer> validatorConfig = null;
    
    /*--------------------------------------------------------------------------*/
    /**
     * Constructs a text input field.
     *
     * @param set       A default value for field.
     * @param size      The size of the field.
     * @param rows      The number of rows in case this is a text area field
     * @param validator A string that specifies a class to perform validation services. The string
     *                  must completely identify the class, so that it can be instantiated. The class must implement
     *                  the <code>RuleValidator</code> interface. If an attempt to instantiate this class fails, no
     *                  validation will be performed.
     * @param validatorParams validator parameters.
     */
    /*--------------------------------------------------------------------------*/
    public TextInputField(String set, int size, int rows, String validator, Map<String, String> validatorParams, boolean numbersOnly)
    {
        this.validatorParams = validatorParams;
        this.hasParams = validatorParams != null;
        
        // ----------------------------------------------------
        // attempt to create an instance of the Validator
        // ----------------------------------------------------
        try
        {
            if (validator != null)
            {
                Debug.trace("Making Validator for: " + validator);
                validationService = (Validator) Class.forName(validator).newInstance();
            }
        }
        catch (Throwable exception)
        {
            validationService = null;
            Debug.trace(exception);
        }

        com.izforge.izpack.gui.FlowLayout layout = new com.izforge.izpack.gui.FlowLayout();
        layout.setAlignment(com.izforge.izpack.gui.FlowLayout.LEADING);
        layout.setVgap(0);
        layout.setHgap(0);
        setLayout(layout);

        // ----------------------------------------------------
        // construct the UI element and add it to the composite
        // ----------------------------------------------------
        if (rows > 1) {
            JTextArea area = new JTextArea(set, rows, size);
            area.setCaretPosition(0);
            area.setWrapStyleWord(true);
            area.setLineWrap(true);
            
            field = area;
            add(ScrollPaneFactory.createScroller(area));
        } else {
            field = new JTextField(set, size);
            field.setCaretPosition(0);
            add(field);
        }
    }

    /*--------------------------------------------------------------------------*/
    /**
     * For use with multiple validators
     */
    /*--------------------------------------------------------------------------*/
    public TextInputField(String set, int size, int rows, List<ValidatorContainer> validators, boolean numbersOnly)
    {
        validatorConfig = validators;

        com.izforge.izpack.gui.FlowLayout layout = new com.izforge.izpack.gui.FlowLayout();
        layout.setAlignment(com.izforge.izpack.gui.FlowLayout.LEADING);
        layout.setVgap(0);
        layout.setHgap(0);
        setLayout(layout);

        // ----------------------------------------------------
        // construct the UI element and add it to the composite
        // ----------------------------------------------------
        if(numbersOnly) {
            JTextField textField = JTextFieldNum(set);
            textField.setColumns(size);
            field = textField;
            JFocusPanel focusBorder = new JFocusPanel(field);
            field.addFocusListener(new FocusListenerRepaint());
            add(focusBorder);
        }
        else if (rows > 1) {
            JTextArea area = new JTextArea(set, rows, size);
            area.setCaretPosition(0);
            area.setWrapStyleWord(true);
            area.setLineWrap(true);
            area.setFont(FontResources.getRegularBaseFont());
            
            field = area;
            field.addFocusListener(new FocusListenerAutoScroll());
            add(ScrollPaneFactory.createScroller(area));
        } else {
            field = new JTextField(set, size);
            field.setCaretPosition(0);
            field.addFocusListener(new FocusListenerAutoScroll());
            JFocusPanel focusBorder = new JFocusPanel(field);
            field.addFocusListener(new FocusListenerRepaint());
            add(focusBorder);
        }
    }

    public JTextField JTextFieldNum(String set) {
        JTextField textField = new JTextField(set) {
            public void processKeyEvent(KeyEvent ev) {
                char c = ev.getKeyChar();
                if (!Character.isLetter(c) && !isSymbol(c)) {
                    super.processKeyEvent(ev);
                }
            }
        };
        return textField;
    }

    private boolean isSymbol(char c) {
        String illegalSymbols = "~!@#$%^&*()_-=+{[}]|\\:;\"\',<>./?";
        return illegalSymbols.contains(""+c);
    }

    /*--------------------------------------------------------------------------*/
    /**
     * Returns the validator parameters, if any. The caller should check for the existence of
     * validator parameters via the <code>hasParams()</code> method prior to invoking this method.
     *
     * @return a java.util.Map containing the validator parameters.
     */
    public Map<String, String> getValidatorParams()
    {
        // if using only one validator
        if (validatorConfig == null) {
            return validatorParams;
        } else {
        // if using multiple validators
            return currentValidatorParams;
        }
    }

    /*---------------------------------------------------------------------------*/
    /**
     * Returns the field contents, assembled acording to the encryption and separator rules.
     *
     * @return the field contents
     */
    /*--------------------------------------------------------------------------*/
    public String getText()
    {
        return (field.getText());
    }

    // javadoc inherited
    public void setText(String value)
    {
        field.setText(value);
    }

    
    public JTextComponent getField()
    {
        return field;
    }

    
    public void setField(JTextComponent field)
    {
        this.field = field;
    }

    // javadoc inherited
    public String getFieldContents(int index)
    {
        return field.getText();
    }

    // javadoc inherited
    public int getNumFields()
    {
        // We've got only one field
        return 1;
    }

    /*--------------------------------------------------------------------------*/
    /**
     * This method validates the field content. Validating is performed through a user supplied
     * service class that provides the validation rules.
     *
     * @return <code>true</code> if the validation passes or no implementation of a validation
     *         rule exists. Otherwise <code>false</code> is returned.
     */
    /*--------------------------------------------------------------------------*/
    public Validator.Status validateContents()
    {
        if (validationService != null)
        {
            Debug.trace("Validating contents");
            return (validationService.validate(this));
        }
        else
        {
            Debug.trace("Not validating contents");
            return (Validator.Status.OK);
        }
    }

    public String validateContentsMultiValidators()
    {
        Validator validator = null;
        if (validatorConfig != null) 
        {
            for (ValidatorContainer valContainer : validatorConfig) {

                validator = valContainer.getValidator();
                if (valContainer.hasParams()) {
		    hasParams = true;
                    currentValidatorParams = valContainer.getValidatorParams();
                } else {
		    hasParams = false;
                    currentValidatorParams = null;
                }
                if (validator.validate(this) != Validator.Status.OK) {
                    String msg = valContainer.getMessage();
                    return msg;
                }
            }
            // if we reached this point, everything validated
            return "";
        }
        else
        {
            return "";
        }
    }

    // javadoc inherited
    public boolean hasParams()
    {
        return hasParams;
    }

    @Override
    public void setEnabled(boolean enabled)
    {
        super.setEnabled(enabled);
        field.setEnabled(enabled);
        field.setEditable(enabled);
    }

    public void setEditable(boolean editable)
    {
        field.setEditable(editable);
        if (!editable) {
            ColorUIResource c = new ColorUIResource(128,128,128);
            field.setBackground(UiResources.inactiveTextField);
            field.setForeground(c);
            field.setFocusable(false);
        }
        else {
            field.setBackground(Color.WHITE);
            field.setForeground(Color.BLACK);
            field.setFocusable(true);
        }
    }
    // ----------------------------------------------------------------------------
}
/*---------------------------------------------------------------------------*/
