/**
 * Copyright 2014 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rx.internal.operators;

import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.times;

import java.util.concurrent.TimeUnit;

import org.junit.Before;
import org.junit.Test;
import org.mockito.InOrder;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import rx.*;
import rx.functions.Func1;
import rx.plugins.RxJavaHooks;
import rx.schedulers.TestScheduler;
import rx.schedulers.TimeInterval;
import rx.subjects.PublishSubject;

public class OperatorTimeIntervalTest {

    private static final TimeUnit TIME_UNIT = TimeUnit.MILLISECONDS;

    @Mock
    private Observer<TimeInterval<Integer>> observer;

    private TestScheduler testScheduler;
    private PublishSubject<Integer> subject;
    private Observable<TimeInterval<Integer>> observable;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        testScheduler = new TestScheduler();
        subject = PublishSubject.create();
        observable = subject.timeInterval(testScheduler);
    }

    @Test
    public void testTimeInterval() {
        InOrder inOrder = inOrder(observer);
        observable.subscribe(observer);

        testScheduler.advanceTimeBy(1000, TIME_UNIT);
        subject.onNext(1);
        testScheduler.advanceTimeBy(2000, TIME_UNIT);
        subject.onNext(2);
        testScheduler.advanceTimeBy(3000, TIME_UNIT);
        subject.onNext(3);
        subject.onCompleted();

        inOrder.verify(observer, times(1)).onNext(
                new TimeInterval<Integer>(1000, 1));
        inOrder.verify(observer, times(1)).onNext(
                new TimeInterval<Integer>(2000, 2));
        inOrder.verify(observer, times(1)).onNext(
                new TimeInterval<Integer>(3000, 3));
        inOrder.verify(observer, times(1)).onCompleted();
        inOrder.verifyNoMoreInteractions();
    }

    @Test
    public void withDefaultScheduler() {

        final TestScheduler scheduler = new TestScheduler();

        RxJavaHooks.setOnComputationScheduler(new Func1<Scheduler, Scheduler>() {
            @Override
            public Scheduler call(Scheduler t) {
                return scheduler;
            }
        });

        try {
            InOrder inOrder = inOrder(observer);
            subject.timeInterval().subscribe(observer);

            scheduler.advanceTimeBy(1000, TIME_UNIT);
            subject.onNext(1);
            scheduler.advanceTimeBy(2000, TIME_UNIT);
            subject.onNext(2);
            scheduler.advanceTimeBy(3000, TIME_UNIT);
            subject.onNext(3);
            subject.onCompleted();

            inOrder.verify(observer, times(1)).onNext(
                    new TimeInterval<Integer>(1000, 1));
            inOrder.verify(observer, times(1)).onNext(
                    new TimeInterval<Integer>(2000, 2));
            inOrder.verify(observer, times(1)).onNext(
                    new TimeInterval<Integer>(3000, 3));
            inOrder.verify(observer, times(1)).onCompleted();
            inOrder.verifyNoMoreInteractions();
        } finally {
            RxJavaHooks.reset();
        }
    }
}
