package com.redhat.installer.installation.processpanel;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;


public class TextReplacer {
    private static final String FILE = "file";
    private static final String REGEX = "regex";
    private static final String TEXT = "text";
    private static final String REPLACE = "replace";
    private static final String SELF = "\\|\\|SELF\\|\\|";
    private static final String REMOVE = "remove";
    private static final AutomatedInstallData idata = AutomatedInstallData.getInstance();
    private static AbstractUIProcessHandler mHandler;

    public static boolean run(AbstractUIProcessHandler handler, String[] args) {
        mHandler = handler;
        ArgumentParser parser = new ArgumentParser();
        parser.parse(args);

        if (!parser.hasProperty(TEXT) || (parser.hasProperty(REGEX) && parser.hasProperty(REPLACE))) {
            ProcessPanelHelper.printToPanel(handler, "ProcessPanel.spec.xml has incorrect parameters", true);
            return false;
        }
        outputMessage(parser);
        List<String> filePaths = parser.getListProperty(FILE);
        for (String fileString : filePaths) {
            Path filePath = Paths.get(fileString);
            if (Files.exists(filePath)) {
                List<String> existingLines;
                try {
                    existingLines = Files.readAllLines(filePath);
                } catch (IOException e) {
                    ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.error"), fileString), true);
                    ProcessPanelHelper.printExceptionToLog(e.getStackTrace());
                    return false;
                }
                List<String> modifiedLines = modifyLines(parser, existingLines);
                int numLinesChanged = getNumLinesChanged(existingLines, modifiedLines);
                writeModifiedFile(filePath, modifiedLines, numLinesChanged);
            } else {
                ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.filenotfound"), filePath.toString()), false);
            }
        }
        return true;
    }

    private static int getNumLinesChanged(List<String> existingLines, List<String> modifiedLines) {
        int modifiedLinesCount = 0;
        for (int i = 0; i < existingLines.size(); i++) {
            if (!existingLines.get(i).equals(modifiedLines.get(i))) {
                modifiedLinesCount++;
            }
        }
        return modifiedLinesCount;
    }

    private static List<String> modifyLines(ArgumentParser parser, List<String> existingLines) {
        List<String> modifiedLines = new ArrayList<>();
        String newString = parser.getStringProperty(TEXT);
        String regexString;
        boolean hasRegex = parser.hasProperty(REGEX);
        boolean hasReplace = parser.hasProperty(REPLACE);
        boolean hasRemove = parser.hasProperty(REMOVE);
        if (hasRegex) {
            regexString = parser.getStringProperty(REGEX);
        } else if (hasReplace) {
            regexString = parser.getStringProperty(REPLACE);
        } else {
            regexString = "";
        }
        if (hasRegex) {
            for (String line : existingLines) {
                if (line.matches(regexString)) {
                    if (hasRemove) {
                        String removeString = parser.getStringProperty(REMOVE);
                        modifiedLines.add(line.replaceAll(removeString, "").trim());
                    } else {
                        String escapedLine = line.replaceAll("\\$", "\\\\\\$");
                        newString = newString.replaceAll(SELF, escapedLine);
                        newString = newString.replaceAll("\\$", "\\\\\\$");
                        line = line.replaceAll(regexString, newString);
                        modifiedLines.add(line);
                    }
                } else {
                    modifiedLines.add(line);
                }
            }
        } else if (hasReplace) {
            for (String line : existingLines) {
                if (line.contains(regexString)) {
                    modifiedLines.add(line.replace(regexString, newString));
                } else {
                    modifiedLines.add(line);
                }
            }
        }
        return modifiedLines;
    }

    private static void writeModifiedFile(Path filePath, List<String> modifiedLines, int linesChanged) {
        try (BufferedWriter writer = Files.newBufferedWriter(filePath)) {
            for (String line : modifiedLines) {
                writer.write(line);
                writer.write(System.getProperty("line.separator"));
            }
            ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.write"), filePath.toString(), linesChanged), false);
        } catch (IOException e) {
            ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.error"), filePath.toString()), true);
            ProcessPanelHelper.printExceptionToLog(e.getStackTrace());
        }
    }

    /**
     * Output formatted messages to GUI or console.
     *
     * @param parser
     */
    private static void outputMessage(ArgumentParser parser) {
        String newString = parser.getStringProperty(TEXT);
        List<String> filePaths = parser.getListProperty(FILE);
        if (parser.hasProperty(REGEX)) {
            String regex = parser.getStringProperty(REGEX);

            if (parser.hasProperty(REMOVE)) {
                String removeString = parser.getStringProperty(REMOVE);
                ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.removal"), removeString), false);

            } else {
                ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.replaceline"), regex, newString), false);

            }
        } else if (parser.hasProperty(REPLACE)) {
            String replaceString = parser.getStringProperty(REPLACE);
            ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("TextReplacer.replacetext"), replaceString, newString), false);
        }
        printFilePaths(mHandler, filePaths);
    }

    /**
     * Print the filepaths in sequence.
     *
     * @param handler
     * @param filePaths
     */
    private static void printFilePaths(AbstractUIProcessHandler handler, List<String> filePaths) {
        for (String file : filePaths) {
            ProcessPanelHelper.printToPanel(handler, file, false);
        }
    }
}

