package com.redhat.installer.installation.processpanel;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import org.apache.commons.lang.SystemUtils;
import org.apache.commons.lang.exception.ExceptionUtils;

import java.io.*;
import java.util.StringJoiner;
import java.util.stream.Collectors;

/**
 * This class is responsible for running arbitrary system commands
 * during the post-install process. The arguments supplied to it are,
 * in order:
 * 1 - Command to run (including full path to it if not a sys command)
 * 2 - arg1
 * 3 - arg2
 * ...
 *
 * @author fcanas@redhat.com
 */
public class CommandRunner {
    private static AbstractUIProcessHandler mHandler;

    public static boolean run(AbstractUIProcessHandler handler, String[] args) {
        mHandler = handler;
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        String workingDirectory = idata.getInstallPath() + File.separator;
        ArgumentParser parser = new ArgumentParser();
        parser.parse(args);
        boolean mute = args[0].equals("--mute=true");

        String[] arguments;
        if (mute) {
            arguments = new String[args.length - 1];
        } else {
            arguments = new String[args.length];
        }

        for (int i = 0; i < arguments.length; i++) {
            if (mute) {
                arguments[i] = args[i + 1];
            } else {
                arguments[i] = args[i];
            }
        }

        ProcessBuilder builder = new ProcessBuilder(arguments);
        builder.directory(new File(workingDirectory));
        ProcessPanelHelper.adjustJbossHome(builder);

        try {
            if (!mute) {
                ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("CommandRunner.running") + " " + builder.command(), false);
            }
            builder.redirectErrorStream(true);
            Process p = builder.start();
            try (BufferedReader in = new BufferedReader(new InputStreamReader(p.getInputStream()));
                 BufferedWriter out = new BufferedWriter(new OutputStreamWriter(p.getOutputStream()))) {
                if (SystemUtils.IS_OS_WINDOWS) {
                    out.newLine();
                    out.flush();
                }
                String processOutput = in.lines().collect(Collectors.joining(System.getProperty("line.separator")));
                ProcessPanelHelper.printToPanel(mHandler, processOutput, false);
            } catch (IOException e) {
                ProcessPanelHelper.printToPanel(mHandler, e.getMessage(), true);
                ProcessPanelHelper.printToLog(ExceptionUtils.getStackTrace(e));
            }
            int exit = p.waitFor();
            if (exit != 0) {
                if (!mute) {
                    ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("CommandRunner.failure") + " " + builder.command() + " " + idata.langpack.getString("CommandRunner.failure2") + " " + p.exitValue(), true);
                } else {
                    ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("CommandRunner.mute.failure") + " " + p.exitValue(), true);
                }
                return false;
            } else {
                if (!mute) {
                    ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("CommandRunner.success") + " " + builder.command() + " " + idata.langpack.getString("CommandRunner.success2"), false);
                }
            }

        } catch (InterruptedException | IOException e) {
            String errorMessage = idata.langpack.getString("CommandRunner.IOfailure");
            ProcessPanelHelper.printToPanel(mHandler, String.format(errorMessage, args[0]), true);
            ProcessPanelHelper.printToLog(ExceptionUtils.getStackTrace(e));
            return false;
        }

        return true;
    }
}

