#!/bin/sh

#
# The script that takes the following arguments(in any order):
#
# This script should be run from the root directory of {JON_REPO}.
#
# Assumed that customer/administrator will:
#
#include the utility library
source ../../../rhq_bash.lib

# stores off important variable definitions
declare -A LOADED;

#Script dependencies
PPX="--JP_";
CREATE_PPACK_FILE="generate-jon-pluginpack-update"
CREATE_PPACK_RDME="${CREATE_PPACK_FILE}-README.adoc"
CREATE_PPACK_CONF="pluginpack-update.conf"
CREATE_PPACK_TMPL="33xpluginTemplate.sh"

# Enable DEBUG_ON if DEBUG environment variable is set. # Defaults to false/off.
if [ -z "$DEBUG" ];
then
  DEBUG_ON="false";
else
  DEBUG_ON="true";
fi
echo "DEBUG_ON is [$DEBUG_ON]";
#========================================================================================
# Description: Initialize. Creates clean UPDATE_ROOT directory.
#========================================================================================
initialize()
{
 echo "initialize()..."; 

## Collect directory information for source generation
# Stores off the module directory
MODULE_DIR=`pwd`;

# Store off absolute path to target folder
TARGET="$MODULE_DIR"
debug "Initial directory: [$TARGET]";

TARGET="$TARGET/target"
debug "Creating target directory: [$TARGET]";

BUILD_DIR="$TARGET/BUILD"
debug "Creating BUILD_DIR directory: [$BUILD_DIR]";

# echo "Listing environment variables used in generating ${UPDATE_OPTIONS} update artifact:"
# environment_variables=( "UPDATE_ROOT" "UPDATE_TYPE" "GA_BREW_ID" "UPDATE_BREW_ID" "UPDATE_ID" "DEBUG_ON" "UPDATE_PLUGIN" )
# print_variables "${environment_variables[@]}"
# echo 

 ## clean target directory
#TODO: make this configurable? Only really matters for local builds and 'rm -rf' is easier...
# debug "rm -rf ${TARGET}"
# rm -rf "${TARGET}"

 debug "mkdir -p ${BUILD_DIR}"
 mkdir -p "${BUILD_DIR}"

 echo "Copying build artifacts into location..."
 debug "cp \"${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_FILE}\" \"${BUILD_DIR}/\"";
 cp "${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_FILE}" "${BUILD_DIR}/"
 debug "cp \"${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_RDME}\" \"${BUILD_DIR}/\"";
 cp "${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_RDME}" "${BUILD_DIR}/"
 debug " cp \"${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_CONF}\" \"${BUILD_DIR}/\"";
 cp "${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_CONF}" "${BUILD_DIR}/"
 debug " cp \"${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_TMPL}\" \"${BUILD_DIR}/\"";
 cp "${MODULE_DIR}/src/main/scripts/${CREATE_PPACK_TMPL}" "${BUILD_DIR}/"
}

#========================================================================================
# Description: Validate and parse input arguments
#========================================================================================
parse_and_validate_options()
{
   print_function_information $FUNCNAME

   #resetting default values.
   PACKAGE_RELEASE=
   BREW_PLUGIN_PACK_NAME=
   PLUGIN_PACK_NAME=
   PREVIOUS_UPDATE_NUMBER=
   NEW_UPDATE_NUMBER=
   UPDATE_ID=
   CHANGESET_DIFF=
   USE_SCRATCH_PPACK=
   MORE_OPTIONS="changeset-file:,plugin-pack-archive:,previous-plugin-pack-archive:,server-archive:,update-debug:,update-type:";
   #MORE_OPTIONS="$MORE_OPTIONS,";

   short_options="h"
   long_options="help,package-release:,brew-plugin-pack-name:,plugin-pack-name:,previous-update-number:,new-update-number:,update-id:,changeset-diff:,use-scratch-ppack:,${MORE_OPTIONS}";

   PROGNAME=${0##*/}
   ARGS=$(getopt -s bash --options $short_options --longoptions $long_options --name $PROGNAME -- "$@" )
   eval set -- "$ARGS"

   while true; do
	 case $1 in
         -h|--help)
            usage
            ;;
         --package-release)
            shift
            PACKAGE_RELEASE="$1"
            LOADED['packageRelease']="${PACKAGE_RELEASE}";
            shift
            ;;
         --brew-plugin-pack-name)
            shift
            BREW_PLUGIN_PACK_NAME="$1"
            LOADED['brewPluginPackName']="${BREW_PLUGIN_PACK_NAME}";
            shift
            ;;
         --plugin-pack-name)
            shift
            PLUGIN_PACK_NAME="$1"
            LOADED['pluginPackName']="${PLUGIN_PACK_NAME}";
            shift
            ;;
         --previous-update-number)
            shift
            PREVIOUS_UPDATE_NUMBER="$1"
            LOADED['previousUpdateNumber']="${PREVIOUS_UPDATE_NUMBER}";
            shift
            ;;
         --new-update-number)
            shift
            NEW_UPDATE_NUMBER="$1"
            LOADED['newUpdateNumber']="${NEW_UPDATE_NUMBER}";
            shift
            ;;
         --update-id)
            shift
            UPDATE_ID="$1"
	    LOADED['updateId']="${UPDATE_ID}";
            shift
            ;;
         --update-debug)
            shift
            DEBUG_ON="$1"
            shift
            ;;
         --changeset-diff)
            shift
            CHANGESET_DIFF="$1"
            shift
            ;;
         --use-scratch-ppack)
            shift
            USE_SCRATCH_PPACK="$1"
            shift
            ;;
         --JP_*|-JP_*)
           shift
           debug "[${key}] is being parsed \n";
           #echo -e "[${key}] is being parsed \n";
           KEY=$(echo "${key}" | cut -d'=' -f 1);
           if [[ "$KEY" == --JP_* ]] ;
           then
             KEY=${KEY:5}
           fi
           if  [[ "$KEY" == -JP_* ]] ;
           then
             KEY=${KEY:4}
           fi

           VAL=$(echo "${key}" | cut -d'=' -f 2);
           debug "\tParsed KEY[${KEY}] and VAL:[${VAL}]\n";
           #Now store the variables to the new value passed in
           LOADED[${KEY}]="${VAL}"; # add to associative array
           shift
         ;;
         --)
            shift
            break
            ;;
         *)
            VALUE=$1;
            shift
            debug "Unrecognized parameter ignoring [$VALUE] and it's content [$1].;"
            #echo "Unrecognized parameter ignoring [$VALUE] and it's content [$1].";
            #usage
            shift
            ;;
	 esac
   done

   if [ -z "$PACKAGE_RELEASE" ];
   then
      usage "The new 'package-release' identifier was not supplied. ! Ex. '105' -> is the incremental brew build number. Necessary for update builds."
   fi

   if [ -z "$BREW_PLUGIN_PACK_NAME" ];
   then
      usage "Brew plugin pack identifier not specified! Ex. 'eap' characters that appear in the plugin pack's file name in brew."
   fi

   if [ -z "$PLUGIN_PACK_NAME" ];
   then
      usage "Plugin pack identifier not specified! Ex. 'eap'. Characters that appear in the final name of plugin pack update."
   fi

   if [ -z "$PREVIOUS_UPDATE_NUMBER" ];
   then
      usage "Previous update number.  Ex. '04'. Update (N-1)."
   fi

   if [ -z "$NEW_UPDATE_NUMBER" ];
   then
      usage "New update number.  Ex. '05'. Update N."
   fi

   if [ -z "$UPDATE_ID" ];
   then
      usage "BZ update identifier not passed in. Ex. '1327633'. Identifies the Bugzilla tracker for this update."
   fi

   if [ -z "$CHANGESET_DIFF" ];
   then
      usage "Changeset is required for updates. Ex. '-rhq-jboss-as-7-plugin-4.12.0.JON330GA.jar'."
   fi

   print_centered "Script Options"
   script_options=( "PACKAGE_RELEASE" "BREW_PLUGIN_PACK_NAME" "PLUGIN_PACK_NAME" "PREVIOUS_UPDATE_NUMBER" "NEW_UPDATE_NUMBER" "UPDATE_ID" "CHANGESET_DIFF" "DEBUG_ON" "USE_SCRATCH_PPACK")
   print_variables "${script_options[@]}"
}

#========================================================================================
# Description: Display usage information then abort the script.
#========================================================================================
usage() 
{
   USAGE=$(
cat << EOF
USAGE:   launch-pluginpack-update.sh OPTIONS

   --package-release=number         [REQUIRED]
      The package release brew update number. Ex. 105. Update releases require a newer build identifier.

   --brew-plugin-pack-name=text     [REQUIRED]
      The plugin pack name as listed in brew Ex. eap

   --plugin-pack-name=text          [REQUIRED]
      The plugin pack name as is to be delivered to the customer. Ex. eap. In some cases the same brew 
      plugin pack can be repackaged as two different plugin packs.

   --previous-update-number=number  [REQUIRED]
      The version identifier of previous update released to customer. Ex. 04

   --new-update-number=number       [REQUIRED]
      The version identifier of new update for release to customer/qe. Ex. 05

   --update-id=number               [REQUIRED]
      The update number for this update artifact. Ex. jon-server-3.2.0.GA-update-01.zip

   --changeset-diff=string,(stringN)[REQUIRED]
      What needs to change from GA. Ex. -rhq-jboss-as-7-plugin-4.12.0.JON330GA.jar

   --update-debug=true              [OPTIONAL]
      Whether to enable debug logging for this build. Ex. true||false

   --use-scratch-ppack=url            [OPTIONAL]
      The brew url to retrieve the scratch plugin build contents from. Ex. http://download.devel.redhat.com/brewroot/work/tasks/...
EOF
)

   EXAMPLE="launch-pluginpack-update.sh --package-release=\"105\" --brew-plugin-pack-name=\"eap\" --plugin-pack-name=\"eap\" --previous-update-number=\"04\" --new-update-number=\"04\" --update-id=\"1327633\""

   abort "$@" "$USAGE" "$EXAMPLE"
}

debug()
{
 if [ "$DEBUG_ON" == "true" ];
  then
   if [ $# -eq 1 ];
    then
     echo -e "[DEBUG] Executing... [$1]"; # Log next steps
    else
     echo -e "[DEBUG] Variable $1 has value [$2]."; # display variable and value
   fi
 fi
}


############ MAIN SCRIPT ############

parse_and_validate_options $@

debug "Initializing...";
initialize

debug "BUILD_DIR" "$BUILD_DIR";

CURRENT=`pwd`;
debug "[pwd] is [$CURRENT]";

debug "LOADED array has ${#LOADED[@]} elements.";

#load all passed in content and override template values(if necessary)
for KEY in "${!LOADED[@]}"
do
  VALUE=${LOADED[$KEY]};
  debug "[KEY]:[$KEY] [VALUE]:[$VALUE]";
  declare "$KEY"="${VALUE}";
  export "${KEY}";
done 

debug "Sourcing ...[${BUILD_DIR}/${CREATE_PPACK_TMPL}]";
source "${BUILD_DIR}/${CREATE_PPACK_TMPL}";

echo "Generating changeFile and other dependent variables.";

##Execute the necessary operations in order using values passed in.
changeFile=BZ${updateId}-changeset.txt
newPluginPack="jon-plugin-pack-${pluginPackName}-${packageVersion}-${packageRelease}.zip"
previousPluginPack="jon-plugin-pack-${pluginPackName}-${packageVersion}$( [ -n "${previousUpdateNumber}" ] && echo -update-${previousUpdateNumber}).zip"
serverArchive="jon-server-${packageVersion}-${packageRelease}.zip"

debug "changeFile" "${changeFile}";
debug "newPluginPack" "${newPluginPack}";
debug "previousPluginPack" "${previousPluginPack}";
debug "serverArchive" "${serverArchive}";

debug "Testing same variables after keyed up load";
debug "brewRootUrl" "${brewRootUrl}";
debug "brewPluginPackName" "${brewPluginPackName}";
debug "updateId" "${updateId}";
debug "changeFile" "${changeFile}";
debug "newPluginPack" "${newPluginPack}";
debug "previousPluginPack" "${previousPluginPack}";
debug "serverArchive" "${serverArchive}";
debug "packageVersion" "${packageVersion}";

#generate changeset file
VALUE=${CHANGESET_DIFF};
debug "[CHANGESET] is [$VALUE]";
if [ -f "${BUILD_DIR}/${changeFile}" ];
then
  debug "[changeFile] is [$changeFile]. Detected and removed for correct regeneration.";
  rm "${BUILD_DIR}/${changeFile}"
fi

for word in $(echo "$VALUE" | sed -n 1'p' | tr ',' '\n')
 do 
  echo "${word}" >> "${BUILD_DIR}/${changeFile}";
done

debug "cd \"$BUILD_DIR\"";
cd "$BUILD_DIR";

debug "Listing the current state of variable values from i)template ii)command line parameters...";
debug "serverArchive" "${serverArchive}";
debug "brewRootUrl" "${brewRootUrl}";
debug "packageName" "${packageName}";
debug "packageVersion" "${packageVersion}";
debug "packageRelease" "${packageRelease}";
debug "-------------------------------------";
debug "newPluginPack" "${newPluginPack}";
debug "brewPluginPackName" "${brewPluginPackName}";
debug "previousPluginPack" "${previousPluginPack}";
debug "-------------------------------------";
debug "productRootURL" "${productRootURL}";
debug "previousUpdateNumber" "${previousUpdateNumber}";

##conditionally pull down content
# Download server archive from Brew
#[ -e "${serverArchive}" ] && ZFLAG="-z ${serverArchive}" || ZFLAG=""
if [ ! -f "${serverArchive}" ];
then
 debug "wget --no-check-certificate \"${brewRootUrl}/packages/${packageName}/${packageVersion}/${packageRelease}/maven/org/jboss/on/jon-server/${packageVersion}/jon-server-${packageVersion}.zip\" -O \"${serverArchive}\"";
 wget --no-check-certificate "${brewRootUrl}/packages/${packageName}/${packageVersion}/${packageRelease}/maven/org/jboss/on/jon-server/${packageVersion}/jon-server-${packageVersion}.zip" -O "${serverArchive}";
 #curl -R -L ${ZFLAG} -o "${serverArchive}" "${brewRootUrl}/packages/${packageName}/${packageVersion}/${packageRelease}/maven/org/jboss/on/jon-server/${packageVersion}/jon-server-${packageVersion}.zip"
fi

# Download plug-in pack archive from Brew
#[ -e "${newPluginPack}" ] && ZFLAG="-z ${newPluginPack}" || ZFLAG=""
if [ ! -f "${newPluginPack}" ];
then
 debug "wget --no-check-certificate -O \"${newPluginPack}\" \"${brewRootUrl}/packages/${packageName}/${packageVersion}/${packageRelease}/maven/org/jboss/on/jon-plugin-pack-${brewPluginPackName}/${packageVersion}/jon-plugin-pack-${brewPluginPackName}-${packageVersion}.zip\"";
 wget --no-check-certificate -O "${newPluginPack}" "${brewRootUrl}/packages/${packageName}/${packageVersion}/${packageRelease}/maven/org/jboss/on/jon-plugin-pack-${brewPluginPackName}/${packageVersion}/jon-plugin-pack-${brewPluginPackName}-${packageVersion}.zip"
 #curl -R -L ${ZFLAG} -o "${newPluginPack}" "${brewRootUrl}/packages/${packageName}/${packageVersion}/${packageRelease}/maven/org/jboss/on/jon-plugin-pack-${brewPluginPackName}/${packageVersion}/jon-plugin-pack-${brewPluginPackName}-${packageVersion}.zip"
fi

## TODO: insert ability to pass in earlier scratch content and replace here.
# Download previous plug-in pack from internal downloads site
#[ -e "${previousPluginPack}" ] && ZFLAG="-z ${previousPluginPack}" || ZFLAG=""
if [ ! -f "${previousPluginPack}" ];
then
 debug "PROCESS_PACKAGE=$( [ -n \"${previousUpdateNumber}\" ] && echo \"${pluginPackName}\" || echo \"${brewPluginPackName}\")";
 PROCESS_PACKAGE=$( [ -n "${previousUpdateNumber}" ] && echo "${pluginPackName}" || echo "${brewPluginPackName}");
 debug "PROCESS_PACKAGE" "${PROCESS_PACKAGE}";
 debug "PROCESS_UPDATE_NUMBER=$( [ -n \"${previousUpdateNumber}\" ] && echo -update-${previousUpdateNumber});";
 PROCESS_UPDATE_NUMBER=$( [ -n "${previousUpdateNumber}" ] && echo -update-${previousUpdateNumber});
 debug "PROCESS_UPDATE_NUMBER" "${PROCESS_UPDATE_NUMBER}";
 #debug "wget --no-check-certificate -O \"${previousPluginPack}\" \"${productRootURL}/${productVersion}/jon-plugin-pack-$( [ -n "${previousUpdateNumber}" ] && echo "${pluginPackName}" || echo "${brewPluginPackName}")-${packageVersion}$( [ -n "${previousUpdateNumber}" ] && echo -update-${previousUpdateNumber}).zip"";
 #wget --no-check-certificate -O "${previousPluginPack}" "${productRootURL}/${productVersion}/jon-plugin-pack-$( [ -n "${previousUpdateNumber}" ] && echo "${pluginPackName}" || echo "${brewPluginPackName}")-${packageVersion}$( [ -n "${previousUpdateNumber}" ] && echo -update-${previousUpdateNumber}).zip"
 debug "wget --no-check-certificate -O \"${previousPluginPack}\" \"${productRootURL}/${productVersion}/jon-plugin-pack-${PROCESS_PACKAGE}-${packageVersion}${PROCESS_UPDATE_NUMBER}.zip\";";
 wget --no-check-certificate -O "${previousPluginPack}" "${productRootURL}/${productVersion}/jon-plugin-pack-${PROCESS_PACKAGE}-${packageVersion}${PROCESS_UPDATE_NUMBER}.zip";
 #curl -R -L ${ZFLAG} -o "${previousPluginPack}" "${productRootURL}/${productVersion}/jon-plugin-pack-$( [ -n "${previousUpdateNumber}" ] && echo "${pluginPackName}" || echo "${brewPluginPackName}")-${packageVersion}$( [ -n "${previousUpdateNumber}" ] && echo -update-${previousUpdateNumber}).zip"
fi

pwd
ls -al *.zip

# generate expected structure for assembly
debug "Creating...[${TARGET}/jon-plugin-pack-${pluginPackName}-${packageVersion}-update-${newUpdateNumber}]";
debug "mkdir \"${TARGET}/jon-plugin-pack-${pluginPackName}-${packageVersion}-update-${newUpdateNumber}\"";
mkdir "${TARGET}/jon-plugin-pack-${pluginPackName}-${packageVersion}-update-${newUpdateNumber}";

