/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package componenttest.vulnerability;

import static org.junit.Assert.assertEquals;

import java.util.Collections;
import java.util.List;

import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;

import com.ibm.websphere.simplicity.log.Log;
import componenttest.annotation.CheckForLeakedPasswords;
import componenttest.topology.impl.LibertyServer;

/**
 * Test rule to verify passwords for tests annotated
 * with the {@link componenttest.annotation.CheckForLeakedPasswords} annotation.
 * To use, create a LeakedPasswordChecker test rule passing the LibertyServer as shown and
 * annotate the test method(s) with the {@link componenttest.annotation.CheckForLeakedPasswords} annotation.
 * <pre>
 * &#064;Rule
 * public TestRule passwordChecker = new LeakedPasswordChecker(server);
 * </pre>
 * 
 */
public class LeakedPasswordChecker implements TestRule {

    private LibertyServer server = null;
    private CheckForLeakedPasswords testPasswords = null;

    public LeakedPasswordChecker(LibertyServer server) {
        this.server = server;
    }

    @Override
    public Statement apply(final Statement statement, final Description description) {
        testPasswords = description.getAnnotation(CheckForLeakedPasswords.class);
        return new Statement() {

            @Override
            public void evaluate() throws Throwable {
                statement.evaluate();
                if (testPasswords != null) {
                    Log.info(LeakedPasswordChecker.class, "evaluate",
                             "Verifying that there were no passwords leaked for test " + description.getMethodName());
                    String[] passwords = testPasswords.value();
                    testPasswords = null;
                    for (String password : passwords) {
                        checkForPasswordInTrace(password);
                    }
                }
            }
        };
    }

    /**
     * Checks to see if the specified password was found in the trace.
     * 
     * @param password
     */
    public void checkForPasswordInTrace(String password) {
        List<String> passwordsInTrace = Collections.emptyList();
        try {
            passwordsInTrace = server.findStringsInLogsAndTrace(password);
        } catch (Exception e) {
        }
        assertEquals("Should not find password in the log file", Collections.emptyList(), passwordsInTrace);
    }

    /**
     * Checks to see if any XOR encoded passwords were found in the trace.
     * 
     * @param password
     */
    public void checkForXOREncodedPasswordsInTrace() {
        checkForPasswordInTrace("\\{xor\\}");
    }

    /**
     * Checks to see if the specified password was found in the trace,
     * including XOR encoded
     * 
     * @param password
     */
    public void checkForPasswordInAnyFormat(String password) {
        checkForPasswordInTrace(password);
        checkForXOREncodedPasswordsInTrace();
    }
}
