/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.mqst.jetsam;

import java.io.*;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Enumeration;

/******************************************************************************
 * @(#) mqjava/com/ibm/mqst/jetsam/JETSAMLog.java, mqjava, base 1.13 02/09/18 14:27:28
 ******************************************************************************
 * Copyright I.B.M. Corporation 1998, 2002. All Rights Reserved.
 ******************************************************************************/

/**
  * This class defines the JETSAM logging functions. It may be
  * advisable to extend this model slightly for the treatment
  * of Exceptions - for instance, if there is a concept of a
  * linked Exception then they will need to be logged also.
  *
  * @version 1.0 - September 1999
  * @author Ewan Withers
  *

  *
  * Modified 10/07/01 by Ben Walsh   //@BW
  * - Added a setErrors() method that will set the number of errors generated by the log 
  *
  * Modified 16/07/01 by Ben Walsh   //@BW2
  * - Added synchronization to improve sequencing of output, within threaded tests etc ...
  *
  * Modified 17/07/01 by Ben Walsh   //@BW3
  * - Added flushing to improve sequencing of event logs
  *
  * Modified 08/04/02 by Gareth Matthews  //@GM10
  * - Added a displayTestInfo method and made all tests print this out to their log
  *   file before running the test.
  **/

public class JETSAMLogImpl implements JETSAMLog
{
  // Added at version 1.13
  private static final long serialVersionUID = -9046705396963563375L;

  /** The performance monitor. */
  private transient JETSAMPerformanceMonitor perfMon;

  /** The output mode, maximal output or minimal. */
  protected boolean outputToScreen;
  
  /** 
   * Whether we write the output to a file... 
   * This is predominantly for the ability to disable all output in case
   * you want to run stress-tests.
   */
  protected boolean outputToFile;

  /** The filename of the associated log file. */
  protected String fileName;

  /** The PrintWriter writing to the log file. */
  protected transient PrintWriter logFile = null;

  /** The incremental error counter. */
  protected int errors = 0;

	/**
	 * Utility constructor that automatically produces output to a file.
	 */
	public JETSAMLogImpl(String fn, boolean screen)
	{
		this(fn, screen, true);	
	}
	
  /** Construct a new JETSAMLog object.
    * 
    * @param fn The name of the log file.
    * @param v The output mode - true for verbose, false for quiet.
    */
  public JETSAMLogImpl(String fn, boolean screen, boolean file)
  {
    fileName = fn;
    outputToScreen = screen;
    outputToFile = file;
    // Create a new perfance monitor object in case we need it
    perfMon = new JETSAMPerformanceMonitor();
  }

  public static JETSAMLog createLog(
    String fileName,
    boolean outputToScreen,
    boolean fileOutput)
  {

    JETSAMLog log = new JETSAMLogImpl(fileName, outputToScreen, fileOutput);
    return log;

  }

  /** Open the logfile for use. Creates a new PrintWriter,
    * using the class variable fileName. If the boolean
    * parameter is true, any existing file of the supplied name
    * is overwritten. Otherwise, the file is opened
    * for append, thus enabling updating if the logfile is
    * closed and then re-opened. If an error is detected opening
    * the file then the tests are not run - the logs are absolutely
    * required for sensible testing.
    *
    * @param newFile Whether to overwrite an existing file of the same name.
    */

  public synchronized void open(boolean newFile)
  {
  	
  	// We now have the option of not outputting to a file.
  	if (!outputToFile)
  	{
  		if (outputToScreen)
  		{
  			System.err.println("You selected not to produce file output");	
  		}
  		
  		// early method return.
  		logFile = null;
  		return;
  	}
  	
    errors = 0;
    boolean append = !newFile;
    
    // First of all, check whether the directory that is supplied exists.
    // Calculate the path to the file.
    String path = fileName.substring(0, fileName.lastIndexOf(File.separatorChar));
    
    // Create the directory if it does not exist.
    File dir = new File(path);
    if (!dir.exists())
    {
    	dir.mkdir();
    	
    }
    

    try
    {
      if (logFile == null)
        logFile = new PrintWriter(new FileOutputStream(fileName, append));
    } catch (Exception e)
    {
      // No FFDC code needed
      System.err.println(
        "Unable to open log file due to the following excepion:");
      e.printStackTrace(System.err);
      System.err.flush(); //@BW3
      System.exit(1);
    }
  }

  /** Add a String comment to the logfile. This is echoed to the
    * screen if the output mode is set to true.
    * 
    * @param text The comment to be logged.
    */

  public synchronized void comment(String text) //@BW2
  {
    synchronized(System.err)
    {
    
    if (outputToScreen)
    {

      System.err.print("  ");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.print("  ");
    	logFile.flush(); //@BW3
		}

    if (outputToScreen)
    {

      System.err.println(text);
      System.err.flush(); //@BW3

    }

		if (outputToFile  && logFile != null)
		{
      // p  181814  Multi-line comments in file logs  
      // We need to check if there are any newline characters in this text,
      // and if so, then print it a line at a time.
      if (text.indexOf("\n") != -1)
      {
        StringTokenizer st = new StringTokenizer(text.trim(), "\n");
        while (st.hasMoreTokens())
        {
          String nextLine = st.nextToken();
          logFile.println(nextLine);
          
        }//while
        
      } else
      {
        // Print a single line comment.
        logFile.println(text);
      }
    	
    	logFile.flush(); //@BW3
		}
    
    }//sync

  }

  /** Add an Exception comment to the logfile. This is echoed to the
    * screen if the output mode is set to true.
    * 
    * @param e The Exception to be logged.
    */

  public synchronized void comment(Exception e) //@BW2
  {

		String filteredStack = JETSAMLogImpl.getFilteredTrace(e);
		synchronized(System.err)
    {
    
    if (outputToScreen)
    {

      System.err.print("  ");
      System.err.flush(); //@BW3
      System.err.println(filteredStack);
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
	    logFile.print("  ");
	    logFile.flush(); //@BW3
	    logFile.println(filteredStack);
	    logFile.flush();
	
	    // XXX Should probably provide another option for JDK1.4 getCause option.
	
	    logFile.flush(); //@BW3
		}
    
    }//sync
    
  }

  /** Add an Error comment to the logfile. This is echoed to the
    * screen if the output mode is set to true.
    *
    * @param e The Error to be logged.
    */

  public synchronized void comment(Throwable e) //@BW2
  {
  	
  	String filteredStack = JETSAMLogImpl.getFilteredTrace(e);
    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.print("  ");
      System.err.flush(); //@BW3
      //e.printStackTrace();
      System.err.println(filteredStack);
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
	    logFile.print("  ");
	    logFile.flush(); //@BW3
	    //e.printStackTrace(logFile);
	    logFile.println(filteredStack);
	    logFile.flush(); //@BW3
		}
    
    }//sync
    
  }

  /** Add a String and an Exception comment to the logfile.
    * These are appended to the log as well as echoed to the
    * screen if the output mode is set to true.
    * 
    * @param text The comment to be logged.
    * @param e The Exception to be logged.
    */

  public synchronized void comment(String text, Exception e) //@BW2
  {
    synchronized(System.err)
    {
      
    comment(text);
    comment(e);
   
    }//sync
    
  }

  /** Add a String and an Error comment to the logfile.
    * These are appended to the log as well as echoed to the
    * screen if the output mode is set to true.
    * 
    * @param text The comment to be logged.
    * @param e The Error to be logged.
    */

  public synchronized void comment(String text, Error e) //@BW2
  {
    synchronized(System.err)
    {
      
    comment(text);
    comment(e);
    
    }//sync
  }

  /** Add a String error to the logfile. This is appended
    * to the log and echoed to the screen if the output mode is
    * set to true. The log error count is also incremented.
    * 
    * @param text The error text to be logged.
    */

  public synchronized void error(String text) //@BW2
  {

    errors++;
    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.print("**** ERROR ****");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.print("**** ERROR ****");
    	logFile.flush(); //@BW3
		}

    comment(text);
    
    }//sync

  }

  /** Add an Exception error to the logfile. This is appended
    * to the log as well as echoed to the screen if the output mode is
    * set to true. The log error count is also incremented.
    * 
    * @param e The Exception error to be logged.
    */

  public synchronized void error(Exception e) //@BW2
  {

    errors++;
    synchronized(System.err)
    {
    
    if (outputToScreen)
    {

      System.err.print("**** EXCEPTION ****");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.print("**** EXCEPTION ****");
    	logFile.flush(); //@BW3
		}

    comment(e);
    
    }//sync

  }

  /** Add an Error error to the logfile. This is appended
    * to the log as well as echoed to the screen if the output mode is
    * set to true. The log error count is also incremented.
    * 
    * @param e The Error error to be logged.
    */

  public synchronized void error(Throwable e) //@BW2
  {

    errors++;
    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.print("**** EXCEPTION ****");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.print("**** EXCEPTION ****");
    	logFile.flush(); //@BW3
		}

    comment(e);
    
    }//sync

  }

  /** Add a String and an Exception error to the logfile.
    * These are appended to the log as well as echoed to
    * the screen if the output mode is
    * set to true. The log error count is also incremented.
    * 
    * @param text The error text to be logged.
    * @param e The Exception error to be logged.
    */

  public synchronized void error(String text, Exception e) //@BW2
  {

    errors++;
    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.print("**** EXCEPTION ****");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.print("**** EXCEPTION ****");
    	logFile.flush(); //@BW3
		}

    comment(text, e);
    
    }//sync

  }

  /** Add a String and an Error error to the logfile.
    * These are appended to the log as well as echoed to
    * the screen if the output mode is
    * set to true. The log error count is also incremented.
    * 
    * @param text The error text to be logged.
    * @param e The Error error to be logged.
    */

  public synchronized void error(String text, Error e) //@BW2
  {

    errors++;
    
    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.print("**** EXCEPTION ****");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.print("**** EXCEPTION ****");
    	logFile.flush(); //@BW3
		}

    comment(text, e);
    
    }//sync

  }

  /** Gets the current number of errors generated by the log.
    * 
    * @return The current number of logged errors.
    */

  public int getErrors()
  {
    return errors;
  }

  /** Sets the number of errors generated by the log.
    *
    * @param setErrors The number of errors
    *
    */

  public synchronized void setErrors(int setErrors) //@BW2
  { //@BW
    errors = setErrors; //@BW
  } //@BW

  /** Inserts a blank line into the logfile.
    * Echoed to the screen if the output mode is set to true.
    */

  public synchronized void blankLine()
  {
    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.println("");
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.println("");
    	logFile.flush(); //@BW3
		}
    
    }//sync
		
  }

  /** Inserts a section (non-indented comment) into the log.
    * Echoed to the screen if the output mode is set to true.
    */

  public synchronized void section(String sectionName)
  {

    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.println(sectionName);
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.println(sectionName);
    	logFile.flush(); //@BW3
		}
    
    }//sync

  }

  /** Inserts a header (a section, a blank line and a timestamp)
    * into the log. Echoed to the screen if the output mode is set to true.
    */

  public synchronized void header(String hdrStr)
  {
    synchronized(System.err)
    {
      
    section(hdrStr);
    blankLine();
    timestamp();
    
    }//sync
  }

  /** Inserts a timestamp into the log.
    * Echoed to the screen if the output mode is set to true.
    */

  public void timestamp()
  {
    Date date = new Date();
    String str = date.toString();

    synchronized(System.err)
    {
      
    if (outputToScreen)
    {

      System.err.println("  Timestamp : " + str);
      System.err.flush(); //@BW3

    }

		if (outputToFile && logFile != null)
		{
    	logFile.println("  Timestamp : " + str);
    	logFile.flush(); //@BW3
		}
    
    }//sync

  }

  /** Starts and stops performance measurement.
    * Call this function once to start the timer, and
    * call it again with the same (arbitrary) String argument to
    * terminate the timer. Typically calls to this
    * method will bracket a call to the function(s) under test.
    *
    * @param name The name of the function being performance-tested.
    */

  public void performance(String name)
  {
    perfMon.performance(name);
  }

  /** Writes all performance statistics into the logfile. */

  public synchronized void performanceStats()
  {
    perfMon.dumpStats(this);
  }

  /** Closes the logfile. */

  public synchronized void close()
  {
    if (logFile != null)
    {
      logFile.flush(); //@BW3
      logFile.close();
      logFile = null;
    }
  }

  /** This method will return whether a log file is open for writing
    * or not.
    *
    * @return Returns true if the logfile is available
    */
  public synchronized boolean isOpen()
  { //@GM10
    if (logFile != null)
    { //@GM10
      return true; //@GM10
    } //@GM10
    else
    { //@GM10
      return false; //@GM10
    } //@GM10
  } //@GM10

  /**
   * Returns the fileName.
   * @return String
   */
  public String getFileName()
  {
  	if (outputToFile)
  	{
    	return fileName;
  	} else
  	{
  		return "";	
  	}
  	
  }
  
    // ********************* STACK FILTERING CODE **************************

	private static String[] patterns = null;
	private static boolean filterEnabled = true;
	
	/**
	 * Use this to set the stack trace lines to be filtered from stack prints.
	 */
	public static void setFilter(String[] newPatterns)
	{
		patterns = newPatterns;
	}
	
	public static void setFilterEnabled(boolean en)
	{
		filterEnabled = en;
	}
	
	// This code is lifted from the JUnit BaseTestRunner class and is used to
	// remove unwanted blocks of trace from a stack trace.
	static boolean filterLine(String line)
  {
    
    // If no patterns have been set then we can't possibly filter them!
    if ((patterns == null) || (!filterEnabled)) return false;
    
    for (int i = 0; i < patterns.length; i++)
    {
      if (line.indexOf(patterns[i]) > 0)
        return true;
    }
    return false;
  }
  
  /**
   * Returns a filtered stack trace
   */
  public static String getFilteredTrace(Throwable t)
  {
    StringWriter stringWriter = new StringWriter();
    PrintWriter writer = new PrintWriter(stringWriter);
    t.printStackTrace(writer);
    StringBuffer buffer = stringWriter.getBuffer();
    String trace = buffer.toString();
    return JETSAMLogImpl.filterStack(trace);
  }

  /**
   * Filters stack frames from internal JUnit classes
   */
  public static String filterStack(String stack)
  {

    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    StringReader sr = new StringReader(stack);
    BufferedReader br = new BufferedReader(sr);

    String line;
    try
    {
      while ((line = br.readLine()) != null)
      {
        if (!filterLine(line))
          pw.println(line);
      }
    } catch (Exception IOException)
    {
      // No FFDC code needed
      return stack; // return the stack unfiltered
    }
    return sw.toString().trim();
  }

}

/** This class acts as the controller for getting performance statistics. It has two
  * methods to do this, the performance method should be called to start and stop the 
  * statistics and the dump stats will display the results in a JETSAMLog.
  *
  * @author Ewan Withers
  *
  * @see com.ibm.mqst.jetsam.JETSAMPerformanceStatistic
  */
class JETSAMPerformanceMonitor
{
  /** This instance variable stores JETSAMPerformanceStatistic objects */
  private Vector stats = null;

  /** Constructor for this class which simply instantiates the stats Vector
    */
  public JETSAMPerformanceMonitor()
  {
    stats = new Vector();
  }

  /** This is the method that will control when performance logging starts and finishes.
    * Passed into this method is a performance name which identifies the logging, this is
    * so that is possible for one test to have many performance statistics which can all be
    * run at the same time or at different times. The first time this method is called, a
    * new JETSAMPerformanceStatistic will be created and added to the stats Vector, marking
    * this object with the performance name. Starting the Performance Statistic object will
    * start timing, and then a call to this method again will cause the performance logging
    * to stop and the statistics will be stored.
    *
    * @param perfName. The name to identify this performance statistic
    */
  public void performance(String perfName)
  {
    // Get the current time so that if we are stopping the performance logging,
    // we get an accurate end time.
    long curTime = System.currentTimeMillis();

    Enumeration elts = null;
    JETSAMPerformanceStatistic nextPS = null;
    boolean found = false;

    // Enumerate the stats vector to see if there is a performance statistic class
    // already for this perfName
    elts = stats.elements();
    while (elts.hasMoreElements() && !found)
    {
      nextPS = (JETSAMPerformanceStatistic) (elts.nextElement());
      found = nextPS.getName().equals(perfName);
    }

    // If there is
    if (found)
    {
      // If it is running then stop it
      if (nextPS.getRunning())
      {
        // Only pass curTime to the end function as we want
        // to bracket the function as closely as poosible
        nextPS.endStat(curTime);
      }
      // Otherwise start it
      else
      {
        nextPS.startStat();
      }
    }
    // Otherwise create a new performance statistic for this perfName and start it
    else
    {
      JETSAMPerformanceStatistic newPS =
        new JETSAMPerformanceStatistic(perfName);
      stats.addElement(newPS);
      newPS.startStat();
    }
  }

  /** This method will dump the statistics obtained by each of the  performance statistic 
    * classes to a JETSAMLog in a nice readable fashion.
    *
    * @param log. The log to write the output to.
    */
  public void dumpStats(JETSAMLogImpl log)
  {
    Enumeration elts = null;
    JETSAMPerformanceStatistic nextPS = null;
    StringBuffer output = null;
    int i = 0;

    log.blankLine();
    log.section("Performance Statistics:");
    log.blankLine();
    log.section(
      "Call          Freq    Total Time    "
        + "Average Time    Standard Dev.");
    log.section(
      "------------------------------------"
        + "-----------------------------");

    // Enumerate all of the performance statistic objects
    elts = stats.elements();
    while (elts.hasMoreElements())
    {
      output = new StringBuffer("");

      nextPS = (JETSAMPerformanceStatistic) (elts.nextElement());

      // Print out the name
      if (nextPS.getName().length() >= 14)
      {
        output.append(nextPS.getName().substring(0, 13));
      } else
      {
        output.append(nextPS.getName());
      }
      for (i = output.length(); i < 14; i++)
        output.append(" ");

      // Output the frequency
      output.append(String.valueOf(nextPS.getFrequency()));
      for (i = output.length(); i < 22; i++)
        output.append(" ");

      // Output the sum of all the times
      output.append(String.valueOf(nextPS.getSumSamples()));
      for (i = output.length(); i < 36; i++)
        output.append(" ");

      // Output the mean
      output.append(String.valueOf(nextPS.getMean()));
      for (i = output.length(); i < 52; i++)
        output.append(" ");

      // Ouput the standard deviation
      String stndDev = String.valueOf(nextPS.getStndDev());
      if (stndDev.length() > 6)
      {
        output.append(stndDev.substring(0, 6));
      } else
      {
        output.append(stndDev);
      }

      log.section(output.toString());
    }
    log.blankLine();
  }
}

/** This class holds all the details of the statistics and calculates all the
  * stats ready to be retrieved by the JETSAMPerformanceMonitor class.
  *
  * @author Ewan Withers
  *
  * @see com.ibm.mqst.jetsam.JETSAMPerformanceMonitor
  */
class JETSAMPerformanceStatistic
{
  /** The name of this statistic */
  private String statName = null;

  /** The start time */
  private long startTime = 0;

  /** Whether we are currently running or not */
  private boolean running = false;

  /** The sum of all the times */
  private long sumSamples = 0;

  /** The squares of all the times */
  private long sumSquares = 0;

  /** The number of times we have been run */
  private int frequency = 0;

  /** Constructor for this class which store the name of this performance statistic
   *
   * @param str. Name of this performance statistic
   */
  public JETSAMPerformanceStatistic(String str)
  {
    statName = str;
  }

  /** This method will start the statistics by storing the current time.
    */
  public void startStat()
  {
    running = true;
    startTime = System.currentTimeMillis();
  }

  /** This method will stop the statistics and calculate the times.
    *
    * @param curTime. This is the time that the end call was given.
    */
  public void endStat(long curTime)
  {
    long newSample = curTime - startTime;
    startTime = 0;
    running = false;
    frequency++;
    sumSamples += newSample;
    sumSquares += (newSample * newSample);
  }

  /** Accessor method for the performance name */
  public String getName()
  {
    return statName;
  }

  /** Accessor method to tell whether the stats are running */
  public boolean getRunning()
  {
    return running;
  }

  /** Accessor method for the frequency */
  public int getFrequency()
  {
    return frequency;
  }

  /** Accessor method for the sum of all the times */
  public long getSumSamples()
  {
    return sumSamples;
  }

  /** This method will return the mean of all runs.
    *
    * @return Returns the mean or 0 if the stats have not been run
    */
  public long getMean()
  {
    if (frequency != 0)
      return sumSamples / (long) frequency;
    else
      return (long) 0;
  }

  /** This method will return the standard deviation of all the runs
    *
    * @return Returns the standard deviation or 0 if the stats have not been run
    */
  public double getStndDev()
  {
    if (frequency != 0)
    {
      double mean = sumSamples / (long) frequency;
      double sigma = (sumSamples * sumSamples) / frequency;
      double variance = sumSquares - sigma;
      variance = variance / frequency;

      // Standard deviation is the square root of the variance
      return Math.sqrt(variance);
    } else
      return (double) 0.0;
  }

  
}
