/*******************************************************************************
 * Copyright (c) 2004, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.sib.comms.server.clientsupport;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import com.ibm.websphere.ras.TraceComponent;
import com.ibm.websphere.sib.exception.SIException;
import com.ibm.websphere.sib.exception.SIResourceException;
import com.ibm.ws.ffdc.FFDCFilter;
import com.ibm.ws.sib.comms.CommsConstants;
import com.ibm.ws.sib.comms.common.CommsByteBuffer;
import com.ibm.ws.sib.comms.common.CommsByteBufferPool;
import com.ibm.ws.sib.comms.server.ConversationState;
import com.ibm.ws.sib.jfapchannel.Conversation;
import com.ibm.ws.sib.jfapchannel.Conversation.ThrottlingPolicy;
import com.ibm.ws.sib.jfapchannel.JFapChannelConstants;
import com.ibm.ws.sib.utils.ras.SibTr;
import com.ibm.wsspi.sib.core.ConsumerSession;
import com.ibm.wsspi.sib.core.SICoreConnection;
import com.ibm.wsspi.sib.core.SICoreConnectionListener;
import com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException;
import com.ibm.wsspi.sib.core.exception.SIConnectionLostException;
import com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException;

/**
 * <p>
 * A connection listener is listening for events that are generated by the
 * core API but cannot easily be processed in the normal line of events.
 * <p>
 * One connection listener is designed to be used per physical link. I.e,
 * if two conversations share the same physical link then they will both use
 * the same instance of this class. This is so that in the event of the ME
 * quiescing, only one notification is sent to clients connected to the same
 * ME, on the same physcial link. In the event that a client is attached to
 * different ME's using the same physical link, then multiple notifications
 * will have to be sent.
 * 
 * @author Gareth Matthews
 */
public class ServerSICoreConnectionListener implements SICoreConnectionListener {
    /** Class name for FFDC's */
    private static String CLASS_NAME = ServerSICoreConnectionListener.class.getName();

    /** Our trace component */
    private static final TraceComponent tc = SibTr.register(ServerSICoreConnectionListener.class,
                                                            CommsConstants.MSG_GROUP,
                                                            CommsConstants.MSG_BUNDLE);

    /**
     * A hashtable containing a list of ME Uuids that quiesce
     * notifications has already been sent to.
     */
    private final Hashtable<String, Object> quiesceNotif = new Hashtable<String, Object>();

    /**
     * A hashtable containing a list of ME Uuids that quiesce
     * notifications has already been sent to.
     */
    private final Hashtable<String, Object> termNotif = new Hashtable<String, Object>();

    /** A hashtable matching an SICoreConnection to a Conversation */
    private final Hashtable<SICoreConnection, Conversation> conversationTable =
                    new Hashtable<SICoreConnection, Conversation>();

    /**
     * Static init to dump out the source info
     */
    static {
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            SibTr.debug(tc,
                        "Source info: @(#)SIB/ws/code/sib.comms.server.impl/src/com/ibm/ws/sib/comms/server/clientsupport/ServerSICoreConnectionListener.java, SIB.comms, WASX.SIB, aa1225.01 1.31");
    }

    /**
     * Creates a new SICoreConnection that this listener is listening
     * for events from.
     * 
     * @param conn The SICoreConnection.
     * @param conversation The associated conversation.
     */
    public void addSICoreConnection(SICoreConnection conn, Conversation conversation) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "addSICoreConnection");

        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            SibTr.debug(tc, "Params: connection, conversation",
                        new Object[] { conn, conversation });
        }

        conversationTable.put(conn, conversation);

        // At this point, if the me has previously been terminated, we know it is
        // back up - so make sure we remove if from our quiesce and terminated table
        // so that if it goes down again, we can notify the clients.
        quiesceNotif.remove(conn.getMeUuid());
        termNotif.remove(conn.getMeUuid());

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "addSICoreConnection");
    }

    /**
     * This method will remove the SICoreConnection from our table that maps
     * the connection to conversations. This is not essentially needed but the
     * smaller the table, the faster it is to search.
     * <p>
     * This would typically be called when the connection is closing.
     * 
     * @param conn
     */
    public void removeSICoreConnection(SICoreConnection conn) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "removeSICoreConnection");

        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            SibTr.debug(tc, "Params: connection", conn);
        }

        conversationTable.remove(conn);

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "removeSICoreConnection");
    }

    /**
     * This event is generated if an exception is thrown during the processing of
     * an asynchronous callback. In practise this should never occur as we should
     * ensure that we catch all the errors in the place they occur as no state
     * is available here.
     * 
     * @param session
     * @param e
     */
    @Override
    public void asynchronousException(ConsumerSession session, Throwable e) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "asynchronousException",
                        new Object[]
                        {
                         session,
                         e
                        });

        FFDCFilter.processException(e,
                                    CLASS_NAME + ".asynchronousException",
                                    CommsConstants.SERVERSICORECONNECTIONLISTENER_ASYNC_01,
                                    this);

        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            SibTr.debug(tc, "Caught an async exception:", e);

        try {
            sendMeNotificationEvent(CommsConstants.EVENTID_ASYNC_EXCEPTION, null, session, e);
        } catch (SIException e2) {
            FFDCFilter.processException(e,
                                        CLASS_NAME + ".asynchronousException",
                                        CommsConstants.SERVERSICORECONNECTIONLISTENER_ASYNC_02,
                                        this);

            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                SibTr.debug(tc, e.getMessage(), e);

            SibTr.error(tc, "COMMUNICATION_ERROR_SICO2018", e);
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "asynchronousException");
    }

    /**
     * This is used to notify us that the ME is quiescing. In this case we should
     * in turn inform the clients. This will be done in the most efficient way
     * possible by only informing client on the same physical socket once.
     * <p>
     * This connection listener instance is used across the link layer of a
     * conversation - as such, this listener is used for every SICoreConnection
     * that is on this socket. As such, we simply need to send the notification
     * that ME 'x' is quiescing down one conversation. These are fanned out at
     * the other end. Once we have successfully sent one notification, none will
     * be sent for this ME name.
     * <p>
     * If the ME is brought back up, the <code>meStarted</code> method will be
     * called, meaning that quiesce notifications will once again be sent.
     * 
     * @param conn
     */
    @Override
    public void meQuiescing(SICoreConnection conn) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "meQuiescing");

        // Right, so the ME that the core connection is connected to
        // is going down. First of all get the ME Uuid that the connection
        // refers to
        String meUuid = conn.getMeUuid();
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            SibTr.debug(tc, "ME Uuid: ", meUuid);

        // Have we already sent a notification about this ME quiescing
        // to the client?
        if (quiesceNotif.get(meUuid) == null) {
            // No we have not. So, get the Conversation and send the client
            // the message about this ME. It will be up to the client to fan
            // out this message to other clients connected to this ME.
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                SibTr.debug(tc, "We have not sent a notification about this ME");

            // Start f179464
            try {
                sendMeNotificationEvent(CommsConstants.EVENTID_ME_QUIESCING, conn, null, null);

                // and save that we have sent a response concerning this ME
                // Note, just using a Hashtable for fast searching.
                quiesceNotif.put(meUuid, new Object());
            } catch (SIException e) {
                FFDCFilter.processException(e,
                                            CLASS_NAME + ".meQuiescing",
                                            CommsConstants.SERVERSICORECONNECTIONLISTENER_MEQ_01,
                                            this);

                if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                    SibTr.debug(tc, e.getMessage(), e);

                SibTr.error(tc, "COMMUNICATION_ERROR_SICO2018", e);
            }
        } else {
            // Yes we have - so do not do anything more
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                SibTr.debug(tc, "Already sent notification about this ME");
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "meQuiescing");
    }

    /**
     * This method is called when the ME terminates.
     * 
     * @param conn
     */
    @Override
    public void meTerminated(SICoreConnection conn) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "meTerminated");

        //Remember the fact that the ME has terminated so we don't issue spurious FFDCs
        final Conversation conversation = conversationTable.get(conn);

        //If conversation is null we do nothing.
        if (conversation != null) {
            final ConversationState convState = (ConversationState) conversation.getAttachment();
            convState.setMETerminated();
        }

        // Right, so the ME that the core connection is connected to
        // is dead. First of all get the ME Uuid that the connection
        // refers to
        String meUuid = conn.getMeUuid();
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            SibTr.debug(tc, "ME Uuid: ", meUuid);

        // Have we already sent a notification about this ME terminating
        // to the client?
        if (termNotif.get(meUuid) == null) {
            // No we have not. So, get the Conversation and send the client
            // the message about this ME. It will be up to the client to fan
            // out this message to other clients connected to this ME.
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                SibTr.debug(tc, "We have not sent a notification about this ME");

            try {
                sendMeNotificationEvent(CommsConstants.EVENTID_ME_TERMINATED, conn, null, null);

                // and save that we have sent a response concerning this ME
                // Note, just using a Hashtable for fast searching.
                termNotif.put(meUuid, new Object());
            } catch (SIException e) {
                FFDCFilter.processException(e,
                                            CLASS_NAME + ".meTerminated",
                                            CommsConstants.SERVERSICORECONNECTIONLISTENER_MET_01,
                                            this);

                if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                    SibTr.debug(tc, e.getMessage(), e);

                SibTr.error(tc, "COMMUNICATION_ERROR_SICO2018", e);
            }
        } else {
            // Yes we have - so do not do anything more
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                SibTr.debug(tc, "Already sent notification about this ME");
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "meTerminated");
    }

    /**
     * This private method will actually send the notification to the client
     * using the specified event Id.
     * 
     * @param eventId The event ID
     * @param conn The core connection that is dieing
     * @param sess The Consumer Session that generated the exception
     * @param t The exception to be sent back
     * 
     * @throws SIConnectionLostException
     * @throws SIConnectionDroppedException
     */
    private void sendMeNotificationEvent(short eventId, SICoreConnection conn, ConsumerSession sess, Throwable t)
                    throws SIConnectionLostException,
                    SIConnectionDroppedException,
                    SIConnectionUnavailableException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "sendMeNotificationEvent",
                        new Object[]
                        {
                         "" + eventId,
                         conn,
                         sess,
                         t
                        });

        Conversation conversation = null;
        int consumerId = -1;

        // Get the conversation. If we have an SICoreConnection handed to us, we can dive into the 
        // hashtable and retrieve the correct one.
        if (conn != null) {
            conversation = conversationTable.get(conn);
        } else {
            // If no SICoreConnection was passed in (for example when sending an async exception we 
            // could retrieve that from the ConsumerSession. However, it is most likely that the reason 
            // we are sending the async exception is because the session has gone down and so the most 
            // reliable way is to iterate over all the Conversations we know about. We expect this
            // operation to be a fairly 'one off' - and so we don't need the performance
            for (Iterator<Conversation> i = conversationTable.values().iterator(); i.hasNext();) {
                conversation = i.next();
                ConversationState convState = (ConversationState) conversation.getAttachment();

                // For each conversation look for the right Consumer Session
                List allObjs = convState.getAllObjects();
                for (int x = 0; x < allObjs.size(); x++) {
                    if (allObjs.get(x) instanceof CATMainConsumer) {
                        CATMainConsumer mainConsumer = (CATMainConsumer) allObjs.get(x);
                        ConsumerSession ourSession = mainConsumer.getConsumerSession();

                        // If we have found it, save the client session Id
                        if (ourSession == sess) {
                            consumerId = mainConsumer.getClientSessionId();
                            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                                SibTr.debug(this, tc, "Located consumer. Session Id is", "" + consumerId);
                            break;
                        }
                    }
                }

                if (consumerId != -1)
                    break;
            }
        }

        // Check here to see if the consumer Id is 0. If this is the case then this consumer is 
        // currently synchronous and probably in the middle of an operation such as a receiveWithWait.
        // In this case, do not send the notification back to the peer (as a consumerId of 0 will
        // break the client and cause the connection to be terminated) but let the consumer sessions
        // connection listener send the notification back.
        if (consumerId == 0) {
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                SibTr.debug(this, tc, "Consumer Id was 0 - ignoring this notification");
        } else {
            if (conversation == null) {
                // Not a lot we can do here. So FFDC, but do not register
                // that we have notified this client about the termination
                // of this ME
                NullPointerException npe = new NullPointerException("No conversation was located");

                FFDCFilter.processException(npe,
                                            CLASS_NAME + ".sendMeNotificationEvent",
                                            CommsConstants.SERVERSICORECONNECTIONLISTENER_MEN_01,
                                            this);

                if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                    SibTr.debug(tc, npe.getMessage(), npe);
            } else if (sess != null && consumerId == -1) {
                // This exception is generated purely for FFDC
                SIResourceException e = new SIResourceException("Unable to locate consumer session");

                FFDCFilter.processException(e,
                                            CLASS_NAME + ".sendMeNotificationEvent",
                                            CommsConstants.SERVERSICORECONNECTIONLISTENER_MEN_01,
                                            this); // d192146

                if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                    SibTr.debug(tc, e.getMessage(), e);
            } else {
                // Otherwise inform the client
                ConversationState convState = (ConversationState) conversation.getAttachment();

                CommsByteBuffer buffer = CommsByteBufferPool.getInstance().allocate();
                buffer.putShort(convState.getConnectionObjectId()); // BIT16 ConnectionObjectId
                buffer.putShort(eventId); // BIT16 Event Id
                if (sess != null) {
                    buffer.putShort(consumerId);
                    buffer.putException(t, null, conversation);
                }

                //in case if event id is EVENTID_ME_QUIESCING or EVENTID_ME_TERMINATED then dont block the thread because
                //this may function may get executed in the context of SCR thread and should not get blocked..
                //but it is very rare foo this function (i.e send)d to get blocked.. however those rare cases can be 
                //avoided by setting the ThrottlingPolicy to DISCARD_TRANSMISSION. In the event of ME termination/Quiescing, this should
                // be harmless
                if (eventId == CommsConstants.EVENTID_ME_TERMINATED || eventId == CommsConstants.EVENTID_ME_QUIESCING) {
                    conversation.send(buffer,
                                      JFapChannelConstants.SEG_EVENT_OCCURRED,
                                      0,
                                      JFapChannelConstants.PRIORITY_HIGH,
                                      true,
                                      ThrottlingPolicy.DISCARD_TRANSMISSION,
                                      null);
                } else {
                    conversation.send(buffer,
                                      JFapChannelConstants.SEG_EVENT_OCCURRED,
                                      0,
                                      JFapChannelConstants.PRIORITY_HIGH,
                                      true,
                                      ThrottlingPolicy.BLOCK_THREAD,
                                      null);

                }
            }
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "sendMeNotificationEvent");
    }

    /**
     * This is used to indicate a communication failure on the client. Seeing
     * as we are the only people who would ever generate this event for the client
     * if this gets invoked on the server then someone has done something not bad,
     * not even wrong, but silly.
     * 
     * @param conn
     * @param e
     */
    @Override
    public void commsFailure(SICoreConnection conn, SIConnectionLostException e) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(tc, "commsFailure",
                        new Object[]
                        {
                         conn,
                         e
                        });

        FFDCFilter.processException(e,
                                    CLASS_NAME + ".commsFailure",
                                    CommsConstants.SERVERSICORECONNECTIONLISTENER_COMMS_01,
                                    this);

        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            SibTr.debug(tc, "Caught a comms exception:", e);

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(tc, "commsFailure");
    }
}
