/*******************************************************************************
 * Copyright (c) 2004, 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.sib.comms.client;

import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.ibm.websphere.ras.TraceComponent;
import com.ibm.ejs.ras.TraceNLS;
import com.ibm.websphere.sib.exception.SIErrorException;
import com.ibm.websphere.sib.exception.SIIncorrectCallException;
import com.ibm.websphere.sib.exception.SIResourceException;
import com.ibm.ws.sib.comms.CommsConstants;
import com.ibm.ws.sib.comms.common.CommsByteBuffer;
import com.ibm.ws.sib.comms.common.CommsLightTrace;
import com.ibm.ws.sib.jfapchannel.Conversation;
import com.ibm.ws.sib.jfapchannel.JFapChannelConstants;
import com.ibm.ws.sib.jfapchannel.Conversation.ThrottlingPolicy;
import com.ibm.ws.sib.utils.ras.SibTr;
import com.ibm.wsspi.sib.core.BifurcatedConsumerSession;
import com.ibm.wsspi.sib.core.SIBusMessage;
import com.ibm.wsspi.sib.core.SIMessageHandle;
import com.ibm.wsspi.sib.core.SITransaction;
import com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException;
import com.ibm.wsspi.sib.core.exception.SIConnectionLostException;
import com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException;
import com.ibm.wsspi.sib.core.exception.SILimitExceededException;
import com.ibm.wsspi.sib.core.exception.SIMessageNotLockedException;
import com.ibm.wsspi.sib.core.exception.SISessionDroppedException;
import com.ibm.wsspi.sib.core.exception.SISessionUnavailableException;

/**
 * This class represents a bifurcated consumer session. A bifurcated session is one that
 * represents an existing session and allows management and reading of locked messages only.
 * 
 * @author Gareth Matthews
 */
public class BifurcatedConsumerSessionProxy extends DestinationSessionProxy implements BifurcatedConsumerSession
{
   /** Class name for FFDC's */
// No FFDCs generated by this class - so commented out to avoid compiler warning.
//   private static String CLASS_NAME = BifurcatedConsumerSessionProxy.class.getName();
   
   /** Register Class with Trace Component */
   private static final TraceComponent tc = SibTr.register(BifurcatedConsumerSessionProxy.class, 
                                                           CommsConstants.MSG_GROUP, 
                                                           CommsConstants.MSG_BUNDLE);
   
   /** Our NLS reference object */
   private static final TraceNLS nls = TraceNLS.getTraceNLS(CommsConstants.MSG_BUNDLE);
   
   /** A lock used when closing the connection */
   private ReentrantReadWriteLock closeLock;
   
   /**
    * Constructor
    * 
    * @param con
    * @param cp
    * @param buf
    */
   public BifurcatedConsumerSessionProxy(Conversation con, ConnectionProxy cp, CommsByteBuffer buf)
   {
      super(con, cp);
         
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "<init>");
      
      inflateData(buf);
      closeLock = cp.closeLock;
         
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "<init>");
   }

   /**
    * This method is used to delete a set of locked messages held by the message processor.
    * This call will simply be passed onto the server who will call the method on the real
    * bifurcated consumer session residing on the server.
    * 
    * @param msgHandles An array of message ids that denote the messages to be deleted.
    * @param tran The transaction to which the delete should be performed.
    * 
    * @throws com.ibm.wsspi.sib.core.exception.SISessionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SISessionDroppedException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.wsspi.sib.core.exception.SILimitExceededException
    * @throws com.ibm.websphere.sib.exception.SIIncorrectCallException
    * @throws com.ibm.wsspi.sib.core.exception.SIMessageNotLockedException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public void deleteSet(SIMessageHandle[] msgHandles, SITransaction tran)
      throws SISessionUnavailableException, SISessionDroppedException,
             SIConnectionUnavailableException, SIConnectionDroppedException,
             SIResourceException, SIConnectionLostException, SILimitExceededException,
             SIIncorrectCallException, SIMessageNotLockedException, 
             SIErrorException   
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "deleteSet",
                                           new Object[] 
                                           {
                                              msgHandles.length + " msg handles", 
                                              tran
                                           });
      try
      {
         closeLock.readLock().lockInterruptibly();
         
         try
         {
            checkAlreadyClosed();
            
            // Now we need to synchronise on the transaction object if there is one.
            if (tran != null)
            {
               synchronized (tran)
               {
                  // Check transaction is in a valid state.
                  // Enlisted for an XA UOW and not rolledback or
                  // completed for a local transaction.
                  if (!((Transaction) tran).isValid())
                  {
                     throw new SIIncorrectCallException(
                        nls.getFormattedMessage("TRANSACTION_COMPLETE_SICO1066", null, null)
                     );
                  }
                  
                  _deleteSet(msgHandles, tran);
               }
            }
            else
            {
               _deleteSet(msgHandles, null);
            }
         }
         finally
         {
            closeLock.readLock().unlock();
         }
      }
      catch (InterruptedException e)
      {
         // No FFDC Code needed
      }
      
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "deleteSet");
   }
   
   /**
    * Actually performs the delete.
    * 
    * @param msgHandles
    * @param tran
    * 
    * @throws SISessionUnavailableException
    * @throws SISessionDroppedException
    * @throws SIConnectionUnavailableException
    * @throws SIConnectionDroppedException
    * @throws SIResourceException
    * @throws SIConnectionLostException
    * @throws SILimitExceededException
    * @throws SIIncorrectCallException
    * @throws SIMessageNotLockedException
    * @throws SIErrorException
    */
   private void _deleteSet(SIMessageHandle[] msgHandles, SITransaction tran)
      throws SISessionUnavailableException, SISessionDroppedException,
             SIConnectionUnavailableException, SIConnectionDroppedException,
             SIResourceException, SIConnectionLostException, SILimitExceededException,
             SIIncorrectCallException, SIMessageNotLockedException, 
             SIErrorException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "_deleteSet",
                                           new Object[] 
                                           {
                                              msgHandles.length + " msg handles", 
                                              tran
                                           });
      
      // 531458 Allow minimal tracing to include message ids and tran id
      if (TraceComponent.isAnyTracingEnabled()) {
        CommsLightTrace.traceMessageIds(tc, "DeleteSetMsgTrace", msgHandles);
      }
      
      CommsByteBuffer request = getCommsByteBuffer();
      
      request.putShort(getConnectionObjectID());
      request.putShort(getProxyID());
      request.putSITransaction(tran);           
      request.putSIMessageHandles(msgHandles);
      
      CommsByteBuffer reply = null;
      if (tran == null)
      {
        reply = jfapExchange(request, 
                               JFapChannelConstants.SEG_DELETE_SET,
                               JFapChannelConstants.PRIORITY_MEDIUM,
                               true);
         try
         {
            short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_DELETE_SET_R);
            if (err != CommsConstants.SI_NO_EXCEPTION)
            {
               checkFor_SISessionUnavailableException(reply, err);
               checkFor_SISessionDroppedException(reply, err);
               checkFor_SIConnectionUnavailableException(reply, err);
               checkFor_SIConnectionDroppedException(reply, err);
               checkFor_SIResourceException(reply, err);
               checkFor_SIConnectionLostException(reply, err);
               checkFor_SILimitExceededException(reply, err);
               checkFor_SIIncorrectCallException(reply, err);
               checkFor_SIMessageNotLockedException(reply, err);
               checkFor_SIErrorException(reply, err);
               defaultChecker(reply, err);
            }
         }
         finally
         {
            reply.release();
         }
      }
      else
      {
         jfapSend(request, 
                  JFapChannelConstants.SEG_DELETE_SET_NOREPLY, 
                  ((Transaction) tran).getLowestMessagePriority(), 
                  true,
                  ThrottlingPolicy.BLOCK_THREAD);
      }
      
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "_deleteSet");
   }

   /**
    * This method is used to read a set of locked messages held by the message processor.
    * This call will simply be passed onto the server who will call the method on the real
    * bifurcated consumer session residing on the server.
    * 
    * @param msgHandles An array of message ids that denote the messages to be read.
    * 
    * @return Returns an array of SIBusMessages
    * 
    * @throws com.ibm.wsspi.sib.core.exception.SISessionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SISessionDroppedException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.websphere.sib.exception.SIIncorrectCallException
    * @throws com.ibm.wsspi.sib.core.exception.SIMessageNotLockedException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public SIBusMessage[] readSet(SIMessageHandle[] msgHandles) 
      throws SISessionUnavailableException, SISessionDroppedException,
             SIConnectionUnavailableException, SIConnectionDroppedException,
             SIResourceException, SIConnectionLostException, 
             SIIncorrectCallException, SIMessageNotLockedException, 
             SIErrorException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "readSet",
                                           new Object[] 
                                           {
                                              msgHandles.length + " msg ids"
                                           });

      SIBusMessage[] messages = null;
      
      try
      {
         closeLock.readLock().lockInterruptibly();

         try
         {         
            checkAlreadyClosed();
            
            CommsByteBuffer request = getCommsByteBuffer();
            
            request.putShort(getConnectionObjectID());
            request.putShort(getProxyID());
            request.putSIMessageHandles(msgHandles);
            
            CommsByteBuffer reply = jfapExchange(request, 
                                                 JFapChannelConstants.SEG_READ_SET,
                                                 JFapChannelConstants.PRIORITY_MEDIUM,
                                                 true);
         
            try
            {
               short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_READ_SET_R);
               if (err != CommsConstants.SI_NO_EXCEPTION)
               {
                  checkFor_SISessionUnavailableException(reply, err);
                  checkFor_SISessionDroppedException(reply, err);
                  checkFor_SIConnectionUnavailableException(reply, err);
                  checkFor_SIConnectionDroppedException(reply, err);
                  checkFor_SIResourceException(reply, err);
                  checkFor_SIConnectionLostException(reply, err);
                  checkFor_SIIncorrectCallException(reply, err);
                  checkFor_SIMessageNotLockedException(reply, err);
                  checkFor_SIErrorException(reply, err);
                  defaultChecker(reply, err);
               }
               
               int numberOfMessages = reply.getInt();
               messages = new SIBusMessage[numberOfMessages];
               
               for (int x = 0; x < numberOfMessages; x++)
               {
                  messages[x] = reply.getMessage(getCommsConnection());               
               }
            }
            finally
            {
               reply.release(false);
            }
         }
         finally
         {
            closeLock.readLock().unlock();
         }
      }
      catch (InterruptedException e)
      {
         // No FFDC Code needed
      }
      
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "readSet");
      return messages;
   }

   /**
    * This method is used to read and then delete a set of locked messages held by the message 
    * processor. This call will simply be passed onto the server who will call the method on the 
    * real bifurcated consumer session residing on the server.
    * 
    * @param msgHandles An array of message ids that denote the messages to be read and then deleted.
    * @param tran The transaction to which the delete should be performed.
    * 
    * @return Returns an array of SIBusMessages
    * 
    * @throws com.ibm.wsspi.sib.core.exception.SISessionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SISessionDroppedException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.wsspi.sib.core.exception.SILimitExceededException
    * @throws com.ibm.websphere.sib.exception.SIIncorrectCallException
    * @throws com.ibm.wsspi.sib.core.exception.SIMessageNotLockedException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public SIBusMessage[] readAndDeleteSet(SIMessageHandle[] msgHandles, SITransaction tran)
      throws SISessionUnavailableException, SISessionDroppedException,
             SIConnectionUnavailableException, SIConnectionDroppedException,
             SIResourceException, SIConnectionLostException, SILimitExceededException,
             SIIncorrectCallException, SIMessageNotLockedException, 
             SIErrorException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "readAndDeleteSet",
                                           new Object[] 
                                           {
                                              msgHandles.length + " msg ids",
                                              tran
                                           });

      SIBusMessage[] messages = null;
      
      try
      {
         closeLock.readLock().lockInterruptibly();
         
         try
         {
            checkAlreadyClosed();
            
            // Now we need to synchronise on the transaction object if there is one.
            if (tran != null)
            {
               synchronized (tran)
               {
                  // Check transaction is in a valid state.
                  // Enlisted for an XA UOW and not rolledback or
                  // completed for a local transaction.
                  if (!((Transaction) tran).isValid())
                  {
                     throw new SIIncorrectCallException(
                        nls.getFormattedMessage("TRANSACTION_COMPLETE_SICO1066", null, null)
                     );
                  }
                  
                  messages = _readAndDeleteSet(msgHandles, tran);
               }
            }
            else
            {
               messages = _readAndDeleteSet(msgHandles, null);
            }
         }
         finally
         {
            closeLock.readLock().unlock();
         }
      }
      catch (InterruptedException e)
      {
         // No FFDC Code needed
      }
      
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "readAndDeleteSet");
      return messages;
   }
   
   /**
    * Actually performs the read and delete.
    * 
    * @param msgHandles
    * @param tran
    * 
    * @return Returns the requested messages.
    * 
    * @throws SISessionUnavailableException
    * @throws SISessionDroppedException
    * @throws SIConnectionUnavailableException
    * @throws SIConnectionDroppedException
    * @throws SIResourceException
    * @throws SIConnectionLostException
    * @throws SILimitExceededException
    * @throws SIIncorrectCallException
    * @throws SIMessageNotLockedException
    * @throws SIErrorException
    */
   private SIBusMessage[] _readAndDeleteSet(SIMessageHandle[] msgHandles, SITransaction tran)
      throws SISessionUnavailableException, SISessionDroppedException,
             SIConnectionUnavailableException, SIConnectionDroppedException,
             SIResourceException, SIConnectionLostException, SILimitExceededException,
             SIIncorrectCallException, SIMessageNotLockedException, 
             SIErrorException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "_readAndDeleteSet",
                                       new Object[] 
                                       {
                                          msgHandles.length + " msg ids",
                                          tran
                                       });
   
      // 531458 Allow minimal tracing to include message ids and tran id
      if (TraceComponent.isAnyTracingEnabled()) {
        CommsLightTrace.traceMessageIds(tc, "ReadAndDeleteSetMsgTrace", msgHandles);
      }
      
      SIBusMessage[] messages = null;
      
      CommsByteBuffer request = getCommsByteBuffer();
      
      request.putShort(getConnectionObjectID());
      request.putShort(getProxyID());
      request.putSITransaction(tran);            
      request.putSIMessageHandles(msgHandles);
      
      CommsByteBuffer reply = jfapExchange(request, 
                                           JFapChannelConstants.SEG_READ_AND_DELETE_SET,
                                           JFapChannelConstants.PRIORITY_MEDIUM,
                                           true);
      
      try
      {
         short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_READ_AND_DELETE_SET_R);
         if (err != CommsConstants.SI_NO_EXCEPTION)
         {
            checkFor_SISessionUnavailableException(reply, err);
            checkFor_SISessionDroppedException(reply, err);
            checkFor_SIConnectionUnavailableException(reply, err);
            checkFor_SIConnectionDroppedException(reply, err);
            checkFor_SIResourceException(reply, err);
            checkFor_SIConnectionLostException(reply, err);
            checkFor_SILimitExceededException(reply, err);
            checkFor_SIIncorrectCallException(reply, err);
            checkFor_SIMessageNotLockedException(reply, err);
            checkFor_SIErrorException(reply, err);
            defaultChecker(reply, err);
         }
         
         int numberOfMessages = reply.getInt();
         messages = new SIBusMessage[numberOfMessages];
         
         for (int x = 0; x < numberOfMessages; x++)
         {
            messages[x] = reply.getMessage(getCommsConnection());
         }
      }
      finally
      {
         reply.release(false);
      }
      
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "_readAndDeleteSet", messages);
      return messages;
   }

   /**
    * This method is used to unlock a set of locked messages held by the message processor.
    * This call will simply be passed onto the server who will call the method on the real
    * bifurcated consumer session residing on the server.
    * 
    * @param msgHandles An array of message ids that denote the messages to be unlocked.
    * 
    * @throws com.ibm.wsspi.sib.core.exception.SISessionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SISessionDroppedException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.websphere.sib.exception.SIIncorrectCallException
    * @throws com.ibm.wsspi.sib.core.exception.SIMessageNotLockedException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public void unlockSet(SIMessageHandle[] msgHandles) 
      throws SISessionUnavailableException, SISessionDroppedException,
             SIConnectionUnavailableException, SIConnectionDroppedException,
             SIResourceException, SIConnectionLostException, 
             SIIncorrectCallException, SIMessageNotLockedException, 
             SIErrorException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "unlockSet",
                                           new Object[] 
                                           {
                                              msgHandles.length + " msg ids"
                                           });
      
      unlockSet(msgHandles, true); //True is the default as it matches with old behaviour.

      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "unlockSet");
   }

   /**
    * This method is used to unlock a set of locked messages held by the message processor.
    * This call will simply be passed onto the server who will call the method on the real
    * bifurcated consumer session residing on the server.
    * 
    * @param msgids An array of message ids that denote the messages to be unlocked.
    * @param incrementLockCount Indicates if the lock count (aka redelivery count) should be incremented
    * 
    * @throws com.ibm.wsspi.sib.core.exception.SISessionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SISessionDroppedException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionUnavailableException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.websphere.sib.exception.SIIncorrectCallException
    * @throws com.ibm.wsspi.sib.core.SIMessageNotLockedException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public void unlockSet(SIMessageHandle[] msgHandles, boolean incrementLockCount) 
   throws SISessionUnavailableException, SISessionDroppedException,
          SIConnectionUnavailableException, SIConnectionDroppedException,
          SIResourceException, SIConnectionLostException, 
          SIIncorrectCallException, SIMessageNotLockedException, 
          SIErrorException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "unlockSet",
                                        new Object[] 
                                        {
                                           msgHandles.length + " msg ids",
                                           Boolean.valueOf(incrementLockCount)
                                        });
            
      try
      {
          closeLock.readLock().lockInterruptibly();
          
          try
          {
             checkAlreadyClosed();
             
             CommsByteBuffer request = getCommsByteBuffer();
             
             request.putShort(getConnectionObjectID());
             request.putShort(getProxyID());
             request.putSIMessageHandles(msgHandles);
             
             //Flowing incrementLockCount is only valid for faps 7, 8 and greater 9
             final int fapLevel = getConversation().getHandshakeProperties().getFapLevel();
             if(!(fapLevel < JFapChannelConstants.FAP_VERSION_7 || fapLevel == JFapChannelConstants.FAP_VERSION_9))
             {
                request.put((byte)((incrementLockCount) ? 1: 0));
             }
             
             CommsByteBuffer reply = jfapExchange(request, 
                                                  JFapChannelConstants.SEG_UNLOCK_SET,
                                                  JFapChannelConstants.PRIORITY_MEDIUM,
                                                  true);
             
             try
             {
                
                short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_UNLOCK_SET_R);
                if (err != CommsConstants.SI_NO_EXCEPTION)
                {
                   checkFor_SISessionUnavailableException(reply, err);
                   checkFor_SISessionDroppedException(reply, err);
                   checkFor_SIConnectionUnavailableException(reply, err);
                   checkFor_SIConnectionDroppedException(reply, err);
                   checkFor_SIResourceException(reply, err);
                   checkFor_SIConnectionLostException(reply, err);
                   checkFor_SIIncorrectCallException(reply, err);
                   checkFor_SIMessageNotLockedException(reply, err);
                   checkFor_SIErrorException(reply, err);
                   defaultChecker(reply, err);
                }
             }
             finally
             {
                reply.release();
             }
          }
          finally
          {
             closeLock.readLock().unlock();
          }
      }
      catch (InterruptedException e)
      {
          // No FFDC Code needed
      }
      
      if (tc.isEntryEnabled()) SibTr.exit(this, tc, "unlockSet");
   }

   /**
    * This method will close the bifurcated consumer session
    * 
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public void close() 
      throws SIResourceException, SIConnectionLostException, SIErrorException, 
             SIConnectionDroppedException
   {
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.entry(this, tc, "close");

      if (!isClosed())
      {
         try
         {
            closeLock.writeLock().lockInterruptibly();

            try
            {               
               CommsByteBuffer request = getCommsByteBuffer();
               
               request.putShort(getConnectionObjectID());
               request.putShort(getProxyID());
               
               CommsByteBuffer reply = jfapExchange(request, 
                                                    JFapChannelConstants.SEG_CLOSE_CONSUMER_SESS,
                                                    JFapChannelConstants.PRIORITY_MEDIUM,
                                                    true);
               
               try
               {
                  short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_CLOSE_CONSUMER_SESS_R);
                  if (err != CommsConstants.SI_NO_EXCEPTION)
                  {
                     checkFor_SIResourceException(reply, err);
                     checkFor_SIConnectionLostException(reply, err);
                     checkFor_SIConnectionDroppedException(reply, err);
                     checkFor_SIErrorException(reply, err);
                     defaultChecker(reply, err);
                  }
               }
               finally
               {
                  reply.release();
               }
               
               setClosed();
            }
            finally
            {
               closeLock.writeLock().unlock();
            }
         }
         catch (InterruptedException e)
         {
            // No FFDC Code needed
         }
      }
      
      if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled()) SibTr.exit(this, tc, "close");
   }
}
