/*******************************************************************************
 * Copyright (c) 2009, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.wsspi.kernel.service.location;

import java.io.File;
import java.net.URI;
import java.util.Iterator;
import java.util.UUID;


/**
 * Service for resolving symbolic locations to concrete/physical resources (e.g.
 * the bootstrap configuration determines where resources reside for an active
 * server instance).
 * <p>
 * Resources are associated with a group, where the group functions as a pseudo-filesystem:
 * a collection of locations that comprise a search path to use when locating resources by
 * name.
 */
public interface WsLocationAdmin {
    /**
     * Print debug information about the configured locations for the current
     * server.
     * 
     * @param formatOutput
     *            If true, the output will be formatted and separated with line
     *            breaks, otherwise, output will be a comma-separated list.
     * 
     * @return String containing a list of configured data locations.
     */
    String printLocations(boolean useLineBreaks);

    /**
     * Returns a persistent unique identifier for the server.
     * The identifier is persisted across normal server restarts but is reset after a clean start.
     * 
     * @return a unique identifier for the configured server
     */
    UUID getServerId();

    /**
     * Get a string representation of the configured server name.
     * 
     * @return server name as a String
     */
    String getServerName();

    /**
     * Return a File with the given name (which may or may not exist, and may or
     * may not be a directory) in the bundle's private working area.
     * <p>
     * This is a convenience method. It will use OSGi {@link org.osgi.framework.FrameworkUtil} to
     * find the {@link Bundle} used to load the caller's class, and then will
     * call {@link org.osgi.framework.BundleContext#getDataFile(String)} to create the
     * File.
     * <p>
     * Use forward slashes in the relative path.
     * 
     * @param caller
     *            Object (this, or static's class) requesting a file in the
     *            bundle's private data area; can not be null
     * @param relativeBundlePath
     *            Relative path for inside of the bundle's private data area
     * @return File inside the bundle's private data area
     * 
     * @throws MalformedLocationException
     *             if relativeBundlePath is not relative (is an absolute path, a
     *             file URI, or starts with a symbol)
     * @throws NullPointerException
     *             if caller is null
     */
    File getBundleFile(Object caller, String relativeBundlePath);

    /**
     * Return an WsResource object with the given name in the runtime install directory
     * (wlp.install.dir).
     * <p>
     * Note: This method MUST not be used to resolve server resources since there is no relationship
     * between the location of the runtime install and the location of server resources.
     * <p>
     * Use forward slashes in the relative path.
     * <p>
     * 
     * @param relativeServerPath
     *            Relative path for inside of the runtime install's directory, can not be
     *            null.
     * @return WsResource inside the runtime install directory.
     * 
     * @throws MalformedLocationException
     *             if relativeRuntimePath is not relative (is an absolute path, a
     *             file URI, or starts with a symbol), or if the path attempts
     *             to navigate to or above the parent (../).
     * @throws IllegalArgumentException
     *             if resource to be created already exists as a different type:
     *             i.e. if the resource uri indicates the resource should be a
     *             directory and the resource exists as a file, or vice versa.
     */
    WsResource getRuntimeResource(String relativeRuntimePath);

    /**
     * Return an WsResource object with the given name in the active server's configuration directory
     * (server.config.dir).
     * <p>
     * Note: Anything generated by the server should be created using {@link #getServerOutputResource}.
     * "Internal" data should be created using {@link #getServerWorkareaResource(String)}.
     * <p>
     * Use forward slashes in the relative path.
     * <p>
     * A File object for the base directory of the public server area can be obtained by calling
     * this method with an empty string as filename.
     * 
     * @param relativeServerPath
     *            Relative path for inside of the server's directory, can not be
     *            null.
     * @return WsResource inside the server's configuration directory.
     * 
     * @throws MalformedLocationException
     *             if relativeServerPath is not relative (is an absolute path, a
     *             file URI, or starts with a symbol), or if the path attempts
     *             to navigate to or above the parent (../).
     * @throws IllegalArgumentException
     *             if resource to be created already exists as a different type:
     *             i.e. if the resource uri indicates the resource should be a
     *             directory and the resource exists as a file, or vice versa.
     */
    WsResource getServerResource(String relativeServerPath);

    /**
     * Return an WsResource object with the given name in the active server's configuration directory
     * (server.output.dir).
     * <p>
     * Note: Server "working" data should be created/managed in either the bundle's data area, or the
     * server's workarea. The server's root directory should contain only user-generated/controllable
     * artifacts: Anything in the server dir should be treated as API.
     * <p>
     * Use forward slashes in the relative path.
     * <p>
     * A File object for the base directory of the public server area can be obtained by calling
     * this method with an empty string as filename.
     * 
     * @param relativeServerPath
     *            Relative path for inside of the server's directory, can not be
     *            null.
     * @return WsResource inside the server's configuration directory.
     * 
     * @throws MalformedLocationException
     *             if relativeServerPath is not relative (is an absolute path, a
     *             file URI, or starts with a symbol), or if the path attempts
     *             to navigate to or above the parent (../).
     * @throws IllegalArgumentException
     *             if resource to be created already exists as a different type:
     *             i.e. if the resource uri indicates the resource should be a
     *             directory and the resource exists as a file, or vice versa.
     */
    WsResource getServerOutputResource(String relativeServerPath);

    /**
     * Return an WsResource object with the given name (which may or may not
     * exist) in the active server's workarea.
     * <p>
     * Data in the server workarea will survive a refresh of the OSGi framework cache
     * <p>
     * Best Practice: use a subdirectory (e.g. using the bundle's PID, or the config group name) to consolidate
     * data associated w/ a particular bundle or subsystem, and reduce the
     * likelihood of name collisions.
     * <p>
     * Use forward slashes in the relative path.
     * <p>
     * A File object for the base directory of the public server area can be obtained by
     * calling this method with an empty string as filename.
     * 
     * @param relativeServerWorkareaPath
     *            Relative path within the configured server's private workarea;
     *            can not be null
     * @return WsResource inside the server's workarea
     * 
     * @throws MalformedLocationException
     *             if relativeServerWorkareaPath is not relative (is an absolute
     *             path, a file URI, or starts with a symbol), or if the path
     *             attempts to navigate to or above the parent (../).
     * @throws IllegalArgumentException
     *             if resource to be created already exists as a different type:
     *             i.e. if the resource uri indicates the resource should be a
     *             directory and the resource exists as a file, or vice versa.
     */
    WsResource getServerWorkareaResource(String relativeServerWorkareaPath);

    /**
     * Resolve the given resource URI (either an absolute URI, or a repository URI
     * containing symbolics), and return an <code>WsResource</code>.
     * <ul>
     * <li>file://blah
     * <li>http://some.other.place/...
     * <li>${shared.app.dir}</li>
     * <li>${shared.resource.dir}/security</li>
     * </ul>
     * <p>
     * Use forward slashes in the resource URI.
     * 
     * @param resourceURI
     *            Resource URI as a string
     * @return <code>WsResource</code> representing the resource specified by the provided URI; may return null
     *         if the <code>resourceURI </code> contains an unknown/unresolvable symbol.
     * @throws MalformedLocationException
     *             if the resourceURI is malformed
     * @throws IllegalArgumentException
     *             if the URI points to an existing resource of the wrong type (i.e. you specified a
     *             file, "a/b/c", but the resource already exists as a directory, "a/b/c/")
     */
    WsResource resolveResource(String resourceURI);

    /**
     * Resolve the given resource URI (either an absolute URI, or an liberty
     * repository URI containing symbolics), and return an <code>WsResource</code>.
     * <ul>
     * <li>file://blah
     * <li>http://some.other.place/...
     * <li>${wlp.install.dir}/applications</li>
     * </ul>
     * <p>
     * Use forward slashes in the resource URI.
     * 
     * @param resourceURI
     *            Resource URI
     * @return <code>WsResource</code> representing the resource specified by the provided URI;
     *         may return null if the <code>resourceURI </code> contains an unknown/unresolvable
     *         symbol.
     * @throws MalformedLocationException
     *             if the resourceURI is malformed
     * @throws IllegalArgumentException
     *             if the URI points to an existing resource of the wrong type (i.e. you specified a
     *             file, "a/b/c", but the resource already exists as a directory, "a/b/c/")
     */
    WsResource resolveResource(URI resourceURI);

    /**
     * Create a <code>WsResource</code> for the provided file.
     * 
     * @param file
     *            The File object that should be converted to a WsResource
     * @param isFile
     *            If true, the file object should be treated as a file;
     *            if false, the file object should be treated as a directory.
     * @return <code>WsResource</code> representing the provided file, or null if file argument is null.
     * @throws IllegalArgumentException
     *             if the File points to an existing resource of the wrong type
     *             (i.e. isFile is true, but the file object references a directory).
     */
    WsResource asResource(File file, boolean isFile);

    /**
     * Resolve all symbols in the given string URI.
     * 
     * <table role="presentation">
     * <tr>
     * <td>file://blah</td>
     * <td>file://blah</td>
     * </tr>
     * <tr>
     * <td>${shared.app.dir}</td>
     * <td>/path/to/usr/shared/apps</td>
     * </tr>
     * <tr>
     * <td>${shared.resource.dir}/security</td>
     * <td>/path/to/usr/shared/resources/security</td>
     * </tr>
     * <tr>
     * <td>${relativePathSymbol}/extra</td>
     * <td>relative/extra</td>
     * </tr>
     * </table>
     * 
     * @param resourceURI
     *            Resource URI as a string
     * @return URI string for the resolved resource; may return null if the <code>resourceURI</code> contains an unknown/unresolvable symbol.
     * @throws MalformedLocationException
     *             if the resourceURI is malformed
     * @throws IllegalArgumentException
     *             if the URI points to an existing resource of the wrong type
     *             (i.e. you specified a file, "a/b/c", but the resource already
     *             exists as a directory, "a/b/c/")
     */
    String resolveString(String string);

    /**
     * Find the resource(s) matching the provided regular expression. Each
     * element in the search path associated with a resource group will be
     * queried for matches until the limit is reached.
     * <p>
     * If a particular path is expected (e.g. a/b/c), include the path elements in the regular expression. Use forward slashes.
     * <p>
     * The iterator will present resources in the order they were matched. If no resources were found, the iterator will have no elements.
     * 
     * @param resourceGroupName
     *            Group of resources to resolve against: bundle, config, etc.
     * @param resourceRegex
     *            Regular expression describing matching resource URIs to find
     *            within the search path defined for the resourceGroup.
     * @param limit
     *            Maximum number of resources to match (0 implies no limit, 1
     *            will only return first match, etc.)
     * 
     * @return Iterator for the set/list of resources matching the provided
     *         regular expression.
     * 
     * @throws IllegalArgumentException
     *             if resource to be created already exists as a different type:
     *             i.e. if the resource uri indicates the resource should be a
     *             directory and the resource exists as a file, or vice versa. This
     *             throw may not occur until the element is reached while
     *             iterating.
     */
    Iterator<WsResource> matchResource(String resourceGroupName, String resourceRegex, int limit);

    /**
     * Add location to the registry of locations.
     * 
     * @param fileName
     *            The name of the file that is the root of the location
     * @param symbolicName
     *            The symbolic name of this resource without the symbol decorations
     *            (e.g. productextension.extension.dir)
     * 
     * @throws NullPointerException
     *             if either the fileName or the symbolicName are null
     * @throws IllegalArgumentException
     *             if the named location exits and is a file
     */
    WsResource addLocation(String fileName, String symbolicName);

}
