/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.web;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OP_ADDR;
import static org.jboss.as.web.Constants.EXECUTOR;
import static org.jboss.as.web.Constants.MAX_CONNECTIONS;
import static org.jboss.as.web.Constants.MAX_POST_SIZE;
import static org.jboss.as.web.Constants.MAX_SAVE_POST_SIZE;
import static org.jboss.as.web.Constants.PROXY_NAME;
import static org.jboss.as.web.Constants.PROXY_PORT;
import static org.jboss.as.web.Constants.REDIRECT_PORT;
import static org.jboss.as.web.Constants.VIRTUAL_SERVER;
import static org.jboss.as.web.WebConnectorDefinition.CONNECTOR_ATTRIBUTES;
import static org.jboss.as.web.WebExtension.SSL_PATH;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executor;

import org.apache.catalina.connector.Connector;
import org.jboss.as.controller.AbstractAddStepHandler;
import org.jboss.as.controller.AttributeDefinition;
import org.jboss.as.controller.OperationContext;
import org.jboss.as.controller.OperationFailedException;
import org.jboss.as.controller.PathAddress;
import org.jboss.as.controller.ServiceVerificationHandler;
import org.jboss.as.controller.SimpleAttributeDefinition;
import org.jboss.as.controller.registry.Resource;
import org.jboss.as.network.OutboundSocketBinding;
import org.jboss.as.network.SocketBinding;
import org.jboss.as.threads.ThreadsServices;
import org.jboss.dmr.ModelNode;
import org.jboss.msc.service.ServiceBuilder;
import org.jboss.msc.service.ServiceController;
import org.jboss.msc.service.ServiceController.Mode;

/**
 * {@code OperationHandler} responsible for adding a web connector.
 *
 * @author Emanuel Muckenhuber
 */
class WebConnectorAdd extends AbstractAddStepHandler {


    static final WebConnectorAdd INSTANCE = new WebConnectorAdd();

    private WebConnectorAdd() {
        //
    }

    @Override
    protected void populateModel(ModelNode operation, ModelNode model) throws OperationFailedException {
        PathAddress address = PathAddress.pathAddress(operation.require(OP_ADDR));
        model.get(WebConnectorDefinition.NAME.getName()).set(address.getLastElement().getValue());

        for (SimpleAttributeDefinition def : CONNECTOR_ATTRIBUTES) {
            def.validateAndSet(operation, model);
        }
        WebConnectorDefinition.VIRTUAL_SERVER.validateAndSet(operation,model);
    }

    @Override
    protected void performRuntime(OperationContext context, ModelNode operation, ModelNode model, ServiceVerificationHandler verificationHandler, List<ServiceController<?>> newControllers) throws OperationFailedException {
        final PathAddress address = PathAddress.pathAddress(operation.get(OP_ADDR));
        ModelNode fullModel = Resource.Tools.readModel(context.readResource(PathAddress.EMPTY_ADDRESS));
        launchServices(context,address,fullModel,verificationHandler,newControllers);
    }

    protected void launchServices(OperationContext context, PathAddress address, ModelNode fullModel, ServiceVerificationHandler verificationHandler, List<ServiceController<?>> newControllers) throws OperationFailedException {
        final String name = address.getLastElement().getValue();
        final String bindingRef = WebConnectorDefinition.SOCKET_BINDING.resolveModelAttribute(context, fullModel).asString();
        final boolean enabled = WebConnectorDefinition.ENABLED.resolveModelAttribute(context, fullModel).asBoolean();
        final String protocol = WebConnectorDefinition.PROTOCOL.resolveModelAttribute(context, fullModel).asString();
        final String scheme = WebConnectorDefinition.SCHEME.resolveModelAttribute(context, fullModel).asString();
        final WebConnectorService service = new WebConnectorService(protocol, scheme);
        service.setSecure(WebConnectorDefinition.SECURE.resolveModelAttribute(context, fullModel).asBoolean());
        service.setEnableLookups(WebConnectorDefinition.ENABLE_LOOKUPS.resolveModelAttribute(context, fullModel).asBoolean());
        ModelNode resolved;
        if ((resolved =  WebConnectorDefinition.PROXY_NAME.resolveModelAttribute(context, fullModel)).isDefined()) {
            service.setProxyName(resolved.asString());
        }
        if ((resolved =  WebConnectorDefinition.PROXY_PORT.resolveModelAttribute(context, fullModel)).isDefined()) {
            service.setProxyPort(resolved.asInt());
        }
        if ((resolved =  WebConnectorDefinition.MAX_POST_SIZE.resolveModelAttribute(context, fullModel)).isDefined()) {
            service.setMaxPostSize(resolved.asInt());
        }
        if ((resolved =  WebConnectorDefinition.MAX_SAVE_POST_SIZE.resolveModelAttribute(context, fullModel)).isDefined()) {
            service.setMaxSavePostSize(resolved.asInt());
        }
        if ((resolved =  WebConnectorDefinition.MAX_CONNECTIONS.resolveModelAttribute(context, fullModel)).isDefined()) {
            service.setMaxConnections(resolved.asInt());
        }
        if ((resolved =  WebConnectorDefinition.VIRTUAL_SERVER.resolveModelAttribute(context, fullModel)).isDefined()) {
            List<String> vServers = new LinkedList<String>();
            for (ModelNode vServer:resolved.asList()){
                vServers.add(vServer.asString());
            }
            service.setVirtualServers(vServers);
        }
        if (fullModel.get(SSL_PATH.getKey(), SSL_PATH.getValue()).isDefined()) {
            service.setSsl(resolveSslExpressions(context, fullModel.get(SSL_PATH.getKey(), SSL_PATH.getValue())));
        }
        final ServiceBuilder<Connector> serviceBuilder = context.getServiceTarget().addService(WebSubsystemServices.JBOSS_WEB_CONNECTOR.append(name), service)
                .addDependency(WebSubsystemServices.JBOSS_WEB, WebServer.class, service.getServer())
                .addDependency(SocketBinding.JBOSS_BINDING_NAME.append(bindingRef), SocketBinding.class, service.getBinding());
        if((resolved =   WebConnectorDefinition.REDIRECT_BINDING.resolveModelAttribute(context, fullModel)).isDefined()) {
            serviceBuilder.addDependency(SocketBinding.JBOSS_BINDING_NAME.append(resolved.asString()), SocketBinding.class, service.getRedirect());
        } else if ((resolved =  WebConnectorDefinition.REDIRECT_PORT.resolveModelAttribute(context, fullModel)).isDefined()) {
            service.setRedirectPort(resolved.asInt());
        }
        if((resolved = WebConnectorDefinition.PROXY_BINDING.resolveModelAttribute(context, fullModel)).isDefined()) {
            serviceBuilder.addDependency(OutboundSocketBinding.OUTBOUND_SOCKET_BINDING_BASE_SERVICE_NAME.append(resolved.asString()),
                    OutboundSocketBinding.class, service.getProxy());
        }
        if ((resolved =  WebConnectorDefinition.EXECUTOR.resolveModelAttribute(context, fullModel)).isDefined()) {
            String executorRef = resolved.asString();
            serviceBuilder.addDependency(ThreadsServices.executorName(executorRef), Executor.class, service.getExecutor());
        }
        serviceBuilder.setInitialMode(enabled ? Mode.ACTIVE : Mode.NEVER);
        if (enabled) {
            serviceBuilder.addListener(verificationHandler);
        }
        final ServiceController<Connector> serviceController = serviceBuilder.install();
        if (newControllers != null) {
            newControllers.add(serviceController);
        }
    }

    private ModelNode resolveSslExpressions(OperationContext context, ModelNode ssl) throws OperationFailedException {
        ModelNode result = new ModelNode();
        for (AttributeDefinition def : WebSSLDefinition.SSL_ATTRIBUTES) {
            result.get(def.getName()).set(def.resolveModelAttribute(context, ssl));
        }
        return result;
    }

}
