/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2016, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.test.integration.domain.ds;

import java.io.File;
import java.net.URL;

import org.jboss.as.controller.client.helpers.Operations;
import org.jboss.as.test.integration.domain.management.util.DomainControllerClientConfig;
import org.jboss.as.test.integration.domain.management.util.DomainLifecycleUtil;
import org.jboss.as.test.integration.domain.management.util.JBossAsManagedConfiguration;
import org.jboss.dmr.ModelNode;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;


/**
 * Base for domain tests for BZ-1289712. Basically turn off DS, kill domain, start it and check if DS is off.
 * 
 * @author baranowb
 * 
 */
public abstract class DSOnOffTestBase {

    private String HOST = "host-ds";
    private DomainLifecycleUtil hostUtils;
    private static String masterAddress = System.getProperty("jboss.test.host.master.address");

    protected ModelNode addressResource;
    protected ModelNode addressConfiguration;

    protected abstract String getDomainXML();
    protected abstract ModelNode getAddressResource();
    protected abstract ModelNode getAddressConfiguration();

    @Before
    public void setupDomain() throws Exception {
        final DomainControllerClientConfig config = DomainControllerClientConfig.create();
        this.addressResource = getAddressResource();
        this.addressConfiguration = getAddressConfiguration();
        this.hostUtils = new DomainLifecycleUtil(getHostConfiguration(), config);
        this.hostUtils.start();
    }

    @After
    public void shutdownDomain() {
        this.hostUtils.stop();
    }

    private JBossAsManagedConfiguration getHostConfiguration() throws Exception {

        final String testName = DSOnOffTestBase.class.getSimpleName();
        File domains = new File("target" + File.separator + "domains" + File.separator + testName);
        final File hostDir = new File(domains, HOST);
        final File hostConfigDir = new File(hostDir, "configuration");
        assert hostConfigDir.mkdirs() || hostConfigDir.isDirectory();

        ClassLoader tccl = Thread.currentThread().getContextClassLoader();
        final JBossAsManagedConfiguration hostConfig = new JBossAsManagedConfiguration();
        hostConfig.setHostControllerManagementAddress(masterAddress);
        hostConfig.setHostCommandLineProperties("-Djboss.test.host.master.address=" + masterAddress);
        URL url = tccl.getResource("domain-configs/"+getDomainXML());
        assert url != null;
        hostConfig.setDomainConfigFile(new File(url.toURI()).getAbsolutePath());
        System.out.println(hostConfig.getDomainConfigFile());
        url = tccl.getResource("host-configs/" + HOST + ".xml");
        assert url != null;
        hostConfig.setHostConfigFile(new File(url.toURI()).getAbsolutePath());
        System.out.println(hostConfig.getHostConfigFile());
        hostConfig.setDomainDirectory(hostDir.getAbsolutePath());
        hostConfig.setHostName(HOST);
        hostConfig.setHostControllerManagementPort(9999);
        hostConfig.setStartupTimeoutInSeconds(120);

        return hostConfig;
    }

    @Test
    public void testDSOffAndRestart() throws Exception {

        final ModelNode readResource = Operations.createOperation("read-attribute");
        readResource.get("address").set(this.addressResource);
        readResource.get("name").set("enabled");
        ModelNode result = hostUtils.executeForResult(readResource);

        Assert.assertTrue("DS has not been enabled!!!"+result.toString(), Boolean.parseBoolean(result.toString()));
        final ModelNode operation = Operations.createOperation("disable");
        operation.get("name").set("ExampleDS");
        operation.get("address").set(addressConfiguration);
        operation.get("operation-headers").get("allow-resource-service-restart").set(true);
        // result "undefined"....

        hostUtils.executeForResult(operation);
        hostUtils.stop();
        hostUtils.start();
        
        result = hostUtils.executeForResult(readResource);
        Assert.assertFalse("DS is still in enabled state!!!", Boolean.parseBoolean(result.toString()));

    }
}
