/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.test.integration.ejb.security.asynchronous.remotingcontext;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.as.test.integration.security.common.AbstractSecurityDomainsServerSetupTask;
import org.jboss.as.test.integration.security.common.Utils;
import org.jboss.as.test.integration.security.common.config.SecurityDomain;
import org.jboss.as.test.integration.security.common.config.SecurityModule;
import org.jboss.as.test.shared.TimeoutUtil;
import org.jboss.as.test.shared.integration.ejb.security.Util;
import org.jboss.ejb.client.ContextSelector;
import org.jboss.ejb.client.EJBClientConfiguration;
import org.jboss.ejb.client.EJBClientContext;
import org.jboss.ejb.client.PropertiesBasedEJBClientConfiguration;
import org.jboss.ejb.client.remoting.ConfigBasedEJBClientContextSelector;
import org.jboss.logging.Logger;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

import javax.naming.Context;
import javax.naming.NamingException;
import java.io.IOException;
import java.net.UnknownHostException;
import java.util.Properties;

import static org.junit.Assert.assertEquals;

/**
 * This tests remote connection context when doing async EJB invocation.
 * </p>
 *
 * @author studensky
 */
@RunWith(Arquillian.class)
@ServerSetup({RemotingContextAsyncTestCase.SecurityDomainSetup.class})
@RunAsClient
public class RemotingContextAsyncTestCase {
    private static final Logger log = Logger.getLogger(RemotingContextAsyncTestCase.class);

    static final String SECURITY_DOMAIN_NAME = "remoting-context-async-test";

    private static final String ARCHIVE_NAME = "RemotingContextAsyncTestCase";

    private static final String CONNECTION_USERNAME = "user1";
    private static final String CONNECTION_PASSWORD = "password1";

    static class SecurityDomainSetup extends AbstractSecurityDomainsServerSetupTask {
        /**
         * Returns SecurityDomains configuration for this testcase.
         *
         * <pre>
         * &lt;security-domain name=&quot;switch-identity-test&quot; cache-type=&quot;default&quot;&gt;
         *     &lt;authentication&gt;
         *         &lt;login-module code=&quot;Remoting&quot; flag=&quot;optional&quot;&gt;
         *             &lt;module-option name=&quot;password-stacking&quot; value=&quot;useFirstPass&quot;/&gt;
         *         &lt;/login-module&gt;
         *         &lt;login-module code=&quot;RealmDirect&quot; flag=&quot;required&quot;&gt;
         *             &lt;module-option name=&quot;password-stacking&quot; value=&quot;useFirstPass&quot;/&gt;
         *         &lt;/login-module&gt;
         *     &lt;/authentication&gt;
         * &lt;/security-domain&gt;
         * </pre>
         *
         * @see org.jboss.as.test.integration.security.common.AbstractSecurityDomainsServerSetupTask#getSecurityDomains()
         */
        @Override
        protected SecurityDomain[] getSecurityDomains() {
            final SecurityModule.Builder loginModuleBuilder = new SecurityModule.Builder().putOption(
                    "password-stacking", "useFirstPass");
            final SecurityDomain sd = new SecurityDomain.Builder()
                    .name(SECURITY_DOMAIN_NAME)
                    .loginModules(
                            loginModuleBuilder.name("Remoting").flag("optional").build(),
                            loginModuleBuilder.name("RealmDirect").flag("required").build())
                    .build();
            return new SecurityDomain[] { sd };
        }
    }

    @Deployment
    public static Archive<?> deploy() {
        JavaArchive jar = ShrinkWrap.create(JavaArchive.class, ARCHIVE_NAME + ".jar");
        jar.addClasses(GoodBye.class, GoodByeBean.class);
        jar.addClasses(Hello.class, HelloBean.class);
        jar.addClass(TimeoutUtil.class);
        log.info(jar.toString(true));
        return jar;
    }

    @ArquillianResource
    private ManagementClient mgmtClient;

    private Context context;
    private ContextSelector<EJBClientContext> previousClientContextSelector;

    @Before
    public void before() throws Exception {
        context = Util.createNamingContext();
        // setup the client context selector
        this.previousClientContextSelector = setupEJBClientContextSelector();
    }

    @After
    public void after() throws Exception {
        if (this.previousClientContextSelector != null) {
            EJBClientContext.setSelector(this.previousClientContextSelector);
        }
        context.close();
    }

    @Test
    public void testRemotingContext() throws Exception {
        Hello bean = lookupEJB(HelloBean.class, Hello.class);
        final String hello = bean.sayHello();
        assertEquals("Hello user1!", hello);
    }

    /**
     * Sets up the EJB client context to use a selector which processes and sets up EJB receivers
     * based on this testcase specific jboss-ejb-client.properties file
     *
     * @return
     * @throws java.io.IOException
     */
    private ContextSelector<EJBClientContext> setupEJBClientContextSelector() throws IOException {
        final Properties properties = createEjbClientConfiguration(Utils.getHost(mgmtClient));
        final EJBClientConfiguration ejbClientConfiguration = new PropertiesBasedEJBClientConfiguration(properties);
        final ConfigBasedEJBClientContextSelector selector = new ConfigBasedEJBClientContextSelector(ejbClientConfiguration);

        return EJBClientContext.setSelector(selector);
    }

    private static Properties createEjbClientConfiguration(String hostName) throws UnknownHostException {
        final Properties pr = new Properties();
        pr.put("remote.connectionprovider.create.options.org.xnio.Options.SSL_ENABLED", "false");
        pr.put("remote.connections", "default");
        pr.put("remote.connection.default.host", hostName);
        pr.put("remote.connection.default.port", "4447");
        pr.put("remote.connection.default.username", CONNECTION_USERNAME);
        pr.put("remote.connection.default.password", CONNECTION_PASSWORD);
        return pr;
    }

    private <T> T lookupEJB(Class<? extends T> beanImplClass, Class<T> remoteInterface) throws NamingException {
        return remoteInterface.cast(context.lookup("ejb:/" + ARCHIVE_NAME + "/" + beanImplClass.getSimpleName() + "!"
                + remoteInterface.getName()));
    }

}
