/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.aspects.remoting;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import org.jboss.aop.Dispatcher;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aop.joinpoint.MethodInvocation;
import org.jboss.aop.util.MethodHashing;
import org.jboss.aop.util.PayloadKey;
import org.jboss.remoting.InvokerLocator;

/**
 * comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author <a href="andrew.rubinger@jboss.org">ALR</a> Maintain only 
 */
public class PojiProxy implements java.io.Serializable, InvocationHandler
{
   private static final long serialVersionUID = 8054816523858555978L;

   private Object oid;

   private InvokerLocator uri;

   private Interceptor[] interceptors;

   public PojiProxy(Object oid, InvokerLocator uri, Interceptor[] interceptors)
   {
      this.oid = oid;
      this.uri = uri;
      this.interceptors = interceptors;
   }

   public Object invoke(Object proxy, Method method, Object[] args) throws Throwable
   {
      MethodInvocation sri = this.constructMethodInvocation(method, args);
      this.addMetadataToInvocation(sri);
      this.addArgumentsToInvocation(sri, args);
      return sri.invokeNext();
   }

   /**
    * Constructs a MethodInvocation from the specified Method and
    * arguments
    * 
    * @param method
    * @param args
    * @return
    */
   protected MethodInvocation constructMethodInvocation(Method method, Object[] args)
   {
      long hash = MethodHashing.calculateHash(method);
      MethodInvocation sri = new MethodInvocation(interceptors, hash, method, method, null);
      return sri;
   }

   /**
    * Adds metadata to the specified MethodInvocation before
    * invocation is called.  Extracted to allow subclasses to control
    * which metadata is added to a PojiProxy 
    * 
    * @param methodInvocation
    */
   protected void addMetadataToInvocation(MethodInvocation methodInvocation)
   {
      methodInvocation.getMetaData().addMetaData(Dispatcher.DISPATCHER, Dispatcher.OID, oid, PayloadKey.AS_IS);
      methodInvocation.getMetaData().addMetaData(InvokeRemoteInterceptor.REMOTING,
            InvokeRemoteInterceptor.INVOKER_LOCATOR, uri, PayloadKey.TRANSIENT);
      methodInvocation.getMetaData().addMetaData(InvokeRemoteInterceptor.REMOTING, InvokeRemoteInterceptor.SUBSYSTEM,
            "AOP", PayloadKey.TRANSIENT);
   }

   /**
    * Sets the specified arguments on the specified invocation.  Extracted to
    * provide indirection such that the arguments set on the invocation may be 
    * different from those originally passed in.
    * 
    * @param invocation
    * @param originalArguments
    */
   protected void addArgumentsToInvocation(MethodInvocation invocation, Object[] originalArguments)
   {
      invocation.setArguments(originalArguments);
   }

   protected Interceptor[] getInterceptors()
   {
      return interceptors;
   }

   protected Object getOid()
   {
      return oid;
   }

   protected InvokerLocator getUri()
   {
      return uri;
   }
}
