/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.jms;

import org.jboss.logging.Logger;
import org.jboss.util.NestedRuntimeException;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.Map;

/**
 * 
 * @author <mailto:clebert.suconic@jboss.org">Clebert Suconic</a>
 * 
 */
public class LegacyJMSTestAdmin extends JMSTestAdmin
{
   private final MBeanServerConnection jmx;

   protected final ObjectName mainDeployer = new ObjectName("jboss.system:service=MainDeployer");

   protected static final ObjectName namingService;
   
   protected static final ObjectName namingService2;
   
   protected static final ObjectName serverPeer;
   
   
   static
   {
      try
      {
         namingService = new ObjectName("jboss:service=Naming");
      }
      catch (Exception e)
      {
         throw new NestedRuntimeException(e);
      }

      try
      {
         namingService2 = new ObjectName("jboss:service=NamingBeanImpl");
      }
      catch (Exception e)
      {
         throw new NestedRuntimeException(e);
      }

      try
      {
          serverPeer = new ObjectName("jboss.messaging:service=ServerPeer");
      }
      catch (Exception e)
      {
         throw new NestedRuntimeException(e);
      }

   }


   
   /** The static log */
   private static final Logger staticLog = Logger.getLogger(LegacyJMSTestAdmin.class);

   public LegacyJMSTestAdmin() throws Exception
   {
      InitialContext ctx = new InitialContext();
      String adaptorName = System.getProperty("jbosstest.server.name", "jmx/invoker/RMIAdaptor");
      jmx = (MBeanServerConnection) ctx.lookup(adaptorName);
      ctx.close();
   }

   @Override
   public boolean isSupported()
   {
      try
      {
         createQueue("deleteMeTest");
         destroyCreatedDestinations();
         
         staticLog.info("Legacy is supported!!!");

         return true;
      } catch (Throwable e)
      {
         staticLog.debug("LegacyJMSTestAdmin is not support because an exception happened", e);
         return false;
      }
   }

   public void createQueue(String name, TestRole... securityConfig) throws Exception
   {

      createQueue(name, new String[]
      { "/queue/" + name, name }, securityConfig);
   }

   public void createQueue(String name, String[] jndiArray, TestRole... securityConfig) throws Exception
   {

      createDestination(false, name, securityConfig);

      deployedQueues.put(name, securityConfig);
   }

   public void createTopic(String name, TestRole... securityConfig) throws Exception
   {
      createTopic(name, new String[]
      { "/topic/" + name, name }, securityConfig);
   }

   public void createTopic(String name, String[] jndiArray, TestRole... securityConfig) throws Exception
   {

      createDestination(true, name, securityConfig);

      deployedTopics.put(name, securityConfig);
   }

   public void deleteQueue(String name) throws Exception
   {
      try
      {
         deployedQueues.remove(name);
         undeploy(false, name);
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         throw e;
      }
   }

   public void deleteTopic(String name) throws Exception
   {
      try
      {
         deployedTopics.remove(name);
         undeploy(true, name);
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         throw e;
      }
   }

   /**
    * This method only exists because hornetQ doesn't actually delete the
    * destination on undeploy. On HornetQ we need to remove every destination
    * from the storage while on Legacy versions a simple undeploy is fine
    */
   public void destroyEveryDestination() throws Exception
   {
      destroyCreatedDestinations();
   }

   @Override
   public void destroyCreatedDestinations() throws Exception
   {
      for (Map.Entry<String, TestRole[]> queue : deployedQueues.entrySet())
      {
         undeploy(false, queue.getKey());
      }

      for (Map.Entry<String, TestRole[]> topic : deployedTopics.entrySet())
      {
         undeploy(true, topic.getKey());
      }

      deployedQueues.clear();
      deployedTopics.clear();
   }

   @Override
   public ObjectName createQueueJMXName(String queueName)
   {
      try
      {
         return new ObjectName("jboss.mq.destination:service=Queue,name=" + queueName);
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         e.printStackTrace(); // >> junit reports
         return null;
      }
   }

   @Override
   public ObjectName createTopicJMXName(String queueName)
   {
      try
      {
         return new ObjectName("jboss.mq.destination:service=Topic,name=" + queueName);
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         e.printStackTrace(); // >> junit reports
         return null;
      }
   }
   
   public ObjectName createDestinationJMXName(boolean isTopic, String destName)
   {
	   if (isTopic)
	   {
		   return createTopicJMXName(destName);
	   }
	   else
	   {
		   return createQueueJMXName(destName);
	   }
   }

   private void undeploy(boolean isTopic, String destinationName) throws Exception
   {

      ObjectName objname;
      
      
      if (isTopic)
      {
    	  objname = new ObjectName("jboss.mq.destination:service=Topic,name=" + destinationName);
      }
      else
      {
    	  objname = new ObjectName("jboss.mq.destination:service=Queue,name=" + destinationName);
      }

      try
      {
          jmx.invoke(objname, "removeAllMessages", null, null);
          if (isTopic)
          {
        	  jmx.invoke(serverPeer, "destroyTopic", new Object[] { destinationName },  new String[] { String.class.getName() } );
          }
          else
          {
        	  jmx.invoke(serverPeer, "destroyQueue", new Object[] { destinationName },  new String[] { String.class.getName() } );
          }
      }
      catch (Exception e)
      {
    	  staticLog.warn(e.getMessage(), e);
      }

      File destFile = createFile(isTopic, destinationName);
      
      try
      {
	      jmx.invoke(mainDeployer, "undeploy", new Object[]
	      { destFile.toURI().toURL() }, new String[]
	      { "java.net.URL" });
      }
      finally
      {
    	  try
    	  {
    		  destFile.delete();
    	  }
    	  catch (Exception ignored)
    	  {
    	  }
      }



   }

   private void createDestination(boolean isTopic, String name, TestRole... securityConfig) throws Exception
   {
      // OK, lets see if we can figure out what it might be.
      File fileout = createFile(isTopic, name);

      if (fileout.exists())
      {
         fileout.delete();
      }

      FileOutputStream out = new FileOutputStream(fileout);

      PrintStream str = new PrintStream(out);
      str.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
      str.println("<server>");
      if (isTopic)
      {
         str.println("  <mbean code=\"org.jboss.mq.server.jmx.Topic\" name=\"jboss.mq.destination:service=Topic,name=" + name + "\">");
      } else
      {
         str.println("  <mbean code=\"org.jboss.mq.server.jmx.Queue\" name=\"jboss.mq.destination:service=Queue,name=" + name + "\">");
      }
      str.println("    <attribute name=\"JNDIName\">" + name + "</attribute>");
      str.println("    <depends optional-attribute-name=\"DestinationManager\">jboss.mq:service=DestinationManager</depends>");
      str.println("    <depends optional-attribute-name=\"SecurityManager\">jboss.mq:service=SecurityManager</depends>");

      if (securityConfig != null && securityConfig.length > 0)
      {
         str.println("    <attribute name=\"SecurityConf\">");
         str.println("      <security>");
         for (TestRole role : securityConfig)
         {
            str.println("        <role name=" + q(role.getName()) +
               " read=" + q(role.isConsume()) +
               " write=" + q(role.isSend()) +
               " create=" + q(role.isCreateDurableQueue() || role.isCreateNonDurableQueue()) + "/>");
         }
         str.println("      </security>");
         str.println("    </attribute>");
      }
      
      str.println("  </mbean>");

      str.println("</server>");

      jmx.invoke(mainDeployer, "deploy", new Object[]
      { fileout.toURI().toURL() }, new String[]
      { "java.net.URL" });
      
      String newName;
      
      if (isTopic)
      {
    	  newName = "/topic/" + name;
      }
      else
      {
    	  newName = "/queue/" + name;
      }
      
      try
      {
    	  jmx.invoke(namingService, "createAlias", new Object[] { newName, name }, new String[] { String.class.getName(), String.class.getName() });
      }
      catch (Throwable e)
      {
    	  staticLog.debug("error creating alias", e);
      }
      
      // AS5 is using a different JMX-name, so we try both
      /*try
      {
    	  jmx.invoke(namingService2, "createAlias", new Object[] { newName, name }, new String[] { String.class.getName(), String.class.getName() });
      }
      catch (Throwable e)
      {
    	  staticLog.debug("error creating alias", e);
      }*/
      

   }

   private File createFile(boolean isTopic, String destinationName)
   {
      String deployDir = System.getProperty("jbosstest.deploy.dir");
      if (deployDir == null)
      {
         deployDir = "output/lib";
      }
      File fileout = new File(deployDir + "/" + "tmpdest_" + (isTopic ? "topic_" : "queue_") + destinationName + "-destination-service.xml");
      return fileout;
   }

   private static final String q(Object s)
   {
      return "\"" + s + "\"";
   }
}
