/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.cluster.test;

import java.io.IOException;
import java.util.Iterator;
import java.util.Set;

import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.servlet.ServletException;

import org.apache.catalina.Context;
import org.apache.catalina.Manager;
import org.apache.catalina.Pipeline;
import org.apache.catalina.Valve;
import org.apache.catalina.connector.Request;
import org.apache.catalina.connector.Response;
import org.jboss.cache.Fqn;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;
import org.jboss.test.cluster.web.mocks.MockRequest;
import org.jboss.test.cluster.web.mocks.RequestHandler;
import org.jboss.test.cluster.web.mocks.RequestHandlerValve;

/**
 * Utilities for session testing.
 * 
 * @author <a href="mailto://brian.stansberry@jboss.com">Brian Stansberry</a>
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 * @version $Revision: 102796 $
 */
public class SessionTestUtil
{  
   private static final String[] TYPES = 
   { String.class.getName(), Object.class.getName() };
   private static final String VERSION_KEY = "VERSION";
   
   private static final Fqn BUDDY_BACKUP_SUBTREE_FQN = Fqn.fromString("_BUDDY_BACKUP_");
   
   private static final ObjectName CACHE_OBJECT_NAME;
   static
   {
      try
      {
         CACHE_OBJECT_NAME =
            new ObjectName("jboss.cache:service=TomcatClusteringCache");
      }
      catch (MalformedObjectNameException e)
      {
         throw new ExceptionInInitializerError(e);
      }
   }
   
   public static Object getSessionVersion(MBeanServerConnection adaptor, String sessionFqn) throws Exception
   {
      return adaptor.invoke(CACHE_OBJECT_NAME, "get", new Object[]
      {sessionFqn, VERSION_KEY}, TYPES);
   }

   public static Object getBuddySessionVersion(MBeanServerConnection adaptor, String sessionFqn) throws Exception
   {
      Object replVersion = null;
      //    Check in the buddy backup tree
      Set buddies = (Set) adaptor.invoke(CACHE_OBJECT_NAME, "getChildrenNames", new Object[]
      {BUDDY_BACKUP_SUBTREE_FQN}, new String[]
      {Fqn.class.getName()});

      if (buddies != null)
      {
         for (Iterator it = buddies.iterator(); it.hasNext() && replVersion == null;)
         {
            Fqn fqn = new Fqn(BUDDY_BACKUP_SUBTREE_FQN, it.next());
            fqn = new Fqn(fqn, Fqn.fromString(sessionFqn));
            replVersion = adaptor.invoke(CACHE_OBJECT_NAME, "get", new Object[]
            {fqn.toString(), VERSION_KEY}, TYPES);
         }
      }

      return replVersion;
   }

   public static void invokeRequest(Manager manager, RequestHandler handler, String sessionId)
      throws ServletException, IOException
   {
      Valve valve = setupPipeline(manager, handler);
      Request request = setupRequest(manager, sessionId);
      invokeRequest(valve, request);
   }
   
   public static void invokeRequest(Valve pipelineHead, Request request)
      throws ServletException, IOException
   {
      pipelineHead.invoke(request, request.getResponse());
      // StandardHostValve calls request.getSession(false) on way out, so we will too
      request.getSession(false);
      request.recycle();
   }
   
   public static Valve setupPipeline(Manager manager, RequestHandler requestHandler)
   {
      Pipeline pipeline = manager.getContainer().getPipeline();
      
      // Clean out any existing request handler
      Valve[] valves = pipeline.getValves();
      RequestHandlerValve mockValve = null;
      for (Valve valve: valves)
      {
         if (valve instanceof RequestHandlerValve)         
         {
            mockValve = (RequestHandlerValve) valve;
            break;
         }
      }
      
      if (mockValve == null)
      {
         mockValve = new RequestHandlerValve(requestHandler);
         pipeline.addValve(mockValve);
      }
      else
      {
         mockValve.setRequestHandler(requestHandler);
      }
      
      return pipeline.getFirst();
   }

   public static Request setupRequest(Manager manager, String sessionId)
   {
      MockRequest request = new MockRequest();
      request.setRequestedSessionId(sessionId);
      request.setContext((Context) manager.getContainer());
      Response response = new Response();
      request.setResponse(response);
      return request;
   }
   
   private SessionTestUtil() {}
}
