/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.service;

import java.lang.reflect.Method;

import javax.management.MBeanException;
import javax.management.MBeanOperationInfo;
import javax.management.NotCompliantMBeanException;

import org.jboss.logging.Logger;
import org.jboss.mx.interceptor.AbstractInterceptor;
import org.jboss.mx.interceptor.Interceptor;
import org.jboss.mx.interceptor.ReflectedDispatcher;
import org.jboss.mx.modelmbean.XMBean;
import org.jboss.mx.server.Invocation;
import org.jboss.mx.server.InvocationContext;
import org.jboss.mx.server.MethodMapper;

/**
 * A XMBean which dispatches to the ServiceContainer instead of to the target bean.
 * 
 * @author <a href="mailto:cdewolf@redhat.com">Carlo de Wolf</a>
 * @version $Revision: $
 */
public class ServiceXMBeanDelegate extends XMBean
{
   private static final Logger log = Logger.getLogger(ServiceXMBeanDelegate.class);
   
   private ServiceContainer container;
   
   /**
    * @param container
    * @param resourceType
    * @throws MBeanException
    * @throws NotCompliantMBeanException
    */
   public ServiceXMBeanDelegate(ServiceContainer container, String resourceType) throws MBeanException, NotCompliantMBeanException
   {
      super(container.getSingleton(), resourceType);
      
      this.container = container;
   }
   
   @Override
   protected void initDispatchers()
   {
      super.initDispatchers();
      
      // map the Methods on the target resource for easy access
      MethodMapper mmap = new MethodMapper(getResource().getClass());
      
      MBeanOperationInfo[] operations = info.getOperations();
      
      // Set the dispatchers for the operations
      for (int i = 0; i < operations.length; ++i)
      {
         MBeanOperationInfo op = operations[i];
         
         // hack for backwards compatibility
         // In the old return type was ignore, in the new it's not and
         // specifying java.lang.Void is wrong.
         if(op.getReturnType().equals("java.lang.Void"))
         {
            op = new MBeanOperationInfo(op.getName(), op.getDescription(), op.getSignature(), "void", op.getImpact());
         }
         
         OperationKey opKey = new OperationKey(op.getName(), op.getSignature());
         InvocationContext ctx = (InvocationContext) operationContextMap.get(opKey);

         Interceptor dispatcher = ctx.getDispatcher();
         
         // Reconfigure if we have a Reflected dispatcher
         if (dispatcher instanceof ReflectedDispatcher)
         {
            // Don't reset the dispatcher
            //dispatcher = null;
            Method m = mmap.lookupOperation(op);
            if (m == null)
            {
               // TODO: ehr, I would say this is an error, but there seems to be
               // something like late binding in the original code. Let's just issue a warning.
               log.warn("operation " + op + " was not found, ignoring");
               /*
               // Look for an method on the model mbean
               m = MethodMapper.lookupOperation(op, this);
               if (m != null)
               {
                  // operation found on the 'this' invoker
                  target = this;
                  dispatcher = new ReflectedDispatcher(m, dynamicResource);
               }
               else
               {
                  // operation not found, use late binding
                  // What is this late binding attempt and should there be a warning?
                  dispatcher = new ReflectedDispatcher(dynamicResource);
               }
               */
            }
            else
            {
               // operation found on the resource
               dispatcher = new ContainerDispatcher(m);
            }
            ctx.setDispatcher(dispatcher);
         }
      }
   }
   
   private class ContainerDispatcher extends AbstractInterceptor
   {
      private Method method;
      
      protected ContainerDispatcher(Method method)
      {
         this.method = method;
      }
      
      @Override
      public Object invoke(Invocation invocation) throws Throwable
      {
         return container.localInvoke(method, invocation.getArgs());
      }
   }
}
