/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.enc;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import org.jboss.logging.Logger;

import javax.naming.NameNotFoundException;

import org.jboss.ejb3.DeploymentScope;
import org.jboss.ejb3.Ejb3Deployment;
import org.jboss.ejb3.PersistenceUnitRegistry;
import org.jboss.ejb3.entity.PersistenceUnitDeployment;

/**
 * Resolves persistence units for @PersistenceContext and @PersistenceUnit
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision: 96883 $
 */
public class DeploymentPersistenceUnitResolver
{
   private static final Logger log = Logger.getLogger(DeploymentPersistenceUnitResolver.class);
   
   protected List<PersistenceUnitDeployment> persistenceUnitDeployments;
   protected DeploymentScope deploymentScope;
   protected LinkedHashMap ejbContainers;

   public DeploymentPersistenceUnitResolver(List<PersistenceUnitDeployment> persistenceUnitDeployments, DeploymentScope deploymentScope, LinkedHashMap ejbContainers)
   {
      this.persistenceUnitDeployments = persistenceUnitDeployments;
      this.deploymentScope = deploymentScope;
      this.ejbContainers = ejbContainers;
   }

   public PersistenceUnitDeployment getPersistenceUnitDeployment(String unitName) throws NameNotFoundException
   {      
      if ("".equals(unitName))
      {
         if (persistenceUnitDeployments == null)
         {
            throw new NameNotFoundException("EMPTY STRING unitName but there is no deployments in scope");
         }
         if (persistenceUnitDeployments.size() == 1 && ejbContainers.size() > 0)
         {
            return persistenceUnitDeployments.get(0);
         }
         else if (persistenceUnitDeployments.size() > 1)
         {
            throw new NameNotFoundException("EMPTY STRING unitName and there is more than one scoped persistence unit");
         }
         throw new NameNotFoundException("There is no default persistence unit in this deployment.");
      }
      int hashIndex = unitName.indexOf('#');
      if (hashIndex != -1)
      {
         String relativePath = unitName.substring(0, hashIndex);
         String name = unitName.substring(hashIndex + 1);
         if (deploymentScope == null)
         {
            String relativeJarName = relativePath.substring(3);
            // look in global EJB jars.
            for (PersistenceUnitDeployment pud : PersistenceUnitRegistry.getPersistenceUnits())
            {
               String jarName = pud.getDeployment().getDeploymentUnit().getShortName() + ".jar";
               if (pud.getDeployment().getEar() == null
                       && jarName.equals(relativeJarName)
                       && pud.getEntityManagerName().equals(name)
                       && pud.isScoped())
               {
                  return pud;
               }
            }
         
            return null;
         }
         Ejb3Deployment dep = deploymentScope.findRelativeDeployment(relativePath);
         if (dep == null)
         {
            return null;
         }
         PersistenceUnitDeployment rtn = dep.getPersistenceUnitDeploymentInternal(name);
         
         return rtn;
      }
      PersistenceUnitDeployment rtn = getPersistenceUnitDeploymentInternal(unitName);
      if (rtn != null)
         return rtn;

      Set<PersistenceUnitDeployment> possibleMatches = new HashSet<PersistenceUnitDeployment>();
      for (PersistenceUnitDeployment deployment : PersistenceUnitRegistry.getPersistenceUnits())
      {
         if (deployment.isScoped()) continue;
         if (deployment.getEntityManagerName().equals(unitName))
         {
            possibleMatches.add(deployment);
         }
      }
      if (possibleMatches.isEmpty())
      {
         log.debug("No matching persistence unit with name " + unitName + " found");
         return null;
      }
      // if only one persistence unit was found then just return it
      if (possibleMatches.size() == 1)
      {
         return possibleMatches.iterator().next();
      }
      log.debug("Multiple persistence units with name " + unitName + " found. Will narrow down based on scope");
      // if multiple persistence units were found, then identify the right one
      // based on scope (JBPAPP-604)
      for (PersistenceUnitDeployment deployment : possibleMatches)
      {
         // 1) get the ear to which the persistence unit belongs to
         DeploymentScope persistenceDeploymentScope = deployment.getDeployment().getEar();
         if (deploymentScope == null)
         {
            log.debug("Persistence unit deployment " + deployment.getKernelName() + " identified as the best match");
            return deployment;
         }
         // check if the persistence unit deployment scope matches the scope of  
         // this resolver
         else if (persistenceDeploymentScope != null
               && persistenceDeploymentScope.getBaseName().equals(deploymentScope.getBaseName()))
         {
            log.debug("Persistence unit deployment " + deployment.getKernelName() + " identified as the best match");
            return deployment;
         }

      }
             
      return null;
   }

   public PersistenceUnitDeployment getPersistenceUnitDeploymentInternal(String unitName)
   {
      if (persistenceUnitDeployments != null)
      {
         for (PersistenceUnitDeployment deployment : persistenceUnitDeployments)
         {
            if (deployment.getEntityManagerName().equals(unitName))
            {
               return deployment;
            }
         }
      }
      return null;
   }
}
