/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.stateful;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.ejb.EJBHome;
import javax.ejb.RemoteHome;
import javax.naming.NamingException;

import org.jboss.annotation.ejb.Clustered;
import org.jboss.annotation.ejb.RemoteBinding;
import org.jboss.aop.Advisor;
import org.jboss.aop.AspectManager;
import org.jboss.aop.Dispatcher;
import org.jboss.aop.advice.AdviceStack;
import org.jboss.aspects.remoting.FamilyWrapper;
import org.jboss.aspects.remoting.Remoting;
import org.jboss.ejb3.Container;
import org.jboss.ejb3.EJBContainer;
import org.jboss.ejb3.JBossProxy;
import org.jboss.ejb3.ProxyFactory;
import org.jboss.ejb3.ProxyFactoryHelper;
import org.jboss.ejb3.SessionContainer;
import org.jboss.ejb3.remoting.RemoteProxyFactory;
import org.jboss.ejb3.stateless.StatelessClusterProxyFactory;
import org.jboss.ha.framework.interfaces.ClusteringTargetsRepository;
import org.jboss.ha.framework.interfaces.DistributedReplicantManager;
import org.jboss.ha.framework.interfaces.FirstAvailable;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.ha.framework.interfaces.LoadBalancePolicy;
import org.jboss.ha.framework.interfaces.RoundRobin;
import org.jboss.ha.framework.server.HATarget;
import org.jboss.logging.Logger;
import org.jboss.naming.Util;
import org.jboss.remoting.InvokerLocator;


/**
 * Comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author Brian Stansberry
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 * @version $Revision: 74347 $
 */
public class StatefulClusterProxyFactory extends BaseStatefulProxyFactory 
   implements RemoteProxyFactory, DistributedReplicantManager.ReplicantListener
{
   private static final Logger log = Logger.getLogger(StatefulClusterProxyFactory.class);
   
//   public static final String FACTORY_ATTRIBUTE = ",element=ProxyFactory,partition=";
   
   private RemoteBinding binding;
   private InvokerLocator locator;
   private DistributedReplicantManager drm;
   private HATarget hatarget;
   private String proxyFamilyName;
   private String homeProxyFamilyName;
   private LoadBalancePolicy lbPolicy;
   private FamilyWrapper wrapper;
   private FamilyWrapper homeWrapper;
   private Object homeProxy;

   public void setRemoteBinding(RemoteBinding binding)
   {
      this.binding = binding;
   }

   protected Class[] getInterfaces()
   {
      StatefulContainer statefulContainer = (StatefulContainer) getContainer();
      RemoteHome remoteHome = (RemoteHome) statefulContainer.resolveAnnotation(RemoteHome.class);

      boolean bindTogether = false;

      if (remoteHome != null && bindHomeAndBusinessTogether(statefulContainer))
         bindTogether = true;

      // Obtain all remote interfaces
      List<Class<?>> remoteInterfaces = new ArrayList<Class<?>>();
      remoteInterfaces.addAll(Arrays.asList(ProxyFactoryHelper.getRemoteInterfaces(statefulContainer)));

      // Add JBossProxy
      remoteInterfaces.add(JBossProxy.class);
      remoteInterfaces.add(javax.ejb.EJBObject.class);
      
      // If binding along w/ home, add home
      if (bindTogether)
      {
         remoteInterfaces.add(remoteHome.value());
      }

      // Return
      return remoteInterfaces.toArray(new Class<?>[]{});
   }
   
   protected boolean bindHomeAndBusinessTogether(Container container)
   {
      return ProxyFactoryHelper.getHomeJndiName(container).equals(ProxyFactoryHelper.getRemoteJndiName(container));
   }

   protected void initializeJndiName()
   {
      jndiName = ProxyFactoryHelper.getRemoteJndiName(getContainer(), binding);
   }

   public void start() throws Exception
   {
      String clientBindUrl = ProxyFactoryHelper.getClientBindUrl(binding);
      locator = new InvokerLocator(clientBindUrl);
      Clustered clustered = (Clustered) ((Advisor)getContainer()).resolveAnnotation(Clustered.class);
      if (clustered == null) throw new RuntimeException("Could not find @Clustered annotation.  Cannot deploy.");
      String partitionName = ((EJBContainer) getContainer()).getPartitionName();
      proxyFamilyName = ((EJBContainer) getContainer()).getDeploymentQualifiedName() + locator.getProtocol() + partitionName;
      HAPartition partition = (HAPartition) getContainer().getInitialContext().lookup("/HAPartition/" + partitionName);
      hatarget = new HATarget(partition, proxyFamilyName, locator, HATarget.ENABLE_INVOCATIONS);
      ClusteringTargetsRepository.initTarget(proxyFamilyName, hatarget.getReplicants());
      ((SessionContainer) getContainer()).getClusterFamilies().put(proxyFamilyName, hatarget);
      if (clustered.loadBalancePolicy() == null || clustered.loadBalancePolicy().equals(LoadBalancePolicy.class))
      {
         lbPolicy = new FirstAvailable();
      }
      else
      {
         lbPolicy = (LoadBalancePolicy) clustered.loadBalancePolicy().newInstance();
      }
      wrapper = new FamilyWrapper(proxyFamilyName, hatarget.getReplicants());
      
      this.drm = partition.getDistributedReplicantManager();
      drm.registerListener(proxyFamilyName, this);
      
      super.start();
      
      Class[] interfaces = {ProxyFactory.class};
      String targetId = getTargetId();
      Object factoryProxy = Remoting.createPojiProxy(targetId, interfaces, ProxyFactoryHelper.getClientBindUrl(binding));
      try
      {
         Util.rebind(getContainer().getInitialContext(), jndiName + PROXY_FACTORY_NAME, factoryProxy);
      } catch (NamingException e)
      {
         NamingException namingException = new NamingException("Could not bind stateful cluster proxy with ejb name " + getContainer().getEjbName() + " into JNDI under jndiName: " + getContainer().getInitialContext().getNameInNamespace() + "/" + jndiName + PROXY_FACTORY_NAME);
         namingException.setRootCause(e);
         throw namingException;
      }
      assert !Dispatcher.singleton.isRegistered(targetId) : targetId + " is already registered";
      Dispatcher.singleton.registerTarget(targetId, this);

      createAndBindHomeProxy();
   }

   public Object createProxy()
   {
      return createProxy(null);
   }

   public Object createProxy(Object id)
   {
      try
      {
         Object containerId = getContainer().getObjectName().getCanonicalName();
         String stackName = "ClusteredStatefulSessionClientInterceptors";
         if (binding.interceptorStack() != null && !binding.interceptorStack().equals(""))
         {
            stackName = binding.interceptorStack();
         }
         AdviceStack stack = AspectManager.instance().getAdviceStack(stackName);
         String partitionName = ((EJBContainer) getContainer()).getPartitionName();
         Object[] args = null;
         if (id != null)
         {
            args = new Object[]{new StatefulClusteredProxy(getContainer(), stack.createInterceptors((Advisor) getContainer(), null), 
                  wrapper, lbPolicy, partitionName, id)};
         }
         else
         {
            args = new Object[]{new StatefulClusteredProxy(getContainer(), stack.createInterceptors((Advisor) getContainer(), null), 
                  wrapper, lbPolicy, partitionName)};            
         }
         
         return proxyConstructor.newInstance(args);
      }
      catch (InstantiationException e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
      catch (IllegalAccessException e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
      catch (IllegalArgumentException e)
      {
         throw new RuntimeException(e);  //To change body of catch statement use Options | File Templates.
      }
      catch (InvocationTargetException e)
      {
         throw new RuntimeException(e.getTargetException());  //To change body of catch statement use Options | File Templates.
      }
   }
   
   public void stop() throws Exception
   {
      Dispatcher.singleton.unregisterTarget(getTargetId());
      hatarget.destroy();
      drm.unregisterListener(proxyFamilyName, this);
      ((SessionContainer) getContainer()).getClusterFamilies().remove(proxyFamilyName);
      ((SessionContainer) getContainer()).getClusterFamilies().remove(homeProxyFamilyName);
      Util.unbind(getContainer().getInitialContext(), jndiName + PROXY_FACTORY_NAME);
      Util.unbind(getContainer().getInitialContext(), ProxyFactoryHelper.getHomeJndiName(getContainer()));
      super.stop();
   }
   
   protected StatefulHandleImpl getHandle()
   {
      StatefulHandleImpl handle = new StatefulHandleImpl();
      RemoteBinding remoteBinding = (RemoteBinding)((Advisor)getContainer()).resolveAnnotation(RemoteBinding.class);
      if (remoteBinding != null)
         handle.jndiName = remoteBinding.jndiBinding();
 
      return handle;
   }
   
   /**
    * @return unique name for this proxy factory
    */
   protected String getTargetId()
   {
      assert jndiName != null : "jndiName is null";      
      String partition = ((EJBContainer) getContainer()).getPartitionName();
      return jndiName + PROXY_FACTORY_NAME + "@" + partition;
   }
   
   public synchronized void replicantsChanged (String key, 
         List newReplicants, 
         int newReplicantsViewId)
   {
      try
      {
         // Update the FamilyClusterInfo with the new targets
         ArrayList targets = new ArrayList(newReplicants);
         wrapper.get().updateClusterInfo(targets, newReplicantsViewId);
         
         if (homeProxy != null)
         {
            if (log.isTraceEnabled())
            {
               log.trace("Updating home proxy for " + getContainer().getEjbName() + " with new replicants " + newReplicants + ", view=" + newReplicantsViewId);
            }

            homeWrapper.get().updateClusterInfo(targets, newReplicantsViewId);
            bindHomeProxy(homeProxy);
         }
      }
      catch (Exception e)
      {
         log.error(e);
      }
   }

   protected void createAndBindHomeProxy() throws Exception
   {
      EJBContainer container = (EJBContainer) getContainer();
      RemoteHome remoteHome = (RemoteHome) container.resolveAnnotation(RemoteHome.class);
      if (remoteHome != null && !bindHomeAndBusinessTogether(container))
      {
         Object homeProxy = createHomeProxy(remoteHome.value(), container);
         bindHomeProxy(homeProxy);
      }
   }

   public Object createHomeProxy(Class homeInterface, EJBContainer container)
   {
      String stackName = "ClusteredStatefulSessionClientInterceptors";
      if (binding.interceptorStack() != null && !binding.interceptorStack().equals(""))
      {
         stackName = binding.interceptorStack();
      }
      
      AdviceStack stack = AspectManager.instance().getAdviceStack(stackName);
      String partitionName = container.getPartitionName();

      homeProxyFamilyName = container.getDeploymentQualifiedName() + locator.getProtocol() + partitionName + StatelessClusterProxyFactory.HOME_FAMILY;
      homeWrapper = new FamilyWrapper(homeProxyFamilyName, hatarget.getReplicants());      
      ((SessionContainer) getContainer()).getClusterFamilies().put(homeProxyFamilyName, hatarget);
      
      if (log.isTraceEnabled())
      {
         log.trace("Creating home clustered proxy with family name " + homeProxyFamilyName + " and replicants " + hatarget.getReplicants());
      }

      /* Round Robin hardcoded for Home interface load balancing */
      StatefulHomeClusteredProxy proxy = new StatefulHomeClusteredProxy(getContainer(), 
            stack.createInterceptors((Advisor) getContainer(), null), homeWrapper, 
            new RoundRobin(), partitionName);

      Class[] interfaces = { homeInterface, EJBHome.class};
      homeProxy = java.lang.reflect.Proxy.newProxyInstance(getContainer()
            .getBeanClass().getClassLoader(), interfaces, proxy);
      
      return homeProxy;      
   }
   
   protected void bindHomeProxy(Object proxy) throws NamingException
   {
      log.debug("Binding home proxy for " + getContainer().getEjbName() + " in JNDI at " + ProxyFactoryHelper.getHomeJndiName(getContainer()));
      
      try 
      {
         Util.rebind(getContainer().getInitialContext(), ProxyFactoryHelper.getHomeJndiName(getContainer()), proxy);
      }
      catch (NamingException e)
      {
         NamingException namingException = new NamingException(
               "Could not bind stateful home proxy with ejb name "
                     + getContainer().getEjbName()
                     + " into JNDI under jndiName: "
                     + getContainer().getInitialContext().getNameInNamespace()
                     + "/" + ProxyFactoryHelper.getHomeJndiName(getContainer()));
         namingException.setRootCause(e);
         throw namingException;
      }      
   }
}
