/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.cluster.test;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import junit.framework.Test;

import org.apache.catalina.Context;
import org.jboss.cache.aop.PojoCache;
import org.jboss.logging.Logger;
import org.jboss.metadata.WebMetaData;
import org.jboss.test.JBossTestCase;
import org.jboss.test.cluster.web.notification.MockClusteredSessionNotificationPolicy;
import org.jboss.test.cluster.web.notification.MockHttpSessionAttributeListener;
import org.jboss.test.cluster.web.notification.MockHttpSessionBindingListener;
import org.jboss.test.cluster.web.notification.MockHttpSessionListener;
import org.jboss.test.cluster.web.util.BasicRequestHandler;
import org.jboss.test.cluster.web.util.CacheConfigTestSetup;
import org.jboss.test.cluster.web.util.InvalidateSessionRequestHandler;
import org.jboss.test.cluster.web.util.RemoveAttributesRequestHandler;
import org.jboss.test.cluster.web.util.SetAttributesRequestHandler;
import org.jboss.test.cluster.web.util.WebSessionTestUtil;
import org.jboss.web.tomcat.service.session.JBossCacheManager;

/**
 * Tests of handling of servlet spec notifications. 
 * 
 * @author Brian Stansberry
 */
public class ClusteredSessionNotificationPolicyTestCase extends JBossTestCase
{
   protected static PojoCache[] pojoCaches = new PojoCache[2];

   protected static long testId = System.currentTimeMillis();
   
   protected static boolean useBuddyRepl = Boolean.valueOf(System.getProperty("jbosstest.cluster.web.cache.br")).booleanValue();
   
   protected Logger log = Logger.getLogger(getClass());   
   
   protected Set<JBossCacheManager> managers = new HashSet<JBossCacheManager>();
   
   protected Map<String, Object> allAttributes;
   protected Map<String, Object> immutables;
   protected Map<String, Object> mutables;
   protected Map<String, Object> attributes;
   protected MockHttpSessionBindingListener attribute = new MockHttpSessionBindingListener();
   protected Map<String, Object> newAttributes;
   protected MockHttpSessionBindingListener newAttribute = new MockHttpSessionBindingListener();
   
   public ClusteredSessionNotificationPolicyTestCase(String name)
   {
      super(name);
   }
   
   public static Test suite() throws Exception
   {
      return CacheConfigTestSetup.getTestSetup(ClusteredSessionNotificationPolicyTestCase.class, pojoCaches, false, !useBuddyRepl, false);
   }

   
   @Override
   protected void setUp() throws Exception
   {
      super.setUp();
      
      attributes = new HashMap<String, Object>();
      attributes.put("KEY", attribute);
      attributes = Collections.unmodifiableMap(attributes);
      
      newAttributes = new HashMap<String, Object>();
      newAttributes.put("KEY", newAttribute);
      newAttributes = Collections.unmodifiableMap(newAttributes);
   }

   @Override
   protected void tearDown() throws Exception
   {
      super.tearDown();
      
      for (JBossCacheManager manager : managers)    
      {
         try
         {
            manager.stop();
         }
         catch (RuntimeException ignored)
         {
            log.debug("tearDown(): Caught exception cleaning up manager -- " + ignored.getLocalizedMessage()); 
         }
      }
      managers.clear();
      
      attribute.invocations.clear();
      newAttribute.invocations.clear();
   }
   
   protected int getReplicationGranularity()
   {
      return WebMetaData.REPLICATION_GRANULARITY_SESSION;
   }
   
   protected int getReplicationTrigger()
   {
      return WebMetaData.SESSION_INVALIDATE_SET_AND_NON_PRIMITIVE_GET;
   }
   
   public void testSessionLifecycleWithNotifications() throws Exception
   {
      log.info("++++ Starting testSessionLifecycleWithNotifications ++++");
      sessionLifecycleTest(true);
   }
   
   public void testSessionLifecycleWithoutNotifications() throws Exception
   {
      log.info("++++ Starting testSessionLifecycleWithoutNotifications ++++");
      sessionLifecycleTest(false);
   }
   
   private void sessionLifecycleTest(boolean notify) throws Exception
   {
      String warname = String.valueOf(++testId);
      
      // A war with a maxInactive of 30 mins maxUnreplicated of 0
      JBossCacheManager[] mgrs = getCacheManagers(warname, 1800, 0);
      JBossCacheManager jbcm0 = mgrs[0];
      JBossCacheManager jbcm1 = mgrs[1];
      
      assertTrue(jbcm0.getNotificationPolicy() instanceof MockClusteredSessionNotificationPolicy);
      MockClusteredSessionNotificationPolicy mcsnp0 = (MockClusteredSessionNotificationPolicy) jbcm0.getNotificationPolicy();
      assertNotNull("capability set", mcsnp0.getClusteredSessionExpriationNotificationCapability());
      mcsnp0.setResponse(notify);
      
      assertTrue(jbcm1.getNotificationPolicy() instanceof MockClusteredSessionNotificationPolicy);
      MockClusteredSessionNotificationPolicy mcsnp1 = (MockClusteredSessionNotificationPolicy) jbcm1.getNotificationPolicy();
      assertNotNull("capability set", mcsnp1.getClusteredSessionExpriationNotificationCapability());
      mcsnp1.setResponse(notify);
      
      MockHttpSessionListener hsl0 = new MockHttpSessionListener();
      MockHttpSessionAttributeListener hsal0 = new MockHttpSessionAttributeListener();      
      Context ctx = (Context) jbcm0.getContainer();
      ctx.setApplicationLifecycleListeners(new Object[]{ hsl0 });  
      ctx.setApplicationEventListeners(new Object[]{ hsal0 });  
      
      MockHttpSessionListener hsl1 = new MockHttpSessionListener();
      MockHttpSessionAttributeListener hsal1 = new MockHttpSessionAttributeListener();      
      ctx = (Context) jbcm1.getContainer();
      ctx.setApplicationLifecycleListeners(new Object[]{ hsl1 });  
      ctx.setApplicationEventListeners(new Object[]{ hsal1 }); 
      
      // Initial request
      SetAttributesRequestHandler setHandler = new SetAttributesRequestHandler(attributes, false);
      WebSessionTestUtil.invokeRequest(jbcm0, setHandler, null);
      
      validateNewSession(setHandler);
      String sessionId = setHandler.getSessionId();
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl0.invocations.size());
         assertEquals(MockHttpSessionListener.Type.CREATED, hsl0.invocations.get(0));
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.ADDED, hsal0.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.BOUND, attribute.invocations.get(0));
         
         validateNoNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ newAttribute });
         clearNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ attribute });
      }
      
      // Modify attribute request
      setHandler = new SetAttributesRequestHandler(newAttributes, false);
      WebSessionTestUtil.invokeRequest(jbcm0, setHandler, sessionId);
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REPLACED, hsal0.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.UNBOUND, attribute.invocations.get(0));
         assertEquals(1, newAttribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.BOUND, newAttribute.invocations.get(0));
         
         validateNoNotifications(hsl0, null, hsl1, hsal1, null);
         clearNotifications(null, hsal0, null, null, new MockHttpSessionBindingListener[]{ attribute, newAttribute });
      }
      
      // Remove attribute request
      RemoveAttributesRequestHandler removeHandler = new RemoveAttributesRequestHandler(newAttributes.keySet(), false);
      WebSessionTestUtil.invokeRequest(jbcm0, removeHandler, sessionId);
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REMOVED, hsal0.invocations.get(0));
         assertEquals(1, newAttribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.UNBOUND, newAttribute.invocations.get(0));
         
         validateNoNotifications(hsl0, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ attribute });
         clearNotifications(null, hsal0, null, null, new MockHttpSessionBindingListener[]{ newAttribute });
      }
      
      // Failover request
      setHandler = new SetAttributesRequestHandler(attributes, false);
      WebSessionTestUtil.invokeRequest(jbcm1, setHandler, sessionId);
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl1.invocations.size());
         assertEquals(MockHttpSessionListener.Type.CREATED, hsl1.invocations.get(0));
         assertEquals(1, hsal1.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.ADDED, hsal1.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.BOUND, attribute.invocations.get(0));
         
         validateNoNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ newAttribute });
         clearNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ attribute });
      }
      
      // Invalidate session request
      InvalidateSessionRequestHandler invalidateHandler = new InvalidateSessionRequestHandler(attributes.keySet(), false);
      WebSessionTestUtil.invokeRequest(jbcm1, invalidateHandler, sessionId);
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl1.invocations.size());
         assertEquals(MockHttpSessionListener.Type.DESTROYED, hsl1.invocations.get(0));
         assertEquals(1, hsal1.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REMOVED, hsal1.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.UNBOUND, attribute.invocations.get(0));
         
         validateNoNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ newAttribute });
         clearNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ attribute });
      }
   }
   
   public void testSessionExpirationWithNotifications() throws Exception
   {
      log.info("++++ Starting testSessionExpirationWithNotifications ++++");
      sessionExpirationTest(true);
   }
   
   public void testSessionExpirationWithoutNotifications() throws Exception
   {
      log.info("++++ Starting testSessionExpirationWithoutNotifications ++++");
      sessionExpirationTest(false);
   }
   
   private void sessionExpirationTest(boolean notify) throws Exception
   {
      String warname = String.valueOf(++testId);
      
      // A war with a maxInactive of 3 secs and a maxUnreplicated of 0
      JBossCacheManager[] mgrs = getCacheManagers(warname, 2, 0);
      JBossCacheManager jbcm0 = mgrs[0];
      JBossCacheManager jbcm1 = mgrs[1];
      
      assertTrue(jbcm0.getNotificationPolicy() instanceof MockClusteredSessionNotificationPolicy);
      MockClusteredSessionNotificationPolicy mcsnp0 = (MockClusteredSessionNotificationPolicy) jbcm0.getNotificationPolicy();
      assertNotNull("capability set", mcsnp0.getClusteredSessionExpriationNotificationCapability());
      mcsnp0.setResponse(notify);
      
      assertTrue(jbcm1.getNotificationPolicy() instanceof MockClusteredSessionNotificationPolicy);
      MockClusteredSessionNotificationPolicy mcsnp1 = (MockClusteredSessionNotificationPolicy) jbcm1.getNotificationPolicy();
      assertNotNull("capability set", mcsnp1.getClusteredSessionExpriationNotificationCapability());
      mcsnp1.setResponse(notify);
      
      MockHttpSessionListener hsl0 = new MockHttpSessionListener();
      MockHttpSessionAttributeListener hsal0 = new MockHttpSessionAttributeListener();      
      Context ctx = (Context) jbcm0.getContainer();
      ctx.setApplicationLifecycleListeners(new Object[]{ hsl0 });  
      ctx.setApplicationEventListeners(new Object[]{ hsal0 }); 
      
      MockHttpSessionListener hsl1 = new MockHttpSessionListener();
      MockHttpSessionAttributeListener hsal1 = new MockHttpSessionAttributeListener();      
      ctx = (Context) jbcm1.getContainer();
      ctx.setApplicationLifecycleListeners(new Object[]{ hsl1 });  
      ctx.setApplicationEventListeners(new Object[]{ hsal1 }); 
      
      // Initial request
      SetAttributesRequestHandler setHandler = new SetAttributesRequestHandler(attributes, false);
      WebSessionTestUtil.invokeRequest(jbcm0, setHandler, null);
      
      validateNewSession(setHandler);
      
      String sessionId = setHandler.getSessionId();
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl0.invocations.size());
         assertEquals(MockHttpSessionListener.Type.CREATED, hsl0.invocations.get(0));
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.ADDED, hsal0.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.BOUND, attribute.invocations.get(0));
         
         validateNoNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ newAttribute });
         clearNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ attribute });         
      }
      
      // Failover request
      setHandler = new SetAttributesRequestHandler(newAttributes, false);
      WebSessionTestUtil.invokeRequest(jbcm1, setHandler, sessionId);
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl1.invocations.size());
         assertEquals(MockHttpSessionListener.Type.CREATED, hsl1.invocations.get(0));
         assertEquals(1, hsal1.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REPLACED, hsal1.invocations.get(0));
         assertEquals(1, newAttribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.BOUND, newAttribute.invocations.get(0));
         
         validateNoNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ attribute });
         clearNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ newAttribute });         
      }
      
      Thread.sleep(2100);
      
      jbcm0.backgroundProcess();
      jbcm1.backgroundProcess();
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl0.invocations.size());
         assertEquals(MockHttpSessionListener.Type.DESTROYED, hsl0.invocations.get(0));
         assertEquals(1, hsl1.invocations.size());
         assertEquals(MockHttpSessionListener.Type.DESTROYED, hsl1.invocations.get(0));
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REMOVED, hsal0.invocations.get(0));
         assertEquals(1, hsal1.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REMOVED, hsal1.invocations.get(0));
         assertEquals(1, newAttribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.UNBOUND, newAttribute.invocations.get(0));
         
         validateNoNotifications(null, null, null, null, new MockHttpSessionBindingListener[]{ attribute });
         clearNotifications(hsl0, hsal0, hsl1, hsal1, new MockHttpSessionBindingListener[]{ newAttribute });         
      }
   }
   
   public void testUndeployWithNotifications() throws Exception
   {
      log.info("++++ Starting testUndeployWithNotifications ++++");
      undeployTest(true);
   }
   
   public void testUndeployWithoutNotifications() throws Exception
   {
      log.info("++++ Starting testUndeployWithoutNotifications ++++");
      undeployTest(false);
   }
   
   private void undeployTest(boolean notify) throws Exception
   {
      String warname = String.valueOf(++testId);
      
      // A war with a maxInactive of 3 secs and a maxUnreplicated of 0
      JBossCacheManager[] mgrs = getCacheManagers(warname, 2, 0);
      JBossCacheManager jbcm0 = mgrs[0];
      JBossCacheManager jbcm1 = mgrs[1];
      
      assertTrue(jbcm0.getNotificationPolicy() instanceof MockClusteredSessionNotificationPolicy);
      MockClusteredSessionNotificationPolicy mcsnp0 = (MockClusteredSessionNotificationPolicy) jbcm0.getNotificationPolicy();
      assertNotNull("capability set", mcsnp0.getClusteredSessionExpriationNotificationCapability());
      mcsnp0.setResponse(notify);
      
      assertTrue(jbcm1.getNotificationPolicy() instanceof MockClusteredSessionNotificationPolicy);
      MockClusteredSessionNotificationPolicy mcsnp1 = (MockClusteredSessionNotificationPolicy) jbcm1.getNotificationPolicy();
      assertNotNull("capability set", mcsnp1.getClusteredSessionExpriationNotificationCapability());
      mcsnp1.setResponse(notify);
      
      MockHttpSessionListener hsl0 = new MockHttpSessionListener();
      MockHttpSessionAttributeListener hsal0 = new MockHttpSessionAttributeListener();      
      Context ctx = (Context) jbcm0.getContainer();
      ctx.setApplicationLifecycleListeners(new Object[]{ hsl0 });  
      ctx.setApplicationEventListeners(new Object[]{ hsal0 });  
      
      MockHttpSessionListener hsl1 = new MockHttpSessionListener();
      MockHttpSessionAttributeListener hsal1 = new MockHttpSessionAttributeListener();      
      ctx = (Context) jbcm1.getContainer();
      ctx.setApplicationLifecycleListeners(new Object[]{ hsl1 });  
      ctx.setApplicationEventListeners(new Object[]{ hsal1 }); 
      
      // Initial request
      SetAttributesRequestHandler setHandler = new SetAttributesRequestHandler(attributes, false);
      WebSessionTestUtil.invokeRequest(jbcm0, setHandler, null);
      
      validateNewSession(setHandler);
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl0.invocations.size());
         assertEquals(MockHttpSessionListener.Type.CREATED, hsl0.invocations.get(0));
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.ADDED, hsal0.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.BOUND, attribute.invocations.get(0));
         
         validateNoNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ newAttribute });
         clearNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ attribute });
         
      }
      
      jbcm0.stop();
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         assertEquals(1, hsl0.invocations.size());
         assertEquals(MockHttpSessionListener.Type.DESTROYED, hsl0.invocations.get(0));
         assertEquals(1, hsal0.invocations.size());
         assertEquals(MockHttpSessionAttributeListener.Type.REMOVED, hsal0.invocations.get(0));
         assertEquals(1, attribute.invocations.size());
         assertEquals(MockHttpSessionBindingListener.Type.UNBOUND, attribute.invocations.get(0));
         
         validateNoNotifications(null, null, hsl1, hsal1, new MockHttpSessionBindingListener[]{ newAttribute });
         clearNotifications(hsl0, hsal0, null, null, new MockHttpSessionBindingListener[]{ attribute });
         
         
      }
      
      jbcm1.stop();
      
      if (!notify)
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
      else
      {
         validateNoNotifications(hsl0, hsal0, hsl1, hsal1);
      }
   }
   
   private void validateNoNotifications(MockHttpSessionListener hsl0, MockHttpSessionAttributeListener hsal0,
         MockHttpSessionListener hsl1, MockHttpSessionAttributeListener hsal1)
   {
      validateNoNotifications(hsl0, hsal0, hsl1, hsal1, new MockHttpSessionBindingListener[]{ attribute, newAttribute});
   }
   
   private void validateNoNotifications(MockHttpSessionListener hsl0, MockHttpSessionAttributeListener hsal0,
         MockHttpSessionListener hsl1, MockHttpSessionAttributeListener hsal1, MockHttpSessionBindingListener[] hsbls)
   {
      if (hsl0 != null)
      {
         assertEquals(0, hsl0.invocations.size());
      }
      if (hsal0 != null)
      {
         assertEquals(0, hsal0.invocations.size());
      }
      if (hsl1 != null)
      {
         assertEquals(0, hsl1.invocations.size());
      }
      if (hsal1 != null)
      {
         assertEquals(0, hsal1.invocations.size());
      }
      
      if (hsbls != null)
      {
         for (int i = 0; i < hsbls.length; i++)
         {
            assertEquals("Binding listener " + i + "  has no notifications", 0, hsbls[i].invocations.size());
         }
      }
      
      clearNotifications(hsl0, hsal0, hsl1, hsal1, hsbls);
   }
   
   private void clearNotifications(MockHttpSessionListener hsl0, MockHttpSessionAttributeListener hsal0,
         MockHttpSessionListener hsl1, MockHttpSessionAttributeListener hsal1)
   {
      clearNotifications(hsl0, hsal0, hsl1, hsal1, new MockHttpSessionBindingListener[]{ attribute, newAttribute});
   }
   
   private void clearNotifications(MockHttpSessionListener hsl0, MockHttpSessionAttributeListener hsal0,
         MockHttpSessionListener hsl1, MockHttpSessionAttributeListener hsal1, MockHttpSessionBindingListener[] hsbls)
   {      

      if (hsl0 != null)
      {
         hsl0.invocations.clear();
      }
      if (hsal0 != null)
      {
         hsal0.invocations.clear();
      }
      if (hsl1 != null)
      {
         hsl1.invocations.clear();
      }
      if (hsal1 != null)
      {
         hsal1.invocations.clear();
      }
      
      if (hsbls != null)
      {
         for (int i = 0; i < hsbls.length; i++)
         {
            hsbls[i].invocations.clear();
         }
      }
   }
   
   protected JBossCacheManager[] getCacheManagers(String warname, int maxInactive, int maxUnreplicated)
      throws Exception
   {
      JBossCacheManager jbcm0 = WebSessionTestUtil.createManager(warname, maxInactive, pojoCaches[0], null);
      WebMetaData metadata = WebSessionTestUtil.getWebMetaData(getReplicationGranularity(), getReplicationTrigger(), true, maxUnreplicated);
      metadata.setClusteredSessionNotificationPolicy(MockClusteredSessionNotificationPolicy.class.getName());
      jbcm0.init(warname, metadata, false, true);
      this.managers.add(jbcm0);
      jbcm0.start();
      
      JBossCacheManager jbcm1 = WebSessionTestUtil.createManager(warname, maxInactive, pojoCaches[1], null);
      metadata = WebSessionTestUtil.getWebMetaData(getReplicationGranularity(), getReplicationTrigger(), true, maxUnreplicated);
      metadata.setClusteredSessionNotificationPolicy(MockClusteredSessionNotificationPolicy.class.getName());
      jbcm1.init(warname, metadata, false, true);
      this.managers.add(jbcm1);
      jbcm1.start();
      
      return new JBossCacheManager[]{jbcm0, jbcm1};
   }
   
   protected void validateExpectedAttributes(Map<String, Object> expected, BasicRequestHandler handler)
   {
      assertFalse(handler.isNewSession());
      
      if (handler.isCheckAttributeNames())
      {
         assertEquals(expected.size(), handler.getAttributeNames().size());
      }
      Map<String, Object> checked = handler.getCheckedAttributes();
      assertEquals(expected.size(), checked.size());
      for (Map.Entry<String, Object> entry : checked.entrySet())
         assertEquals(entry.getKey(), expected.get(entry.getKey()), entry.getValue());
      
   }
   
   protected void validateNewSession(BasicRequestHandler handler)
   {
      assertTrue(handler.isNewSession());
      assertEquals(handler.getCreationTime(), handler.getLastAccessedTime());
      if (handler.isCheckAttributeNames())
      {
         assertEquals(0, handler.getAttributeNames().size());
      }
      Map<String, Object> checked = handler.getCheckedAttributes();
      for (Map.Entry<String, Object> entry : checked.entrySet())
         assertNull(entry.getKey(), entry.getValue());
   }
   

}
