/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.stateless;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.ejb.EJBHome;
import javax.ejb.RemoteHome;
import javax.naming.NamingException;

import org.jboss.annotation.ejb.Clustered;
import org.jboss.annotation.ejb.RemoteBinding;
import org.jboss.aop.Advisor;
import org.jboss.aop.AspectManager;
import org.jboss.aop.advice.AdviceStack;
import org.jboss.aspects.remoting.FamilyWrapper;
import org.jboss.ejb3.EJBContainer;
import org.jboss.ejb3.JBossProxy;
import org.jboss.ejb3.ProxyFactoryHelper;
import org.jboss.ejb3.SessionContainer;
import org.jboss.ejb3.remoting.RemoteProxyFactory;
import org.jboss.ha.framework.interfaces.ClusteringTargetsRepository;
import org.jboss.ha.framework.interfaces.DistributedReplicantManager;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.ha.framework.interfaces.LoadBalancePolicy;
import org.jboss.ha.framework.interfaces.RandomRobin;
import org.jboss.ha.framework.interfaces.RoundRobin;
import org.jboss.ha.framework.server.HATarget;
import org.jboss.logging.Logger;
import org.jboss.naming.Util;
import org.jboss.remoting.InvokerLocator;


/**
 * Comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 * @version $Revision: 74347 $
 */
public class StatelessClusterProxyFactory extends BaseStatelessProxyFactory  
   implements RemoteProxyFactory, DistributedReplicantManager.ReplicantListener
{
   private static final Logger log = Logger.getLogger(StatelessClusterProxyFactory.class);
   
   public static final String HOME_FAMILY = "/Home";

   private RemoteBinding binding;
   private InvokerLocator locator;
   private DistributedReplicantManager drm;
   private HATarget hatarget;
   private String proxyFamilyName;
   private String homeProxyFamilyName;
   private LoadBalancePolicy lbPolicy;
   private FamilyWrapper beanWrapper;
   private FamilyWrapper homeWrapper;
   private Object beanProxy;
   private Object homeProxy;

   public void setRemoteBinding(RemoteBinding binding)
   {
      this.binding = binding;
   }

   protected Class[] getInterfaces()
   {
      StatelessContainer statelessContainer = (StatelessContainer) getContainer();
      RemoteHome remoteHome = (RemoteHome) statelessContainer.resolveAnnotation(RemoteHome.class);

      boolean bindTogether = false;

      if (remoteHome != null && bindHomeAndBusinessTogether(statelessContainer))
         bindTogether = true;

      // Obtain all remote interfaces
      List<Class<?>> remoteInterfaces = new ArrayList<Class<?>>();
      remoteInterfaces.addAll(Arrays.asList(ProxyFactoryHelper.getRemoteInterfaces(statelessContainer)));

      // Ensure remote interfaces defined
      if (remoteInterfaces.size() > 0)
      {
         // Add JBossProxy
         remoteInterfaces.add(JBossProxy.class);
         remoteInterfaces.add(javax.ejb.EJBObject.class);

         // If binding along w/ home, add home
         if (bindTogether)
         {
            remoteInterfaces.add(remoteHome.value());
         }
      }
      else
      {
         // No remote interfaces defined, log warning
         log.warn("[EJBTHREE-933] NPE when deploying web service beans");
      }

      // Return
      return remoteInterfaces.toArray(new Class<?>[]
      {});
   }
   
   protected boolean bindHomeAndBusinessTogether(EJBContainer container)
   {
      return ProxyFactoryHelper.getHomeJndiName(container).equals(ProxyFactoryHelper.getRemoteJndiName(container));
   }

   protected void initializeJndiName()
   {
      jndiName = ProxyFactoryHelper.getRemoteJndiName(getContainer(), binding);
   }

   public void start() throws Exception
   {
      String clientBindUrl = ProxyFactoryHelper.getClientBindUrl(binding);
      locator = new InvokerLocator(clientBindUrl);
      Clustered clustered = (Clustered) ((Advisor)getContainer()).resolveAnnotation(Clustered.class);
      if (clustered == null) throw new RuntimeException("Could not find @Clustered annotation.  Cannot deploy.");
      String partitionName = ((EJBContainer) getContainer()).getPartitionName();      
      proxyFamilyName = ((EJBContainer) getContainer()).getDeploymentQualifiedName() + locator.getProtocol() + partitionName;
      HAPartition partition = (HAPartition) getContainer().getInitialContext().lookup("/HAPartition/" + partitionName);
      hatarget = new HATarget(partition, proxyFamilyName, locator, HATarget.ENABLE_INVOCATIONS);
      ClusteringTargetsRepository.initTarget(proxyFamilyName, hatarget.getReplicants());
      ((SessionContainer) getContainer()).getClusterFamilies().put(proxyFamilyName, hatarget);
      if (clustered.loadBalancePolicy() == null || clustered.loadBalancePolicy().equals(LoadBalancePolicy.class))
      {
         lbPolicy = new RandomRobin();
      }
      else
      {
         lbPolicy = (LoadBalancePolicy) clustered.loadBalancePolicy().newInstance();
      }
      beanWrapper = new FamilyWrapper(proxyFamilyName, hatarget.getReplicants());
      
      this.drm = partition.getDistributedReplicantManager();
      drm.registerListener(proxyFamilyName, this);
      
      super.start();
      
      createAndBindHomeProxy();
   }

   public void stop() throws Exception
   {
      super.stop();
      beanProxy = null;
      homeProxy = null;
      hatarget.destroy();
      drm.unregisterListener(proxyFamilyName, this);
      ((SessionContainer) getContainer()).getClusterFamilies().remove(proxyFamilyName);
      ((SessionContainer) getContainer()).getClusterFamilies().remove(homeProxyFamilyName);
   }

   public Object createProxy()
   {
      Object containerId = getContainer().getObjectName().getCanonicalName();
      String stackName = "ClusteredStatelessSessionClientInterceptors";
      if (binding.interceptorStack() != null && !binding.interceptorStack().equals(""))
      {
         stackName = binding.interceptorStack();
      }
      AdviceStack stack = AspectManager.instance().getAdviceStack(stackName);
      
      if (log.isTraceEnabled())
      {
         log.trace("Advice stack for " + stackName + " is " + stack);
      }
      
      String partitionName = ((StatelessContainer) getContainer()).getPartitionName();
      
      beanProxy = constructProxy(new StatelessClusteredProxy(getContainer(), stack.createInterceptors((Advisor) getContainer(), null), 
            beanWrapper, lbPolicy, partitionName));
      return beanProxy;
   }

   protected StatelessHandleImpl getHandle()
   {
      StatelessHandleImpl handle = new StatelessHandleImpl();
      RemoteBinding remoteBinding = (RemoteBinding)((Advisor)getContainer()).resolveAnnotation(RemoteBinding.class);
      if (remoteBinding != null)
         handle.jndiName = remoteBinding.jndiBinding();
 
      return handle;
   }
   
   public synchronized void replicantsChanged (String key, 
         List newReplicants, 
         int newReplicantsViewId)
   {
      try
      {
         // Update the FamilyClusterInfo with the new targets
         ArrayList targets = new ArrayList(newReplicants);
                  
         // Rebind the proxy as the old one has been serialized
         if (beanProxy != null)
         {
            if (log.isTraceEnabled())
            {
               log.trace("Updating bean proxy for " + getContainer().getEjbName() + " with new replicants " + newReplicants + ", view=" + newReplicantsViewId);
            }
            
            beanWrapper.get().updateClusterInfo(targets, newReplicantsViewId);
            bindProxy(beanProxy);
         }
                     
         // Rebind the proxy as the old one has been serialized
         if (homeProxy != null)
         {
            if (log.isTraceEnabled())
            {
               log.trace("Updating home proxy for " + getContainer().getEjbName() + " with new replicants " + newReplicants + ", view=" + newReplicantsViewId);
            }
            
            homeWrapper.get().updateClusterInfo(targets, newReplicantsViewId);
            bindHomeProxy(homeProxy);
         }
            
      }
      catch (Exception e)
      {
         log.error(e);
      }
   }
   
   protected void createAndBindHomeProxy() throws Exception
   {
      EJBContainer container = (EJBContainer) getContainer();
      RemoteHome remoteHome = (RemoteHome) container.resolveAnnotation(RemoteHome.class);
      if (remoteHome != null && !bindHomeAndBusinessTogether(container))
      {
         Object homeProxy = createHomeProxy(remoteHome.value());
         bindHomeProxy(homeProxy);
      }
   }
   
   public Object createHomeProxy(Class homeInterface)
   {
      String stackName = "ClusteredStatelessSessionClientInterceptors";
      if (binding.interceptorStack() != null && !binding.interceptorStack().equals(""))
      {
         stackName = binding.interceptorStack();
      }
      
      AdviceStack stack = AspectManager.instance().getAdviceStack(stackName);
      String partitionName = ((StatelessContainer) getContainer()).getPartitionName();

      homeProxyFamilyName = ((StatelessContainer) getContainer()).getDeploymentQualifiedName() + locator.getProtocol() + partitionName + HOME_FAMILY;
      homeWrapper = new FamilyWrapper(homeProxyFamilyName, hatarget.getReplicants());      
      ((SessionContainer) getContainer()).getClusterFamilies().put(homeProxyFamilyName, hatarget);
      
      if (log.isTraceEnabled())
      {
         log.trace("Creating home clustered proxy with family name " + homeProxyFamilyName + " and replicants " + hatarget.getReplicants());
      }

      /* Round Robin hardcoded for Home interface load balancing */
      StatelessHomeClusteredProxy proxy = new StatelessHomeClusteredProxy(getContainer(), 
            stack.createInterceptors((Advisor) getContainer(), null), homeWrapper, 
            new RoundRobin(), partitionName);

      Class[] interfaces = { homeInterface, EJBHome.class};
      homeProxy = java.lang.reflect.Proxy.newProxyInstance(getContainer()
            .getBeanClass().getClassLoader(), interfaces, proxy);
      
      return homeProxy;      
   }
   
   protected void bindHomeProxy(Object proxy) throws NamingException
   {
      log.debug("Binding home proxy for " + getContainer().getEjbName() + " in JNDI at " + ProxyFactoryHelper.getHomeJndiName(getContainer()));
      
      try 
      {
         Util.rebind(getContainer().getInitialContext(), ProxyFactoryHelper.getHomeJndiName(getContainer()), proxy);
      }
      catch (NamingException e)
      {
         NamingException namingException = new NamingException(
               "Could not bind stateless home proxy with ejb name "
                     + getContainer().getEjbName()
                     + " into JNDI under jndiName: "
                     + getContainer().getInitialContext().getNameInNamespace()
                     + "/" + ProxyFactoryHelper.getHomeJndiName(getContainer()));
         namingException.setRootCause(e);
         throw namingException;
      }      
   }
}
