/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ws.integration.jboss42;

// $Id: DeployerInterceptorJSE.java 2598 2007-03-12 10:11:14Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;

import javax.jws.WebService;
import javax.xml.ws.WebServiceProvider;

import org.jboss.deployment.DeploymentInfo;
import org.jboss.metadata.WebMetaData;
import org.jboss.ws.WSException;
import org.jboss.ws.core.server.AbstractServiceEndpointPublisher;
import org.jboss.ws.core.server.JAXWSDeployment;
import org.jboss.ws.core.server.UnifiedDeploymentInfo;
import org.jboss.ws.core.server.AbstractServiceEndpointPublisher.RewriteResults;
import org.jboss.ws.core.server.UnifiedDeploymentInfo.DeploymentType;
import org.jboss.ws.integration.ResourceLoaderAdapter;
import org.jboss.ws.integration.UnifiedVirtualFile;
import org.jboss.ws.metadata.builder.jaxrpc.JAXRPCDeployment;
import org.jboss.ws.metadata.umdm.EndpointMetaData;
import org.jboss.ws.metadata.umdm.ServerEndpointMetaData;
import org.jboss.ws.metadata.umdm.ServiceMetaData;
import org.jboss.ws.metadata.umdm.UnifiedMetaData;

/**
 * A deployer service that manages WS4EE compliant Web Services for WAR
 *
 * @author Thomas.Diesler@jboss.org
 * @since 15-Jan-2005
 */
public class DeployerInterceptorJSE extends DeployerInterceptor implements DeployerInterceptorJSEMBean
{
   /** Return true if the deployment is a web service endpoint
    */
   protected boolean isWebserviceDeployment(DeploymentInfo di)
   {
      WebMetaData webMetaData = (WebMetaData)di.metaData;
      boolean isWebserviceDeployment = webMetaData.isWebServiceDeployment();

      // Check if we have a webservices.xml descriptor
      if (isWebserviceDeployment == false)
      {
         if (getWebservicesFile(di) != null)
         {
            di.context.put("UnifiedDeploymentInfo.Type", UnifiedDeploymentInfo.DeploymentType.JAXRPC_JSE);
            isWebserviceDeployment = true;
         }
      }

      // Check if the web.xml contains annotated endpoint impl
      if (isWebserviceDeployment == false)
      {
         Map servletClassMap = webMetaData.getServletClassMap();
         Iterator<String> it = servletClassMap.values().iterator();
         while (it.hasNext() && isWebserviceDeployment == false)
         {
            String servletClassName = it.next();
            try
            {
               Class servletClass = di.annotationsCl.loadClass(servletClassName);
               if (servletClass.isAnnotationPresent(WebService.class) || servletClass.isAnnotationPresent(WebServiceProvider.class))
               {
                  di.context.put("UnifiedDeploymentInfo.Type", UnifiedDeploymentInfo.DeploymentType.JAXWS_JSE);
                  isWebserviceDeployment = true;
               }
            }
            catch (ClassNotFoundException ex)
            {
               log.warn("Cannot load servlet class: " + servletClassName);
            }
         }
      }

      webMetaData.setWebServiceDeployment(isWebserviceDeployment);
      return isWebserviceDeployment;
   }

   protected UnifiedDeploymentInfo createUnifiedDeploymentInfo(DeploymentInfo di) throws Exception
   {
      UnifiedDeploymentInfo udi;
      
      DeploymentType type = (DeploymentType)di.context.get("UnifiedDeploymentInfo.Type");
      if (type == UnifiedDeploymentInfo.DeploymentType.JAXRPC_JSE)
      {
         UnifiedVirtualFile vfsWebservices = getWebservicesFile(di);
         udi = new JAXRPCDeployment(UnifiedDeploymentInfo.DeploymentType.JAXRPC_JSE, vfsWebservices);
         DeploymentInfoAdapterFactory.newInstance().buildDeploymentInfo(udi, di);
      }
      else if (type == UnifiedDeploymentInfo.DeploymentType.JAXWS_JSE)
      {
         udi = new JAXWSDeployment(UnifiedDeploymentInfo.DeploymentType.JAXWS_JSE);
         DeploymentInfoAdapterFactory.newInstance().buildDeploymentInfo(udi, di);
      }
      else 
      {
         throw new WSException("Unexpected type:  " + type);
      }
      
      return udi;
   }

   /** Overwrite to create the webservice
    *
    * This implemantation modifies the servlet entries in web.xml
    */
   protected void createServiceEndpoint(DeploymentInfo di) throws Exception
   {
      super.createServiceEndpoint(di);
      UnifiedDeploymentInfo udi = getUnifiedDeploymentInfo(di);
      if (udi != null)
      {
         AbstractServiceEndpointPublisher endpointPublisher = getServiceEndpointPublisher();
         RewriteResults results = endpointPublisher.rewriteWebXml(udi);
         updateServiceEndpointTargetBeans(udi, results);
      }
   }
   
   private void updateServiceEndpointTargetBeans(UnifiedDeploymentInfo udi, RewriteResults results)
   {
      UnifiedMetaData wsMetaData = getServiceEndpointDeployer().getUnifiedMetaData(udi);
      Map<String, String> sepTargetMap = results.sepTargetMap;

      for (ServiceMetaData serviceMetaData : wsMetaData.getServices())
      {
         for (EndpointMetaData epMetaData : serviceMetaData.getEndpoints())
         {
            ServerEndpointMetaData sepMetaData = (ServerEndpointMetaData)epMetaData;
            String targetBean = sepTargetMap.get(sepMetaData.getLinkName());
            sepMetaData.setServiceEndpointImplName(targetBean);
         }
      }
   }
     
   private UnifiedVirtualFile getWebservicesFile(DeploymentInfo di)
   {
      UnifiedVirtualFile vfsRoot = new ResourceLoaderAdapter(di.localCl);
      try
      {
         return vfsRoot.findChild("WEB-INF/webservices.xml");
      }
      catch (IOException e)
      {
         return null;
      }
   }
}
