/*******************************************************************************
 * Copyright (c) 2015-2016 Red Hat, Inc.
 * Distributed under license by Red Hat, Inc. All rights reserved.
 * This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Red Hat, Inc. - initial API and implementation
 ******************************************************************************/
package org.jboss.tools.openshift.test.ui.application;

import static org.assertj.core.api.Assertions.assertThat;
import static org.jboss.tools.openshift.test.util.ResourceMocks.createObservableTreeItems;
import static org.jboss.tools.openshift.test.util.ResourceMocks.createResources;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.jboss.tools.openshift.internal.ui.treeitem.ObservableTreeItem;
import org.jboss.tools.openshift.internal.ui.wizard.common.IResourceLabelsPageModel;
import org.jboss.tools.openshift.internal.ui.wizard.newapp.NewApplicationWizardModel;
import org.jboss.tools.openshift.test.util.ResourceMocks;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import com.openshift.restclient.IResourceFactory;
import com.openshift.restclient.model.IProject;
import com.openshift.restclient.model.template.IParameter;
import com.openshift.restclient.model.template.ITemplate;
/**
 * @author jeff.cantrill
 */
@RunWith(MockitoJUnitRunner.class)
public class NewApplicationWizardModelTest {

	private TestableNewApplicationWizardModel model;
	@Mock
	private ITemplate template;
	@Mock
	private IProject project;
	@Mock
	private IResourceFactory factory;
	private List<ObservableTreeItem> projectItems;
	
	@Before
	public void setup() throws Exception {
		when(project.getName()).thenReturn(String.valueOf(System.currentTimeMillis()));
		createProjectTemplateItems();
		TestableNewApplicationWizardModel model = new TestableNewApplicationWizardModel();
		this.projectItems = createProjectTemplateItems();
		model.setProjectItems(projectItems);
		model.setProject(project);
		model.setResourceFactory(factory);

		this.model = spy(model);
		doReturn(mock(InputStream.class)).when(this.model).createInputStream(anyString());
	}

	/**
	 * Creates a tree of items:
	 * 
	 *  - project1
	 *     - template1
	 *  - project2
	 *     - template2
	 *     - template3
	 *  - project3
	 *     - template4
	 *     - template5
	 *     - template6
	 * @return 
	 */
	private List<ObservableTreeItem> createProjectTemplateItems() {
		List<ObservableTreeItem> projectItems = createObservableTreeItems(createResources(3, IProject.class,
				resource -> {
					when(resource.getName()).thenReturn(String.valueOf(System.currentTimeMillis()));
					}));
		for (int i = 0; i < 3; i++) {
			projectItems.get(i).setChildren(createObservableTreeItems(createResources(i + 1, ITemplate.class)));;
		}
		return projectItems;
	}

	@Test
	public void getProjectItemsShouldReturnAllItemsSet() {
		// pre-conditions
		model.setProjectItems(Collections.emptyList());

		// operations
		model.setProjectItems(projectItems);

		// verification
		assertThat(model.getProjectItems()).containsExactlyElementsOf(projectItems);
	}

	@Test
	public void setProjectShouldReturnSameProject() {
		// pre-conditions
		assertThat(model.getProjectItems().size()).isGreaterThan(2); 
		IProject project2 = (IProject) model.getProjectItems().get(1).getModel();

		// operations
		model.setProject(project2);

		// verification
		assertThat(model.getProject()).isEqualTo(project2);
	}

	@Test
	public void setProjectItemsShouldPreserveSelectedProjectIfContained() {
		// pre-conditions
		this.projectItems.add(new ObservableTreeItem(project));
		model.setProject(project);
		IProject selectedProject = model.getProject(); 
		assertThat(selectedProject).isNotNull();
		
		// operations
		model.setProjectItems(projectItems);

		// verification
		assertThat(model.getProject()).isEqualTo(selectedProject);
	}
	
	@Test
	public void setProjectItemsShouldSelect1stProjectIfCurrentNotContained() {
		// pre-conditions
		model.setProject(project);

		// operations
		model.setProjectItems(projectItems);

		// verification
		ObservableTreeItem projectItem = projectItems.get(0);
		assertThat(projectItem).isNotNull();
		assertThat(model.getProject()).isEqualTo(projectItem.getModel());
	}

	@Test
	public void setNullProjectShouldSet1stProject() {
		// pre-conditions

		// operations
		model.setProject(null);

		// verification
		assertThat(model.getProject()).isEqualTo(getProject(0));
	}
	
	@Test
	public void setNullProjectShouldSetNullIfNoProjectsAvailable() {
		// pre-conditions
		model.setProjectItems(Collections.emptyList());
		
		// operations
		model.setProject(null);

		// verification
		assertThat(model.getProject()).isNull();
	}

	@Test
	public void setProjectToProject2ShouldHaveGetTemplatesReturnTemplatesForProject2() {
		// pre-conditions
		IProject project2 = getProject(1); 

		// operations
		model.setProject(project2);
		List<ObservableTreeItem> templates = model.getTemplates();
		
		// verification
		assertThat(templates).containsAll(getTemplateItemsForProject(1));
	}

	@Test
	public void setServerTemplateShouldSetUseLocalTemplateToFalse() {
		// pre-conditions
		ITemplate template = ResourceMocks.createResource(ITemplate.class, null);

		// operations
		model.setServerTemplate(template );

		// verification
		assertThat(model.isUseLocalTemplate()).isFalse();
	}
	
	@Test
	public void setLocalTemplateFilenameShouldSetUseLocalTemplateToTrue() {
		// pre-conditions

		// operations
		model.setLocalTemplateFileName("test.json");

		// verification
		assertThat(model.isUseLocalTemplate()).isTrue();
	}

	@Test
	public void setTemplateFileNameShouldLoadAndParseTheTemplate() {
		when(factory.create(any(InputStream.class))).thenReturn(template);
		model.setUseLocalTemplate(true);
		model.setLocalTemplateFileName("resources/eap6-basic-sti.json");
		
		verify(factory).create(any(InputStream.class));
		assertEquals(template, model.getSelectedTemplate());
	}

	@Test
	public void setTemplateShouldCopyParametersAndLabels() {
		Map<String, IParameter> parameters = givenTheTemplateHasParameters();
		HashMap<String, String> labels = givenTheTemplateHasObjectLabels();
		Collection<IResourceLabelsPageModel.Label> modelLabels = new ArrayList<IResourceLabelsPageModel.Label>();
		for (Entry<String, String> label : labels.entrySet()) {
			modelLabels.add(new IResourceLabelsPageModel.Label(label.getKey(), label.getValue()));
		}
		model.setUseLocalTemplate(false);
		model.setServerTemplate(template);

		assertArrayEquals(parameters.values().toArray(), model.getParameters().toArray());
		assertArrayEquals(modelLabels.toArray(), model.getLabels().toArray());
	}
	
	private HashMap<String, String> givenTheTemplateHasObjectLabels() {
		HashMap<String, String> labels = new HashMap<String, String>();
		labels.put("abc", "xyz");
		when(template.getObjectLabels()).thenReturn(labels);
		return labels;
	}

	@Test
	public void resetParameterShouldSetTheOriginalValue() {
		IParameter param = mock(IParameter.class);
		when(param.getName()).thenReturn("foo");
		when(param.getValue()).thenReturn("abc");
		when(param.clone()).thenReturn(param);
		
		model.setParameters(Arrays.asList(new IParameter[] {param}));
		model.resetParameter(param);
		
		verify(param).setValue("abc");
	}
	
	@Test
	public void updateParameterValueShouldUpdateTheParameterValue() {
		IParameter param = mock(IParameter.class);
		model.updateParameterValue(param, "abc123");
		
		verify(param).setValue(eq("abc123"));
	}
	
	@Test
	public void getParametersShouldReturnAnEmptyMapWhenTemplateIsNull() {
		model.setServerTemplate(null);
		assertNotNull("Exp. an empty map",model.getParameters());
	}
	
	@Test
	public void getParametersShouldReturnAParameterMapWhenTemplateIsNotNull() {
		Map<String, IParameter> parameters = givenTheTemplateHasParameters();
		model.setUseLocalTemplate(false);
		model.setServerTemplate(template);
		
		assertArrayEquals(parameters.values().toArray(), model.getParameters().toArray());
	}
	
	private Map<String, IParameter> givenTheTemplateHasParameters() {
		IParameter param = mock(IParameter.class);
		when(param.getName()).thenReturn("foo");
		when(param.clone()).thenReturn(param);
		Map<String, IParameter> parameters = new HashMap<String, IParameter>();
		parameters.put(param.getName(), param );
		when(template.getParameters()).thenReturn(parameters);
		return parameters;
	}
	
	private IProject getProject(int i) {
		assertThat(projectItems.size()).isGreaterThan(i + 1);

		return (IProject) projectItems.get(i).getModel();
	}

	private List<ObservableTreeItem> getTemplateItemsForProject(int i) {
		assertThat(projectItems.size()).isGreaterThan(i + 1);

		return projectItems.get(i).getChildren().stream()
				.collect(Collectors.<ObservableTreeItem>toList());
	}

	public static class TestableNewApplicationWizardModel extends NewApplicationWizardModel {
		@Override
		public void setProjectItems(List<ObservableTreeItem> projects) {
			super.setProjectItems(projects);
		}
	}
}
