"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_ux_1 = require("cli-ux");
const fs = require("fs");
const yaml = require("js-yaml");
const Listr = require("listr");
const kube_1 = require("../../api/kube");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const common_tasks_1 = require("./common-tasks");
class OperatorTasks {
    constructor() {
        this.operatorServiceAccount = 'codeready-operator';
        this.operatorRole = 'codeready-operator';
        this.operatorClusterRole = 'codeready-operator';
        this.operatorRoleBinding = 'codeready-operator';
        this.operatorClusterRoleBinding = 'codeready-operator';
        this.cheClusterCrd = 'checlusters.org.eclipse.che';
        this.operatorName = 'codeready-operator';
    }
    /**
     * Returns tasks list which perform preflight platform checks.
     */
    startTasks(flags, command) {
        const clusterRoleName = `${flags.chenamespace}-${this.operatorClusterRole}`;
        const clusterRoleBindingName = `${flags.chenamespace}-${this.operatorClusterRoleBinding}`;
        const kube = new kube_1.KubeHelper(flags);
        if (util_1.isStableVersion(flags)) {
            command.warn('Consider using the more reliable \'OLM\' installer when deploying a stable release of CodeReady Workspaces (--installer=olm).');
        }
        return new Listr([
            common_tasks_1.copyOperatorResources(flags, command.config.cacheDir),
            common_tasks_1.createNamespaceTask(flags),
            {
                title: `Create ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'service_account.yaml';
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create Role ${this.operatorRole} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleExist(this.operatorRole, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'role.yaml';
                        const statusCode = yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRole ${clusterRoleName}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleExist(clusterRoleName);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'cluster_role.yaml';
                        const statusCode = yield kube.createClusterRoleFromFile(yamlFilePath, clusterRoleName);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create RoleBinding ${this.operatorRoleBinding} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.operatorRoleBinding, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'role_binding.yaml';
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRoleBinding ${clusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleBindingExist(clusterRoleBindingName);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(clusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, clusterRoleName);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create CRD ${this.cheClusterCrd}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.crdExist(this.cheClusterCrd);
                    const yamlFilePath = ctx.resourcesPath + 'crds/org_v1_che_crd.yaml';
                    if (exist) {
                        const checkCRD = yield kube.isCRDCompatible(this.cheClusterCrd, yamlFilePath);
                        if (!checkCRD) {
                            cli_ux_1.cli.error(`It is not possible to proceed the installation of CodeReady Workspaces. The existed ${this.cheClusterCrd} is different from a new one. Please update it to continue the installation.`);
                        }
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createCrdFromFile(yamlFilePath);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Create deployment ${this.operatorName} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(this.operatorName, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createDeploymentFromFile(ctx.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            common_tasks_1.createEclipseCheCluster(flags, kube)
        ], { renderer: flags['listr-renderer'] });
    }
    preUpdateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Checking versions compatibility before updating',
                task: (ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const operatorDeployment = yield kube.getDeployment(this.operatorName, flags.chenamespace);
                    if (!operatorDeployment) {
                        command.error(`${this.operatorName} deployment is not found in namespace ${flags.chenamespace}.\nProbably CodeReady Workspaces was initially deployed with another installer`);
                        return;
                    }
                    const deployedCheOperator = this.retrieveContainerImage(operatorDeployment);
                    const deployedCheOperatorImageAndTag = deployedCheOperator.split(':', 2);
                    ctx.deployedCheOperatorImage = deployedCheOperatorImageAndTag[0];
                    ctx.deployedCheOperatorTag = deployedCheOperatorImageAndTag.length === 2 ? deployedCheOperatorImageAndTag[1] : 'latest';
                    const newCheOperatorImageAndTag = flags['che-operator-image'].split(':', 2);
                    ctx.newCheOperatorImage = newCheOperatorImageAndTag[0];
                    ctx.newCheOperatorTag = newCheOperatorImageAndTag.length === 2 ? newCheOperatorImageAndTag[1] : 'latest';
                })
            }
        ]);
    }
    updateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        const clusterRoleName = `${flags.chenamespace}-${this.operatorClusterRole}`;
        const clusterRoleBindingName = `${flags.chenamespace}-${this.operatorClusterRoleBinding}`;
        return new Listr([
            common_tasks_1.copyOperatorResources(flags, command.config.cacheDir),
            {
                title: `Updating ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    const yamlFilePath = ctx.resourcesPath + 'service_account.yaml';
                    if (exist) {
                        yield kube.replaceServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating Role ${this.operatorRole} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleExist(this.operatorRole, flags.chenamespace);
                    const yamlFilePath = ctx.resourcesPath + 'role.yaml';
                    if (exist) {
                        const statusCode = yield kube.replaceRoleFromFile(yamlFilePath, flags.chenamespace);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        const statusCode = yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRole ${clusterRoleName}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleExists = yield kube.clusterRoleExist(clusterRoleName);
                    const legacyClusterRoleExists = yield kube.clusterRoleExist(this.operatorClusterRole);
                    const yamlFilePath = ctx.resourcesPath + 'cluster_role.yaml';
                    if (clusterRoleExists) {
                        const statusCode = yield kube.replaceClusterRoleFromFile(yamlFilePath, clusterRoleName);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...updated.`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleExists) {
                        const statusCode = yield kube.replaceClusterRoleFromFile(yamlFilePath, this.operatorClusterRole);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `Updating ClusterRole ${this.operatorClusterRole}...updated.`;
                    }
                    else {
                        const statusCode = yield kube.createClusterRoleFromFile(yamlFilePath, clusterRoleName);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...created a new one.`;
                    }
                })
            },
            {
                title: `Updating RoleBinding ${this.operatorRoleBinding} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.operatorRoleBinding, flags.chenamespace);
                    const yamlFilePath = ctx.resourcesPath + 'role_binding.yaml';
                    if (exist) {
                        yield kube.replaceRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRoleBinding ${clusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleBindExists = yield kube.clusterRoleBindingExist(clusterRoleBindingName);
                    const legacyClusterRoleBindExists = yield kube.clusterRoleBindingExist(this.operatorClusterRoleBinding);
                    if (clusterRoleBindExists) {
                        yield kube.replaceClusterRoleBinding(clusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, clusterRoleName);
                        task.title = `${task.title}...updated.`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleBindExists) {
                        yield kube.replaceClusterRoleBinding(this.operatorClusterRoleBinding, this.operatorServiceAccount, flags.chenamespace, this.operatorClusterRole);
                        task.title = `Updating ClusterRoleBinding ${this.operatorClusterRoleBinding}...updated.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(clusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, clusterRoleName);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating CodeReady Workspaces cluster CRD ${this.cheClusterCrd}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const crd = yield kube.getCrd(this.cheClusterCrd);
                    const yamlFilePath = ctx.resourcesPath + 'crds/org_v1_che_crd.yaml';
                    if (crd) {
                        if (!crd.metadata || !crd.metadata.resourceVersion) {
                            throw new Error(`Fetched CRD ${this.cheClusterCrd} without resource version`);
                        }
                        yield kube.replaceCrdFromFile(yamlFilePath, crd.metadata.resourceVersion);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createCrdFromFile(yamlFilePath);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Updating deployment ${this.operatorName} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(this.operatorName, flags.chenamespace);
                    if (exist) {
                        yield kube.replaceDeploymentFromFile(ctx.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createDeploymentFromFile(ctx.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting newer operator to be run',
                task: (_ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(1000);
                    yield kube.waitLatestReplica(this.operatorName, flags.chenamespace);
                })
            }
        ], { renderer: flags['listr-renderer'] });
    }
    /**
     * Returns list of tasks which remove CodeReady Workspaces operator related resources
     */
    deleteTasks(flags) {
        let kh = new kube_1.KubeHelper(flags);
        const clusterRoleName = `${flags.chenamespace}-${this.operatorClusterRole}`;
        const clusterRoleBindingName = `${flags.chenamespace}-${this.operatorClusterRoleBinding}`;
        return [{
                title: `Delete the Custom Resource of type ${constants_1.CHE_CLUSTER_CRD}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kh.deleteCheCluster(flags.chenamespace);
                    yield cli_ux_1.cli.wait(2000); //wait a couple of secs for the finalizers to be executed
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete role binding ${this.operatorRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleBindingExist(this.operatorRoleBinding, flags.chenamespace)) {
                        yield kh.deleteRoleBinding(this.operatorRoleBinding, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete role ${this.operatorRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleExist(this.operatorRole, flags.chenamespace)) {
                        yield kh.deleteRole(this.operatorRole, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete cluster role binding ${clusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleBindExists = yield kh.clusterRoleBindingExist(clusterRoleBindingName);
                    const legacyClusterRoleBindExists = yield kh.clusterRoleBindingExist(this.operatorClusterRoleBinding);
                    if (clusterRoleBindExists) {
                        yield kh.deleteClusterRoleBinding(clusterRoleBindingName);
                        task.title = yield `${task.title}...OK`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleBindExists) {
                        yield kh.deleteClusterRoleBinding(this.operatorClusterRoleBinding);
                        task.title = yield `Delete cluster role binding ${this.operatorClusterRoleBinding}...OK`;
                    }
                })
            },
            {
                title: `Delete cluster role ${clusterRoleName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleExists = yield kh.clusterRoleExist(clusterRoleName);
                    const legacyClusterRoleExists = yield kh.clusterRoleExist(this.operatorClusterRole);
                    if (clusterRoleExists) {
                        yield kh.deleteClusterRole(clusterRoleName);
                        task.title = yield `${task.title}...OK`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleExists) {
                        yield kh.deleteClusterRole(this.operatorClusterRole);
                        task.title = yield `Delete cluster role ${this.operatorClusterRole}...OK`;
                    }
                })
            },
            {
                title: 'Delete server and workspace rolebindings',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleBindingExist('che', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che', flags.chenamespace);
                    }
                    if (yield kh.roleBindingExist('che-workspace-exec', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che-workspace-exec', flags.chenamespace);
                    }
                    if (yield kh.roleBindingExist('che-workspace-view', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che-workspace-view', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete service accounts ${this.operatorServiceAccount}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace)) {
                        yield kh.deleteServiceAccount(this.operatorServiceAccount, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete PVC codeready-operator',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.persistentVolumeClaimExist('codeready-operator', flags.chenamespace)) {
                        yield kh.deletePersistentVolumeClaim('codeready-operator', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
        ];
    }
    evaluateTemplateOperatorImage(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (flags['che-operator-image']) {
                return flags['che-operator-image'];
            }
            else {
                const filePath = flags.templates + '/codeready-workspaces-operator/operator.yaml';
                const yamlFile = fs.readFileSync(filePath);
                const yamlDeployment = yaml.safeLoad(yamlFile.toString());
                return yamlDeployment.spec.template.spec.containers[0].image;
            }
        });
    }
    retrieveContainerImage(deployment) {
        const containers = deployment.spec.template.spec.containers;
        const namespace = deployment.metadata.namespace;
        const name = deployment.metadata.name;
        if (containers.length === 0) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. Containers list are empty`);
        }
        if (containers.length > 1) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. It has multiple containers`);
        }
        const container = containers[0];
        if (!container.image) {
            throw new Error(`Container ${container.name} in deployment ${namespace}/${name} must have image specified`);
        }
        return container.image;
    }
}
exports.OperatorTasks = OperatorTasks;
//# sourceMappingURL=operator.js.map