"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const commandExists = require("command-exists");
const execa = require("execa");
const Listr = require("listr");
const version_1 = require("../../api/version");
/**
 * Helper for Code Ready Container
 */
class CRCHelper {
    preflightCheckTasks(flags, command) {
        return new Listr([
            {
                title: 'Verify if oc is installed',
                task: (_ctx, task) => {
                    if (!commandExists.sync('oc')) {
                        command.error('E_REQUISITE_NOT_FOUND');
                    }
                    else {
                        task.title = `${task.title}...done.`;
                    }
                }
            },
            {
                title: 'Verify if crc is installed',
                task: (_ctx, task) => {
                    if (!commandExists.sync('crc')) {
                        command.error('E_REQUISITE_NOT_FOUND', { code: 'E_REQUISITE_NOT_FOUND' });
                    }
                    else {
                        task.title = `${task.title}...done.`;
                    }
                }
            },
            {
                title: 'Verify if CodeReady Containers is running',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const crcIsRunning = yield this.isCRCRunning();
                    if (!crcIsRunning) {
                        command.error('E_PLATFORM_NOT_READY');
                    }
                    else {
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            version_1.VersionHelper.getOpenShiftCheckVersionTask(flags),
            version_1.VersionHelper.getK8sCheckVersionTask(flags),
            {
                title: 'Retrieving CodeReady Containers IP and domain for routes URLs',
                enabled: () => flags.domain !== undefined,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const ip = yield this.getCRCIP();
                    flags.domain = ip + '.nip.io';
                    task.title = `${task.title}...${flags.domain}.`;
                })
            },
        ], { renderer: flags['listr-renderer'] });
    }
    isCRCRunning() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { exitCode, stdout } = yield execa('crc', ['status'], { timeout: 60000, reject: false });
            if (exitCode === 0 &&
                stdout.includes('CRC VM:          Running') &&
                stdout.includes('OpenShift:       Running')) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    getCRCIP() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { stdout } = yield execa('crc', ['ip'], { timeout: 10000 });
            return stdout;
        });
    }
}
exports.CRCHelper = CRCHelper;
//# sourceMappingURL=crc.js.map