/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.felix.framework;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;

import org.apache.felix.framework.cache.Content;

public class PatchAwareContent implements Content {

    private final Content content;
    private final File dir;

    public static URL getPatch(String symbolicName, String version, String entry) {
        try {
            File file = getFile(System.getProperty("felix.patch.location"), symbolicName, version, entry);
            return file != null && file.isFile() ? file.toURI().toURL() : null;
        } catch (IOException e) {
            return null;
        }
    }

    private static File getFile(String... paths) {
        File file = null;
        for (String p : paths) {
            if (p == null) {
                return null;
            }
            if (file == null) {
                file = new File(p);
            } else {
                file = new File(file, p);
            }
        }
        return file;
    }

    public PatchAwareContent(Content content, String symbolicName, String version) {
        this(content, getFile(System.getProperty("felix.patch.location"), symbolicName, version));
    }

    public PatchAwareContent(Content content, File dir) {
        this.content = content;
        this.dir = dir;
    }

    public void close() {
        content.close();
    }

    public boolean hasEntry(String name) {
        return content.hasEntry(name);
    }

    public Enumeration getEntries() {
        return content.getEntries();
    }

    public byte[] getEntryAsBytes(String name) {
        InputStream is = null;
        try {
            is = getEntryAsStream(name);
            return getContent(is);
        } catch (IOException e) {
            return null;
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
            }
        }
    }

    private static byte[] getContent(InputStream is) throws IOException {
        if (is == null) {
            return null;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream(4096);
        byte[] buf = new byte[4096];
        int n;
        while ((n = is.read(buf, 0, buf.length)) >= 0) {
            baos.write(buf, 0, n);
        }
        return baos.toByteArray();
    }

    public InputStream getEntryAsStream(String name) throws IOException {
        File file = dir != null ? new File(dir, name) : null;
        if (file != null && file.isFile()) {
            return new FileInputStream(file);
        } else {
            return content.getEntryAsStream(name);
        }
    }

    public Content getEntryAsContent(String name) {
        Content c = content.getEntryAsContent(name);
        return c != null ? new PatchAwareContent(c, dir) : null;
    }

    public String getEntryAsNativeLibrary(String name) {
        return content.getEntryAsNativeLibrary(name);
    }

    public URL getEntryAsURL(String name) {
        return content.getEntryAsURL(name);
    }
}
