/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.processor.async;

import java.util.concurrent.atomic.AtomicInteger;

import org.apache.camel.CamelContext;
import org.apache.camel.ContextTestSupport;
import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.model.ProcessorDefinition;
import org.apache.camel.spi.InterceptStrategy;

/**
 * Using a custom interceptor which is not a {@link org.apache.camel.AsyncProcessor} which Camel
 * detects and uses a bridge to adapt to so the asynchronous engine can still run. Albeit not
 * the most optimal solution but it runs. Camel will log a WARN so user can see the issue
 * and change his interceptor to comply.
 *
 * @version 
 */
public class AsyncEndpointCustomInterceptorTest extends ContextTestSupport {

    private static String beforeThreadName;
    private static String afterThreadName;
    private MyInterceptor interceptor = new MyInterceptor();

    public void testAsyncEndpoint() throws Exception {
        getMockEndpoint("mock:before").expectedBodiesReceived("Hello Camel");
        getMockEndpoint("mock:after").expectedBodiesReceived("Bye Camel");
        getMockEndpoint("mock:result").expectedBodiesReceived("Bye Camel");

        String reply = template.requestBody("direct:start", "Hello Camel", String.class);
        assertEquals("Bye Camel", reply);

        assertMockEndpointsSatisfied();

        assertEquals(8, interceptor.getCounter());

        assertFalse("Should use different threads", beforeThreadName.equalsIgnoreCase(afterThreadName));
    }

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                context.addComponent("async", new MyAsyncComponent());
                context.addInterceptStrategy(interceptor);

                from("direct:start")
                        .to("mock:before")
                        .to("log:before")
                        .process(new Processor() {
                            public void process(Exchange exchange) throws Exception {
                                beforeThreadName = Thread.currentThread().getName();
                            }
                        })
                        .to("async:bye:camel")
                        .process(new Processor() {
                            public void process(Exchange exchange) throws Exception {
                                afterThreadName = Thread.currentThread().getName();
                            }
                        })
                        .to("log:after")
                        .to("mock:after")
                        .to("mock:result");
            }
        };
    }

    // START SNIPPET: e1
    private static class MyInterceptor implements InterceptStrategy {
        private AtomicInteger counter = new AtomicInteger();

        public Processor wrapProcessorInInterceptors(final CamelContext context, final ProcessorDefinition<?> definition,
                                                     final Processor target, final Processor nextTarget) throws Exception {

            return new Processor() {
                public void process(Exchange exchange) throws Exception {
                    // we just want to count number of interceptions
                    counter.incrementAndGet();

                    // and continue processing the exchange
                    target.process(exchange);
                }
            };
        }

        public int getCounter() {
            return counter.get();
        }
    }
    // END SNIPPET: e1

}