/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.seda;

import org.apache.camel.ContextTestSupport;
import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.NotifyBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.spi.Synchronization;

/**
 * Unit test to verify unit of work with seda. That the UnitOfWork is able to route using seda
 * but keeping the same UoW.
 *
 * @version 
 */
public class SedaUnitOfWorkTest extends ContextTestSupport {

    private static volatile String sync;
    private static volatile String lastOne;

    public void testSedaUOW() throws Exception {
        NotifyBuilder notify = new NotifyBuilder(context).whenDone(2).create();

        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedMessageCount(1);

        template.sendBody("direct:start", "Hello World");

        assertMockEndpointsSatisfied();
        notify.matchesMockWaitTime();

        assertEquals("onCompleteA", sync);
        assertEquals("onCompleteA", lastOne);
    }

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                context.setTracing(true);

                from("direct:start")
                        .process(new MyUOWProcessor("A"))
                        .to("seda:foo");

                from("seda:foo")
                        .process(new Processor() {
                            public void process(Exchange exchange) throws Exception {
                                assertEquals(null, sync);
                            }
                        })
                        .process(new Processor() {
                            public void process(Exchange exchange) throws Exception {
                                lastOne = "processor";
                            }
                        })
                        .to("mock:result");
            }
        };
    }

    private static final class MyUOWProcessor implements Processor {

        private String id;

        private MyUOWProcessor(String id) {
            this.id = id;
        }

        public void process(Exchange exchange) throws Exception {
            exchange.getUnitOfWork().addSynchronization(new Synchronization() {
                public void onComplete(Exchange exchange) {
                    sync = "onComplete" + id;
                    lastOne = sync;
                }

                public void onFailure(Exchange exchange) {
                    sync = "onFailure" + id;
                    lastOne = sync;
                }
            });
        }
    }

}
