/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.felix.utils.manifest;

import org.osgi.framework.Version;

import java.util.ArrayList;
import java.util.List;

public class Attribute
{

    private final String name;
    private final String value;
    private final String type;

    public Attribute(String name, String value)
    {
        this(name, value, null);
    }

    public Attribute(String name, String value, String type)
    {
        this.name = name;
        this.value = value;
        this.type = type;
    }

    public String getName()
    {
        return name;
    }

    public String getValue()
    {
        return value;
    }

    public String getType()
    {
        return type;
    }

    public Object getNormalizedValue()
    {
        if (type == null || type.equals("String"))
        {
            return value;
        }
        else if (type.equals("Double"))
        {
            return new Double(value.trim());
        }
        else if (type.equals("Version"))
        {
            return new Version(value.trim());
        }
        else if (type.equals("Long"))
        {
            return new Long(value.trim());
        }
        else if (type.startsWith("List"))
        {
            int startIdx = type.indexOf('<');
            int endIdx = type.indexOf('>');
            if (((startIdx > 0) && (endIdx <= startIdx)) || ((startIdx < 0) && (endIdx > 0)))
            {
                throw new IllegalArgumentException("Invalid Provide-Capability attribute list type for '" + value + "' : " + type);
            }

            String listType = "String";
            if (endIdx > startIdx)
            {
                listType = type.substring(startIdx + 1, endIdx).trim();
            }

            String[] tokens = Parser.parseDelimitedString(value, ",", false);
            List<Object> values = new ArrayList<Object>(tokens.length);
            for (String token : tokens)
            {
                if (listType.equals("String"))
                {
                    values.add(token);
                }
                else if (listType.equals("Double"))
                {
                    values.add(new Double(token.trim()));
                }
                else if (listType.equals("Version"))
                {
                    values.add(new Version(token.trim()));
                }
                else if (listType.equals("Long"))
                {
                    values.add(new Long(token.trim()));
                }
                else
                {
                    throw new IllegalArgumentException("Unknown Provide-Capability attribute list type for '" + name + "' : " + type);
                }
            }
            return values;
        }
        else
        {
            throw new IllegalArgumentException("Unknown Provide-Capability attribute type for '" + name + "' : " + type);
        }
    }

}
