/*
 * Copyright 2013 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package org.jbosson.plugins.fuse.utils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.jbosson.plugins.fuse.JBossFuseProfileGroupManager;
import org.linkedin.util.clock.Timespan;
import org.linkedin.zookeeper.client.ZKClient;

/**
 * Implements {@link FabricRegistry} using LinkedIn Zookeeper library.
 *
 * @author dbokde
 */
public class LinkedInFabricRegistryImpl implements FabricRegistry {

    private static final Log LOG = LogFactory.getLog(LinkedInFabricRegistryImpl.class);

    private final String url;
    private final long timeout;
    private final String password;
    private final String containerName;

    private ZKClient client;
    private boolean connected;

    public static boolean isSupported() {
        try {
            Class.forName("org.linkedin.zookeeper.client.ZKClient");
            return true;
        } catch (ClassNotFoundException e) {
            return false;
        }
    }

    public LinkedInFabricRegistryImpl(String zooKeeperUrl, String zooKeeperPassword, long timeout,
                                      String containerName) {
        this.url = zooKeeperUrl;
        this.timeout = timeout;
        this.password = zooKeeperPassword;
        this.containerName = containerName;
    }

    public void connect() throws Exception {
        final Timespan fabricTimeout = Timespan.seconds(timeout);
        client = new ZKClient(url, fabricTimeout, new Watcher() {
            public void process(WatchedEvent watchedEvent) {
                // just log the events for now
                if (connected && watchedEvent.getState().equals(Event.KeeperState.Disconnected)) {
                    LOG.warn("Disconnected from Fabric Registry");
                }
            }
        });

        try {
            // need to call lifecycle methods
            client.start();
            if (!client.isConnected()) {
                // wait for connection timeout period
                Thread.sleep(fabricTimeout.getDurationInMilliseconds());
                if (!client.isConnected()) {
                    throw new IllegalAccessException("Unable to connect to Fabric Registry in " +
                            fabricTimeout.getDurationInSeconds() + " seconds");
                }
            }

            // use the ZK password after connecting
            if (password != null) {
                client.addAuthInfo("digest", ("fabric:" + password).getBytes("UTF-8"));
            }

            connected = true;

        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw e;
        }
    }

    public void disconnect() throws Exception {
        if (client != null) {
            connected = false;
            try {
                client.destroy();
            } catch (Throwable t) {
                // ignore destroy errors
            }
        }
    }

    public String getContainerVersion() throws Exception {
        return client.getStringData(String.format(JBossFuseProfileGroupManager.CONTAINER_VERSION_PATH_FORMAT, containerName));
    }

    public String[] getContainerProfiles(String containerVersion) throws Exception {
        return client.getStringData(
                String.format(JBossFuseProfileGroupManager.CONTAINER_PROFILES_PATH_FORMAT, containerVersion, containerName)).split(" ");
    }

    /**
     * Can't return {@link org.linkedin.zookeeper.client.IZKClient} here as its dynamically loaded.
     * @return IZKClient instance.
     */
    public Object getClient() {
        return client;
    }

}
