/*
 * #%L
 * Fuse EAP :: Config
 * %%
 * Copyright (C) 2015 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package org.jboss.fuse.eap.config;

import static org.wildfly.extras.config.LayerConfig.Type.INSTALLING;
import static org.wildfly.extras.config.LayerConfig.Type.OPTIONAL;
import static org.wildfly.extras.config.LayerConfig.Type.REQUIRED;

import static org.wildfly.extras.config.NamespaceConstants.NS_DOMAIN;
import static org.wildfly.extras.config.NamespaceConstants.NS_SECURITY;

import java.net.URL;
import java.util.Arrays;
import java.util.List;

import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.Text;
import org.wildfly.extras.config.ConfigContext;
import org.wildfly.extras.config.ConfigPlugin;
import org.wildfly.extras.config.ConfigSupport;
import org.wildfly.extras.config.LayerConfig;
import org.wildfly.extras.config.NamespaceRegistry;


public class SrampConfigPlugin implements ConfigPlugin {

    private static final String NS_OVERLORD_CONFIG = "urn:jboss:domain:overlord-configuration";
    private static final String NS_OVERLORD_DEPLOY = "urn:jboss:domain:overlord-deployment";
    private static final String NS_MESSAGING = "urn:jboss:domain:messaging";

    private NamespaceRegistry registry;

    public SrampConfigPlugin()
    {
        registerNamespaceVersions(new NamespaceRegistry());
    }

    public SrampConfigPlugin(NamespaceRegistry registry)
    {
        registerNamespaceVersions(registry);
    }

    @Override
    public String getConfigName() {
        return "sramp";
    }

    @Override
    public List<LayerConfig> getLayerConfigs() {
        return Arrays.asList(
            new LayerConfig("fuse", REQUIRED, -10),
            new LayerConfig("bpms", OPTIONAL, -9),
            new LayerConfig("soa", OPTIONAL, -11),
            new LayerConfig ("artificer", INSTALLING, -12));
    }

    @Override
    public void applyStandaloneConfigChange(ConfigContext context, boolean enable) {
        updateExtensions (context, enable);
        updateVault (context, enable);
        updateSecuritySetting (context, enable);
        updateJmsTopic (context, enable);
        updateSecurityDomain (context, enable);
        updateOverlordConfigSubsystem (context, enable);
        updateOverlordDeploySubsystem (context, enable);
    }

    @Override
    public void applyDomainConfigChange(ConfigContext context, boolean enable) {
        applyStandaloneConfigChange(context, enable);
    }

    private void updateExtensions(ConfigContext context, boolean enable) {
        String extPrefix = "org.overlord.commons.eap.extensions.";
        String[] extensionNames = new String[2];
        extensionNames[0] = extPrefix + "config";
        extensionNames[1] = extPrefix + "deploy";

        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Element extensions = ConfigSupport.findChildElement(context.getDocument().getRootElement(), "extensions", domainNamespaces);
        ConfigSupport.assertExists(extensions, "Did not find the <extensions> element");
        Namespace domainNamespace = context.getDocument().getRootElement().getNamespace();
        for (String extName : extensionNames) {
            Element element = ConfigSupport.findElementWithAttributeValue(extensions, "extension", "module", extName, domainNamespaces);
            if (enable && element == null) {
                extensions.addContent(new Text("    "));
                extensions.addContent(new Element("extension", domainNamespace).setAttribute("module", extName));
                extensions.addContent(new Text("\n    "));
            }
            if (!enable && element != null) {
                element.getParentElement().removeContent(element);
            }
        }
    }

    private void updateVault (ConfigContext context, boolean enable) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Element root = context.getDocument().getRootElement();
        Element vault = ConfigSupport.findChildElement(root, "vault", domainNamespaces);
        Namespace domainNamespace = root.getNamespace();
        if (enable && vault == null) {
            // Insert the vault section before the management section.
            Element management = ConfigSupport.findChildElement(root, "management", domainNamespaces);

            ConfigSupport.assertExists (management, "Did not find the <management> element");
            int managementIndex = root.indexOf (management);
            URL resource = SrampConfigPlugin.class.getResource ("/xml/vault.xml");
            Element vaultElement = ConfigSupport.loadElementFrom(resource).setNamespace(domainNamespace);
            List<Element> children = vaultElement.getChildren();
            for (Element child : children)
            {
                child.setNamespace(domainNamespace);
            }

            root.addContent(managementIndex - 1, vaultElement);
            root.addContent(managementIndex - 1, new Text("\n"));
        }

        if (!enable && vault != null) {
            root.removeContent (vault);
        }
    }

    private void updateSecuritySetting (ConfigContext context, boolean enable) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Namespace[] messagingNamespaces = null;

        try {
            messagingNamespaces = registry.getNamespaces(NS_MESSAGING);
        } catch (IllegalArgumentException ex) {
            return;
        }

        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument(), domainNamespaces);
        for (Element profile : profiles) {
            Element messaging = ConfigSupport.findChildElement(profile, "subsystem", messagingNamespaces);

            /* Messaging may not be present in every configuration, so only
             * change it if it exists.
             */

            if (messaging != null) {
                Element hornetqServer = ConfigSupport.findChildElement(messaging, "hornetq-server", messagingNamespaces);
                ConfigSupport.assertExists (hornetqServer, "Did not find the <hornetqServer> element.");
                Element securitySettings =  ConfigSupport.findChildElement(hornetqServer, "security-settings", messagingNamespaces);
                ConfigSupport.assertExists (securitySettings, "Did not find the <security-settings> element.");
                Element securitySetting = ConfigSupport.findElementWithAttributeValue (securitySettings, "security-setting", "match", "jms.topic.ArtificerTopic", messagingNamespaces);
                if (enable && securitySetting == null) {
                    URL resource = SrampConfigPlugin.class.getResource ("/xml/security-setting.xml");
                    securitySettings.addContent (new Text ("    "));
                    securitySettings.addContent (ConfigSupport.loadElementFrom (resource));
                    securitySettings.addContent (new Text ("\n    "));
                }

                if (!enable && securitySetting != null) {
                    securitySetting.getParentElement().removeContent (securitySetting);
                }
            }
        }
    }

    private void updateJmsTopic (ConfigContext context, boolean enable) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Namespace[] messagingNamespaces = null;
        try {
            messagingNamespaces = registry.getNamespaces(NS_MESSAGING);
        } catch (IllegalArgumentException ex) {
            return;
        }

        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument(), domainNamespaces);
        for (Element profile : profiles) {
            Element messaging = ConfigSupport.findChildElement(profile, "subsystem", messagingNamespaces);
            if (messaging != null) {
                Element hornetqServer = ConfigSupport.findChildElement(messaging, "hornetq-server", messagingNamespaces);
                ConfigSupport.assertExists (hornetqServer, "Did not find the <hornetq-server> element.");
                Element jmsDestinations = ConfigSupport.findChildElement(hornetqServer, "jms-destinations", messagingNamespaces);
                if (enable && jmsDestinations == null) {
                    jmsDestinations = new Element ("jms-destinations", hornetqServer.getNamespace());
                    hornetqServer.addContent (new Text ("    "));
                    hornetqServer.addContent (jmsDestinations);
                    hornetqServer.addContent (new Text ("\n    "));
                }

                Element jmsTopic = ConfigSupport.findElementWithAttributeValue (jmsDestinations, "jms-topic", "name", "ArtificerTopic", messagingNamespaces);
                if (enable && jmsTopic == null) {
                    URL resource = SrampConfigPlugin.class.getResource ("/xml/jms-topic.xml");
                    jmsDestinations.addContent (new Text ("    "));
                    jmsDestinations.addContent (ConfigSupport.loadElementFrom (resource));
                    jmsDestinations.addContent (new Text ("\n    "));
                }

                if (!enable && jmsTopic != null) {
                    jmsTopic.getParentElement().removeContent (jmsTopic);
                }

            }
        }
    }

    private void updateSecurityDomain (ConfigContext context, boolean enable) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Namespace[] securityNamespaces = registry.getNamespaces(NS_SECURITY);

        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument (), domainNamespaces);
        for (Element profile : profiles) {
            Element security = ConfigSupport.findChildElement(profile, "subsystem", securityNamespaces);
            ConfigSupport.assertExists (security, "Did not find the security subsystem");
            Element domains = ConfigSupport.findChildElement(security, "security-domains", securityNamespaces);
            ConfigSupport.assertExists(domains, "Did not find the <security-domains> element");

            String[] domainNames = { "overlord-idp", "overlord-sp", "overlord-jaxrs" };
            for (String domainName : domainNames) {
                Element domain = ConfigSupport.findElementWithAttributeValue (domains, "security-domain", "name", domainName, securityNamespaces);
                if (enable && domain == null) {
                    URL resource = SrampConfigPlugin.class.getResource ("/xml/security-" + domainName + ".xml");
                    domains.addContent (new Text("    "));
                    domains.addContent (ConfigSupport.loadElementFrom (resource));
                    domains.addContent (new Text ("\n            "));
                }

                if (!enable && domain != null) {
                    domain.getParentElement().removeContent (domain);
                }
            }
        }
    }

    private void updateOverlordConfigSubsystem (ConfigContext context, boolean enable)
    {
        updateSubsystem (context, enable, NS_OVERLORD_CONFIG, "/xml/overlord-config-subsystem.xml");
    }

    private void updateOverlordDeploySubsystem (ConfigContext context, boolean enable) {
        updateSubsystem (context, enable, NS_OVERLORD_DEPLOY, "/xml/overlord-deploy-subsystem.xml");
    }


    protected void updateSubsystem (ConfigContext context, boolean enable, String ns, String resourceName) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Namespace[] subsystemNamespaces = registry.getNamespaces(ns);

        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument(), domainNamespaces);
        for (Element profile : profiles) {
            Element element = ConfigSupport.findChildElement(profile, "subsystem", subsystemNamespaces);
            if (enable && element == null) {
                URL resource = SrampConfigPlugin.class.getResource (resourceName);
                profile.addContent (new Text ("    "));
                profile.addContent (ConfigSupport.loadElementFrom (resource));
                profile.addContent (new Text ("\n    "));
            }
            if (!enable && element != null) {
                element.getParentElement().removeContent (element);
            }
        }
    }

    private void registerNamespaceVersions (NamespaceRegistry registry)
    {
        this.registry = registry;
        registry.registerNamespace(NS_DOMAIN, "1.8");
        registry.registerNamespace(NS_DOMAIN, "1.7");
        registry.registerNamespace(NS_SECURITY, "1.2");
        registry.registerNamespace(NS_OVERLORD_CONFIG, "1.0");
        registry.registerNamespace(NS_OVERLORD_DEPLOY, "1.0");
    }
}
