;;
;; Multithreaded Breakout "library".
;;

;---------------------------------------------------------------------------
; Preamble

(require 'Gwish)
(require 'random)

(use-library tcl)
(use-interface tcl)
(use-interface tclhack)

;---------------------------------------------------------------------------
; Global variable definitions

(define *canvas* #f)

(define puck-r 5)

;; How big a canvas?
;;
(define play-w 520)
(define play-h 520)

;; Where does the play area start
;;
(define bounds-x 4)
(define bounds-y 4)

;; Where is the paddle ul cornder?
;;
(define paddle-x 0)
(define paddle-y 375)

;; Paddle size:
;;
(define paddle-height 10)
(define paddle-width 64)

;; How big is the in-bounds area for the puck?
;;
(define bounds-w 512)
(define bounds-h (- paddle-y puck-r 1))

;; How many blocks per row?
;;
(define n-row 16)

;; Row y positions
;;
(define row0-y 32)
(define row1-y 64)

;; Individual block size
;;
(define row-height 16)
(define row-width (/ bounds-w n-row))

;; Each entry either the name of a canvas
;; item for the block or #f if the block
;; has been eliminated:
;;
(define row0 (make-vector n-row #f))
(define row1 (make-vector n-row #f))

(define game-playable #t)
(define game-playing #f)
(define total-score 0)
(define pucks-per-game 3)
(define n-pucks pucks-per-game)
(define score-report #f)
(define total-score-report #f)
(define puck-report #f)
(define game-over-report #f)

;---------------------------------------------------------------------------
; Utility functions

(define (coin-toss . from)
  (list-ref from (random (length from))))

(proc game-q ignored
      (cond
       (game-playing (set! game-playing #f) (new-game))
       ((not game-playable)
	(new-game)
	(report-game-state))
       (else (destroy (*canvas* 'get-canvas-name)))))


(proc game-p ignored
      (cond
       (game-playing (set! game-playing #f))
       (else (play-game))))

;---------------------------------------------------------------------------
; Create a canvas object
;
; The canvas object recognizes one message, get-canvas-name. All
; other messages are passed to Tk. Access to Tk is synchronized with
; a monitor macro.

(define (make-canvas window-name window)

  (define (subwindow window-name rel)
    (let ((wname
	   (cond ((symbol? window-name) (symbol->string window-name))
		 (#t window-name))))
      (string->symbol (string-append wname "." rel))))
  
  (define (name->window name)
    (eval (string->symbol name)))
  
  (let* ((the-canvas-name (subwindow window-name "c"))
	 (the-canvas (name->window (canvas the-canvas-name))))
    ; Synchronize access to Tk, which is not thread-safe.
    (the-canvas 'configure :width play-w :height play-h)
    (pack the-canvas-name :fill "both" :expand #t)
    (bind the-canvas-name "<Motion>"
	  (tcl-lambda ("%x %y" (number x) (number y))
		      (center-paddle-at-canvas-coord x)
		      ""))
    (bind the-canvas-name '<q> 'game-q)
    (bind the-canvas-name '<Q> 'game-q)
    (bind the-canvas-name '<p> 'game-p)
    (bind the-canvas-name '<P> 'game-p)

    (lambda (arg . rest)
      (cond 
       ((eq? arg 'get-canvas-name) the-canvas-name)
       ; Synchronize access to Tk, which is not thread-safe.
       (else (monitor
	      (apply the-canvas (cons arg rest))))))))


;---------------------------------------------------------------------------
; Create a puck object.
;
; pucks handle the following messages:
;
;   start - Creates a new thread executing a play loop. Returns the 
;           new thread object.
;
;   stop  - Terminate the play loop thread. Returns when the thread has
;           terminated.
;

(define (make-puck number vx0 vy0 color)
  (let* ((id number)
	 (radius puck-r)
	 (x 0)
	 (y (- paddle-y radius 1))
	 (max-vx 2.2)
	 (max-vy 2.2)
	 (init-init-vx vx0)
	 (init-init-vy vy0)
	 (init-vx init-init-vx)
	 (init-vy init-init-vy)
	 (vx init-vx)
	 (vy init-vy)
	 (thread-id #f)
	 (score-report #f)
	 (score 0)
	 (puck 
	  (*canvas* 'create 'oval
		    (+ bounds-x (- x radius))
		    (+ bounds-y (- y radius))
		    (+ bounds-x (+ x radius))
		    (+ bounds-y (+ y radius))
		    :fill color))
	 (score-x-offset (+ bounds-x 20 (* 120 (quotient id 3))))
	 (score-y-offset (+ bounds-y 60 (* (remainder id 3) 30)))
	 (score-ball 
	  (*canvas* 'create 'oval 
		    (+ score-x-offset (- x radius))
		    (+ score-y-offset (- y radius))
		    (+ score-x-offset (+ x radius))
		    (+ score-y-offset (+ y radius))
		    :fill color)))
    
    (define (paddle-sweet? x)
      (and (> (abs vx) .00001)
	   (let ((r (/ paddle-width 2)))
	     (< (abs (- x (+ r paddle-x)))
		r))))
    
    (define (paddle-sour? x)
      (let ((r (/ paddle-width 2)))
	(< (abs (- x (+ r paddle-x)))
	   (+ (* 3 radius) r))))

    (define (report-score)
      (and score-report (*canvas* 'delete score-report))
      (set! score-report
	    (*canvas* 'create 'text 
		      (+ 10 score-x-offset) 
		      (- (+ score-y-offset paddle-y) radius)
		      :font  "-adobe-helvetica-bold-r-normal-*-18-*-*-*-*-*-*-*"
		      :anchor 'w))
      (*canvas* 'insert score-report 0
		(string-append ": " (number->string score)))
      
      ; Update the total score. Protect with monitor.
      (monitor
       (and total-score-report (*canvas* 'delete total-score-report))
       (set! total-score-report
	     (*canvas* 'create 'text
		       (- bounds-w 150)
		       494 ; (+ paddle-y 40)
		       :font "-adobe-helvetica-bold-r-normal-*-18-*-*-*-*-*-*-*"
		      :anchor 'w))
      (*canvas* 'insert total-score-report 0
		(string-append "Total: " (number->string total-score)))))


    (define (hit-puck-at-game-x!? row x-game)
      (let* ((x (- x-game bounds-x))
	     (i (inexact->exact (floor (/ x row-width)))))
	; This is a critical section.
	(monitor
	 (and (>= i 0)
	      (< i (vector-length row))
	      (vector-ref row i)
	      (begin
		(*canvas* 'delete (vector-ref row i))
		(vector-set! row i #f)
		(set! n-blocks (- n-blocks 1))
		(set! total-score (+ 1 total-score))
		(set! score (+ 1 score))
		#t)))))

    (report-score)

    (lambda (msg . rest)

      (cond 
       ((eq? msg 'start)
	(begin 
	  (define (loop)
	    (begin
	      (let ((old-x x)
		    (old-y y))
		(set! x (+ x vx))
		(set! y (+ y vy))
		(cond
		 ((or (and (< y (+ row1-y row-height))
			   (>= y row1-y)
			   (hit-puck-at-game-x!? row1 x)
			   row1-y)
		      (and (< y (+ row0-y row-height))
			   (>= y row0-y)
			   (hit-puck-at-game-x!? row0 x)
			   row0-y)
		      (and (< y 0)
			   0))
		  => (lambda (yref)
		       (report-score)
		       (set! vy (- vy))
		       (set! y (+ yref (- y yref)))
		       (if (= 0 n-blocks)
			   (begin
			     ;; You've won.
			     ;; We don't do anything other than
			     ;; delete the puck.
			     (*canvas* 'delete puck)
			     (set! puck #f)))))
		 
		 ((<= bounds-h y)
		  (begin
		    (cond
		     ((paddle-sweet? x)
		      (begin
 			(set! vy (- vy))
			(set! y (+ bounds-h (- bounds-h y)))))
		     ((paddle-sour? x)
		      (begin
			(set! vy (- vy))
			(let ((total (+ (* vx vx) (* vy vy))))
			  (set! vx ((coin-toss + -) (random init-vx)))
			  (set! vy (- (sqrt (- total (* vx vx))))))
			(set! y (+ bounds-h (- bounds-h y)))))
		     (else
		      (*canvas* 'delete puck)
		      (set! puck #f)))))
		 
		 ((< x 0)
		  (begin
		    (set! vx (- vx))
		    (set! x (- x))))
		 
		 ((<= bounds-w x)
		  (begin
		    (set! vx (- vx))
		    (set! x (+ bounds-w (- bounds-w x))))))
		
		(*canvas* 'move puck (- x old-x) (- y old-y))
		
		(if puck
		    (loop)))))
	  
	  (set! thread-id (begin-thread
			   (loop)))))

       ((eq? msg 'stop)
	(begin
	  (set! game-playing #f)
	  (join-thread thread-id))) ))))


;---------------------------------------------------------------------------
;; Hooks

(define n-blocks #f)
(define (new-level)
  (*canvas* 'delete 'all)
  (make-row! row0 row0-y 'green)
  (make-row! row1 row1-y 'blue)
  (set! n-blocks (* 2 n-row))
  (new-paddle)
  (define (make-pucks n)
    (if (> n 0)
	(begin
	  ((make-puck (- n 1)
		      (* 1.5 (+ 1 (/ (- (random 80) 40) 100)))
		      (* -1.5 (+ 1 (/ (- (random 80) 40) 100))) 
		      (array-ref 
		       #(red green blue black purple yellow orange) (- n 1))) 'start)
	  (make-pucks (- n 1)))))
  (make-pucks 3))
	  


(define (new-mt-game)
  (set! game-playable #t)
  (set! game-playing #f)
  (new-level)
  (set! n-pucks pucks-per-game)
  (set! total-score 0))


;; Make the window and playing area
;;
(define play-geom (string-append (number->string play-w)
				 'x
				 (number->string play-h)))


;; Make the two rows:
;;
(define (make-row! v y color)
  (let loop ((n 0))
    (if (= n 16)
	v
	(begin
	  (vector-set! v n (*canvas* 'create 'rectangle
				     (+ bounds-x (* n row-width))
				     (+ bounds-y y)
				     (+ bounds-x (* (+ 1 n) row-width))
				     (+ bounds-y (+ y row-height))
				     :fill color
				     :width 3))
	  (loop (+ n 1))))))


;; Drawing the paddle:
;;
(define (paddle-x-max) (+ paddle-x paddle-width))
(define (paddle-y-max) (+ paddle-y paddle-height))
(define paddle-color 'red)
(define paddle #f)
(define (new-paddle)
  (set! paddle
	(*canvas* 'create 'rectangle
		    (+ bounds-x paddle-x)
		    (+ bounds-y paddle-y)
		    (+ bounds-x (paddle-x-max))
		    (+ bounds-y (paddle-y-max))
		    :fill paddle-color)))

(define (center-paddle-at-canvas-coord x)
  (let ((old-x paddle-x))
    (set! paddle-x (- x bounds-x (/ paddle-width 2)))
    (*canvas* 'move paddle (- paddle-x old-x) 0)))

(define (initialize-mt-breakout window-name window)
  (set! *canvas* (make-canvas window-name window)))

(provide 'mt-breaklib)
