/* $Id: pl.c,v 1.39 2006/04/02 22:29:18 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres procedural language interfaces
 */
#include <setjmp.h>

#include <Python.h>
#include <compile.h>
#include <marshal.h>
#include <structmember.h>

#include <postgres.h>
#include <fmgr.h>
#include <funcapi.h>
#include <miscadmin.h>
#include <access/htup.h>
#include <access/heapam.h>
#include <access/xact.h>
#include <catalog/pg_class.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_language.h>
#include <catalog/indexing.h>
#include <commands/trigger.h>
#include <mb/pg_wchar.h>
#include <nodes/memnodes.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/builtins.h>
#include <utils/hsearch.h>
#include <utils/syscache.h>
#include <utils/relcache.h>
#include <utils/typcache.h>

#include <pypg/python.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <pypg/ci.h>
#include <pypg/encoding.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/function.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/error.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/trigger.h>
#include <pypg/cis.h>
PyPgCI_Define();

#ifndef IF_PATH
#error IF_PATH must be defined to the module of the version specific \
package space
#endif

#ifndef PGV_TAG
#error PGV_TAG must be defined to the version of PostgreSQL that the PL \
is being built against, e.g. 80beta
#endif

#ifndef EX_IFM
#error EX_IFM must be defined to the specific module to be imported
#endif

#ifndef PL_STATE
#error PL_STATE must be defined
#endif

#define DEFAULT_POSTGRES_MODULE IF_PATH"."PL_STATE"_v"PGV_TAG"."EX_IFM
/*
 * PySpec - Inspect a PyObj and print some info about it.
 * This is here to provide a function that the debugger can call.
 */
void
PySpec(PyObj ob)
{
	PySPEC(ob);
}

/*
 * indent
 *
 * Reallocate src with a volume of
 *		len(src) + fpad + lpad + (\n_count(src) * infact)
 * Start to copy src at msrc + fpad address
 * And set 'infact' number of characters following a \n to \t
 */
static char *
indent
(
	const char *const src,
	const unsigned short fpad,
	const unsigned short lpad,
	const unsigned short infact
)
{
	char *nsrc = NULL;
	register const char *frame = NULL;
	register char *msrc = NULL;
	register size_t sv, mv = 1 * infact;

	/*
	 * Existing Newline count, and source length.
	 */
	for (frame = src; *frame != '\0'; ++frame)
		if (*frame == '\n')
			mv += infact;

	sv = frame - src;
	mv += sv + 1;
	nsrc = msrc = malloc(mv+fpad+lpad);

	msrc += fpad;

	/*
	 * Copy src to msrc with a new tab(s) following each newline.
	 */
	*msrc++ = '\t';
	for (frame = src; *frame != '\0'; ++frame)
	{
		*msrc = *frame;

		if (*(msrc++) == '\n')
			for (sv = infact; sv > 0; --sv, ++msrc)
				*msrc = '\t';
	}
	*msrc = '\0';

	return(nsrc);
}

/*
 * transform - make a code fragment a real live function
 */
#define FUNCENC "# -*- encoding: utf-8 -*-\n"
#define FUNCNAME "PGFunction"
#define FUNCDEF FUNCENC "def " FUNCNAME "(self, args):\n"
#define FUNCDEFLEN sizeof(FUNCDEF)
static char *
transform(char *src)
{
	char *msrc;
	unsigned int msrcl;

	msrc = indent(src, FUNCDEFLEN - 1, 0, 1);
	if (msrc == NULL) return(NULL);

	memcpy(msrc, FUNCDEF, FUNCDEFLEN - 1);
	msrcl = strlen(msrc) - 1;
	if (msrc[msrcl] == '\t')
		msrc[msrcl] = '\n';

	return(msrc);
}

/*
 * linecache_update - handle pg_proc oids
 */
static PyObj lc_updatecache;
static PyObj lc_cache;
static PyObj
linecache_update(PyObj self, PyObj args)
{
	PyObj rob = NULL;
	char *filename;
	Oid fn_oid;

	if (!PyArg_ParseTuple(args, "s", &filename))
		return(NULL);

	if (filename[0] == '/')
		filename = filename + 1;

	fn_oid = DatumGetObjectId(strtol(filename, NULL, 10));
	if (fn_oid != InvalidOid)
	{
		char *msrc = NULL;
		char *fn = "PostgreSQL Function";

		if (PyMapping_HasKeyString(lc_cache, filename))
			PyDict_DelItemString(lc_cache, filename);

		PG_TRY();
		{
			Relation proc;
			HeapTuple ht;
			Datum srctext;
			char *srctxt;
			bool isnull = false;

			proc = RelationIdGetRelation(ProcedureRelationId);
			if (!RelationIsValid(proc))
				ereport(ERROR, (
					errmsg("failed to get pg_proc Relation"),
					errdetail("RelationIdGetRelation(ProcedureRelationId = %d)",
						ProcedureRelationId)
				));

			ht = SearchSysCache(PROCOID, ObjectIdGetDatum(fn_oid), 0, 0, 0);
			if (ht == NULL)
			{
				ereport(ERROR, (
					errmsg("failed to get procedure entry with oid \"%d\"", fn_oid)
				));
			}
			srctext = fastgetattr(ht, Anum_pg_proc_prosrc,
							RelationGetDescr(proc), &isnull
						);
			RelationClose(proc);

			srctxt = (char *) DirectFunctionCall1(textout, srctext);
			msrc = transform(srctxt);
			pfree(srctxt);
			if (msrc == NULL)
			{
				ereport(ERROR, (
					errmsg("failed to transform source for linecache")
				));
			}
			ReleaseSysCache(ht);
		}
		PG_CATCH();
		{
			if (msrc != NULL) free(msrc);
			PyErr_SetString(PyExc_IOError, "couldn't get procedure source");
		}
		PG_END_TRY();

		if (PyErr_Occurred())
			return(NULL);

		rob = PyList_New(0);
		if (rob != NULL)
		{
			char *cur;
			PyObj cache;

			/*
			 * For each line, append the PyString into rob.
			 */
			for (cur = msrc; cur[0] != '\0';)
			{
				int linelen = 0;
				PyObj line;

				while (cur[linelen] != '\n' && cur[linelen] != '\0')
					++linelen;
				if (cur[linelen] == '\n')
					++linelen;

				line = PyString_FromStringAndSize(cur, linelen);
				PyList_Append(rob, line);
				cur = cur + linelen;
			}

			cache = Py_BuildValue("(iiOs)", strlen(msrc), 0, rob, fn);
			if (cache != NULL)
				PyDict_SetItemString(lc_cache, filename, cache);
		}
		free(msrc);
	}
	else
	{
		rob = PyObject_Call(lc_updatecache, args, NULL);
	}

	return(rob);
}
static PyMethodDef linecache_update_def = {
	"pg_linecache_update", (PyCFunction) linecache_update, METH_VARARGS,
	"custom updater allows linecache to see PostgreSQL pg_proc files"
};
static PyObj linecache_update_obj;
static int
initialize_linecache(void)
{
	PyObj lc_mod = PyImport_ImportModule("linecache");
	if (lc_mod == NULL) return(-1);

	if (lc_updatecache == NULL)
	{
		lc_updatecache = PyObject_GetAttrString(lc_mod, "updatecache");
		if (lc_updatecache == NULL) goto fail;
	}

	if (lc_cache == NULL)
	{
		lc_cache = PyObject_GetAttrString(lc_mod, "cache");
		if (lc_cache == NULL) goto fail;
	}

	if (linecache_update_obj == NULL)
	{
		linecache_update_obj = PyCFunction_New(&linecache_update_def, lc_mod);
		if (linecache_update_obj == NULL) goto fail;
	}

	if (PyObject_SetAttrString(lc_mod, "updatecache", linecache_update_obj) < 0)
		goto fail;

	Py_DECREF(lc_mod);
	return(0);
fail:
	Py_DECREF(lc_mod);
	return(-1);
}

static PyObj
warnings_showwarning(PyObj self, PyObj args, PyObj kw)
{
	static char *kwords[] = {
		"message", "category", "filename", "lineno", "file", NULL
	};
	PyObj msg, cat, file, obstr;
	char *filename;
	long lineno;

	if (!PyArg_ParseTupleAndKeywords(args, kw, "OOsd|O", kwords,
			&msg, &cat, &filename, &lineno, &file))
		return(NULL);

	obstr = PyObject_Str(msg);
	if (obstr != NULL)
	{
		errstart(WARNING, filename, lineno, PG_FUNCNAME_MACRO);
		errmsg(PyString_AS_STRING(obstr));
		errfinish(0);
		Py_DECREF(obstr);
	}
	else
		return(NULL);

	Py_INCREF(Py_None);
	return(Py_None);
}
static PyMethodDef warnings_showwarning_def = {
	"pg_warnings_showwarning",
	(PyCFunction) warnings_showwarning,
	METH_VARARGS|METH_KEYWORDS,
	"custom showwarning displays the warning using ereport"
};

/*
 * tbtostr - Python Traceback To String
 */
static char *
tbtostr(PyObj tb)
{
	static PyObj tb_print_tb = NULL;
	static PyObj StringIO = NULL;

	PyObj io, ob;
	char *rs = NULL;

	Assert(tb != NULL && PyTraceBack_Check(tb));

	if (tb_print_tb == NULL)
	{
		PyObj mod_traceback = PyImport_ImportModule("traceback");
		if (mod_traceback == NULL) goto fail;

		tb_print_tb = PyObject_GetAttrString(mod_traceback, "print_tb");
		Py_DECREF(mod_traceback);
		if (!tb_print_tb) goto fail;
	}

	if (StringIO == NULL)
	{
		PyObj mod_stringio = PyImport_ImportModule("StringIO");
		if (mod_stringio == NULL) goto fail;

		StringIO = PyObject_GetAttrString(mod_stringio, "StringIO");
		Py_DECREF(mod_stringio);
		if (!StringIO) goto fail;
	}
	
	io = Py_Call(StringIO);
	if (io == NULL) goto fail;

	ob = Py_Call(tb_print_tb, tb, Py_None, io);
	if (ob == NULL)
		goto DECREF_io;
	else
		Py_DECREF(ob);

	ob = PyObject_CallMethod(io, "seek", "i", 0);
	if (ob == NULL)
		goto DECREF_io;
	else
		Py_DECREF(ob);

	ob = PyObject_CallMethod(io, "read", "");
	if (ob == NULL)
		goto DECREF_io;

	if (!PyString_Check(ob))
	{
		Py_DECREF(ob);
		goto DECREF_io;
	}

	rs = strdup(PyString_AS_STRING(ob));
	Py_DECREF(ob);
	Py_DECREF(io);

	return(rs);
DECREF_io:
	Py_DECREF(io);
fail:
	/*
	 * Merely clear any errors generated from traceback rendering.
	 */
	if (PyErr_Occurred()) PyErr_Clear();
	return(NULL);
}

/*
 * errorstr
 *
 *	This function constructs an error string that correlates with
 * the current python exception.
 */
static char *
errorstr(void)
{
	char *vstr = "NULL Exception Instance",
		  *estr = "NULL Exception",
		  *tbstr = NULL;
	PyObj e = NULL, v = NULL, tb = NULL;
	PyObj epystr = NULL, vpystr = NULL;

	char *tbstr_header = "";
	char *errstr = NULL;
	size_t errstrlen;

	Assert(PyErr_Occurred() != NULL);

	PyErr_Fetch(&e, &v, &tb);
	PyErr_NormalizeException(&e, &v, &tb);

	if (e != NULL)
	{
		epystr = PyObject_Str(e);
		Py_DECREF(e);
		if (epystr)
			estr = PyString_AS_STRING(epystr);
	}

	if (v != NULL)
	{
		vpystr = PyObject_Str(v);
		Py_DECREF(v);
		if (vpystr)
			vstr = PyString_AS_STRING(vpystr);
	}

	if (tb != NULL)
	{
		tbstr = tbtostr(tb);
		Py_DECREF(tb);
	}

	/*
	 * The extra 3 account for the NULL terminator, the ':' and the space.
	 */
	errstrlen = 3 + strlen(estr) + strlen(vstr);
	if (tbstr)
	{
#define traceback_header "(most recent call last):\n"
		tbstr_header = traceback_header;
		errstrlen += (sizeof(traceback_header) - 1) + strlen(tbstr);
#undef traceback_header
	}
	else
		tbstr = "";

	errstr = palloc(errstrlen);
	snprintf(errstr, errstrlen, "%s%s%s: %s", tbstr_header, tbstr, estr, vstr);
	if (tbstr[0] != '\0') free(tbstr);

	Py_XDECREF(epystr);
	Py_XDECREF(vpystr);

	return(errstr);
}
#define PythonExceptionDetail errdetail(errorstr())

unsigned int ist_count;
static PyObj
ist_begin(PyObj self)
{
	static bool warned = false;
	if (warned == false)
	{
		warned = true;
		if (PyErr_Warn(PyExc_DeprecationWarning,
			"BEGIN, RELEASE, and ROLLBACK are scheduled for removal") < 0)
			return(NULL);
	}

	PG_TRY();
	{
		if (ist_count == -1)
		{
			ereport(ERROR,(
				errcode(ERRCODE_SAVEPOINT_EXCEPTION),
				errmsg("internal subtransactions prohibited"),
				errhint("Attempt to begin an IST in a prohibited area.")
			));
		}
		BeginInternalSubTransaction(NULL);
	}
	PYPG_CATCH_END(return(NULL));

	ist_count += 1;
	RETURN_NONE;
}
static PyMethodDef ist_begin_def = {
	"pg_ist_begin",
	(PyCFunction) ist_begin,
	METH_NOARGS,
	"begin an internal subtransaction"
};

static PyObj
ist_release(PyObj self)
{
	PG_TRY();
	{
		if (ist_count == 0 || ist_count == -1)
		{
			ereport(ERROR,(
				errcode(ERRCODE_SAVEPOINT_EXCEPTION),
				errmsg("no current internal subtransaction"),
				errhint("Attempt to release the current IST, when none running.")
			));
		}
		ReleaseCurrentSubTransaction();
	}
	PYPG_CATCH_END(return(NULL));

	ist_count -= 1;
	RETURN_NONE;
}
static PyMethodDef ist_release_def = {
	"pg_ist_release",
	(PyCFunction) ist_release,
	METH_NOARGS,
	"release the current internal subtransaction"
};

static PyObj
ist_rollback(PyObj self)
{
	PG_TRY();
	{
		if (ist_count == 0)
		{
			ereport(ERROR,(
				errcode(ERRCODE_SAVEPOINT_EXCEPTION),
				errmsg("no current internal subtransaction"),
				errhint("Attempt to rollback the current IST, when none running.")
			));
		}
		RollbackAndReleaseCurrentSubTransaction();
	}
	PYPG_CATCH_END(return(NULL));

	ist_count -= 1;
	RETURN_NONE;
}
static PyMethodDef ist_rollback_def = {
	"pg_ist_rollback",
	(PyCFunction) ist_rollback,
	METH_NOARGS,
	"rollback and release the current internal subtransaction"
};

/*
 * ExprContext Callback function to clean up after VPC-SRFs
 */
static PyObj fnExtraCalls = NULL;
static void
srfeccb(Datum arg)
{
	int ind;
	FmgrInfo *fli = (FmgrInfo *) DatumGetPointer(arg);
	PyObj plc = (PyObj) fli->fn_extra;

	ind = PySequence_Index(fnExtraCalls, plc);
	if (ind >= 0)
		PySequence_DelItem(fnExtraCalls, ind);

	fli->fn_extra = NULL;
}

PyObj XactDict;
static void
XactHook(XactEvent xev, void *arg)
{
	switch (xev)
	{
		case XACT_EVENT_COMMIT:
		case XACT_EVENT_ABORT:
			if (fnExtraCalls != NULL)
				while (PyList_GET_SIZE(fnExtraCalls) > 0)
					PySequence_DelItem(fnExtraCalls, 0);
			ist_count = 0;

			PyDict_Clear(XactDict);
			PyGC_Collect();
		break;
	}
}

/*
 * Use the library's on-load facility to initialize Python as soon as it's
 * loaded. Providing a warranty that Python is initialized from any point.
 * Allowing ignorance of the Python initialization state within the initializor
 * and the validator, the only entry points.
 */
#ifdef __APPLE__
__attribute__((constructor))
#endif
void
_init(void)
{
	PyObj ob;

	Py_SetProgramName("Postgres");
	Py_Initialize();

	ob = PyImport_ImportModule("warnings");
	if (ob != NULL)
	{
		PyObj wd, sw;
		wd = PyModule_GetDict(ob);
		Py_DECREF(ob);
		if (wd == NULL) goto showwarning_failure;

		sw = PyCFunction_New(&warnings_showwarning_def, ob);
		if (sw == NULL) goto showwarning_failure;

		if (PyDict_SetItemString(wd, "showwarning", sw) < 0)
		{
			Py_DECREF(sw);
			goto showwarning_failure;
		}
	}
	else
	{
showwarning_failure:
		ereport(INFO, (
			errmsg("failed to overload warnings.showwarning"),
			errhint(
				"Make sure Python's sys.path has the directory containing the "
				"standard warnings module."
			),
			PythonExceptionDetail
		));
	}

	XactDict = PyDict_New();
	RegisterXactCallback(XactHook, NULL);
}
#if defined(_WIN32)
BOOL WINAPI
DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
	switch (fdwReason) {
		case DLL_PROCESS_ATTACH:
			_init();
		break;
	}
	return(TRUE);
}
#endif

static bytea *
mkbytea(const char *buf, Size len)
{
	bytea *ba;
	Size ba_len = 0;

	ba_len = len + VARHDRSZ;
	ba = (bytea *) palloc(ba_len);

	VARATT_SIZEP(ba) = ba_len;
	memcpy(VARDATA(ba), buf, len);

	return(ba);
}

static HeapTuple
pg_proc_probin(HeapTuple procTuple, Datum probin)
{
	Relation rel;
	HeapTuple tup;

	Datum	values[Natts_pg_proc] = {0};
	char nulls[Natts_pg_proc];
	char replace[Natts_pg_proc];

	memset(replace, ' ', Natts_pg_proc);
	memset(nulls, ' ', Natts_pg_proc);

	replace[Anum_pg_proc_probin-1] = 'r';
	values[Anum_pg_proc_probin-1] = probin;

	rel = relation_open(ProcedureRelationId, RowExclusiveLock);
	if (!RelationIsValid(rel))
		ereport(ERROR,(
			errmsg("failed to open pg_proc system catalog")
		));
#if PGV_MM == 80
	tup = heap_modifytuple(procTuple, rel, values, nulls, replace);
#else
	tup = heap_modifytuple(procTuple,
				RelationGetDescr(rel), values, nulls, replace);
#endif
	if (tup == NULL)
		ereport(ERROR,(
			errmsg("failed to modify procedure's tuple")
		));

	simple_heap_update(rel, &tup->t_self, tup);
	CatalogUpdateIndexes(rel, tup);

	heap_close(rel, RowExclusiveLock);
	return(tup);
}

/*
 * compile - make a code object from prosrc
 */
static PyObj
compile(char *src)
{
	PyCompilerFlags flags = {CO_OPTIMIZED};
	PyObj locals = NULL;
	PyObj rob = NULL;
	char *msrc;

	msrc = transform(src);
	if (msrc == NULL) return(NULL);

	locals = PyDict_New();
	if (locals == NULL)
	{
		free(msrc);
		return(NULL);
	}

	PyRun_StringFlags(msrc, Py_file_input, locals, locals, &flags);
	free(msrc);

	if (!PyErr_Occurred())
	{
		rob = PyDict_GetItemString(locals, FUNCNAME);
		if (rob != NULL)
		{
			Assert(PyFunction_Check(rob));
			rob = PyFunction_GET_CODE(rob);
			Py_INCREF(rob);
		}
	}

	Py_DECREF(locals);

	return(rob);
}

static void
PostgresError_FromPythonException(void)
{
	PyObj ob, e, v, tb;

	PyErr_Fetch(&e, &v, &tb);
	PyErr_NormalizeException(&e, &v, &tb);
	errstart(ERROR, __FILE__, __LINE__, PG_FUNCNAME_MACRO);

	ob = PyObject_GetAttrString(v, "code");
	if (PyInt_Check(ob))
		errcode(PyInt_AS_LONG(ob));
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "message");
	if (PyString_Check(ob))
		errmsg(PyString_AS_STRING(ob));
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "detail");
	if (PyString_Check(ob) && tb == NULL)
		errdetail(PyString_AS_STRING(ob));
	else
	{
		char *tbstr = NULL;
		char *obstr = NULL;
		if (PyString_Check(ob))
			obstr = PyString_AS_STRING(ob);
		if (tb != NULL)
			tbstr = tbtostr(tb);

		if (tbstr != NULL)
		{
			if (obstr != NULL)
				errdetail("%s\n(most recent call last):\n%s", obstr, tbstr);
			else
				errdetail("(most recent call last):\n%s", tbstr);
			free(tbstr);
		}
		else if (obstr != NULL)
			errdetail(obstr);
	}
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "context");
	if (PyString_Check(ob))
		errcontext(PyString_AS_STRING(ob));
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "hint");
	if (PyString_Check(ob))
		errhint(PyString_AS_STRING(ob));
	Py_DECREF(ob);

	PyErr_Clear();
	errfinish(0);
}

/*
 * pl_validator
 *		fmgr interface to pl_compile()
 */
PG_FUNCTION_INFO_V1(pl_validator);
Datum
pl_validator(PG_FUNCTION_ARGS)
{
	Oid fn_oid = PG_GETARG_OID(0);
	volatile HeapTuple procTuple;

	Datum srctext;
	bytea *probin;
	char *src;
	bool isnull = false;

	PyObj code, mo;

	Assert(fn_oid != InvalidOid);
	Assert(Py_IsInitialized());
	
	procTuple = SearchSysCacheCopy(PROCOID, fn_oid, 0, 0, 0);
	if (procTuple == NULL)
		ereport(ERROR, (
			errmsg("failed to find procedure in system catalog"),
			errcontext("Python function validation"),
			errdetail("SearchSysCacheCopy(PROCOID = %d, fn_oid = %d, 0, 0, 0)",
				PROCOID, fn_oid)
		));

	PG_TRY();
	{
		Relation proc;
		proc = RelationIdGetRelation(ProcedureRelationId);
		if (!RelationIsValid(proc))
			ereport(ERROR, (
				errmsg("failed to get pg_proc Relation"),
				errdetail("RelationIdGetRelation(ProcedureRelationId = %d)",
					ProcedureRelationId)
			));
		srctext = fastgetattr(
						procTuple,
						Anum_pg_proc_prosrc,
						RelationGetDescr(proc),
						&isnull
					);
		RelationClose(proc);

		if (isnull == true || srctext == 0)
			ereport(ERROR,(
				errmsg("failed to get valid source from pg_proc tuple \"%d\"",
					HeapTupleGetOid(procTuple))
			));

		srctext = DirectFunctionCall1(textout, srctext);
		if (PG_UTF8 != GetDatabaseEncoding())
			src = (char *) pg_do_encoding_conversion(
				(unsigned char *) DatumGetPointer(srctext),
				strlen(DatumGetPointer(srctext)),
				GetDatabaseEncoding(), PG_UTF8
			);
		else
			src = DatumGetPointer(srctext);
		if (PointerGetDatum(src) != srctext)
			pfree(DatumGetPointer(srctext));
		code = compile(src);
		pfree(src);

		if (code == NULL)
			ereport(ERROR,(
				errmsg("procedure compilation failure"),
				PythonExceptionDetail
			));

		/*
		 * Set the filename to a decimal string from the Oid. This is how our
		 * linecache_update will identify that the file comes from the database.
		 */
		Py_DECREF(((PyCodeObject *) code)->co_filename);
		((PyCodeObject *) code)->co_filename = PyString_FromFormat(
			"/%d", HeapTupleGetOid(procTuple)
		);

		Py_DECREF(((PyCodeObject *) code)->co_name);
		src = format_procedure(fn_oid);
		((PyCodeObject *) code)->co_name = PyString_FromString(src);
		pfree(src);

		mo = PyMarshal_WriteObjectToString(code, 0);
		Py_DECREF(code);

		if (mo == NULL)
			ereport(ERROR,(
				errmsg("failed to marshal function bytecode"),
				PythonExceptionDetail
			));

		probin = mkbytea(PyString_AS_STRING(mo), PyString_Size(mo));
		Py_DECREF(mo);

		if (probin == NULL)
			ereport(ERROR, (
				errmsg("failed to make probin bytea from function bytecode")
			));

		pg_proc_probin(procTuple, PointerGetDatum(probin));
		pfree(probin);
	}
	PG_CATCH();
	{
		heap_freetuple(procTuple);
		PG_RE_THROW();
	}
	PG_END_TRY();

	heap_freetuple(procTuple);

	/*
	 * The validator may be ran prior to the initialization of the
	 * procedural language. This causes a problem here as the C Interface cannot
	 * be resolved unless the language has been initialized. To solve this issue,
	 * only attempt to delete the item in the cache if PyPgCI_Root_ has been
	 * established. If it's NULL, the language hasn't been loaded and the cache
	 * is empty anyways, so this is "optimization". -jwp 2006.2.14
	 */
	if (PyPgCI_Root_ && PyPgFunction_DelCached(fn_oid) != 0)
		ereport(WARNING, (
			errmsg("failed to delete function in cache(%d)", (int) fn_oid),
			PythonExceptionDetail
		));

	return(0);
}

static Datum
pull_trigger(PG_FUNCTION_ARGS)
{
	PyObj tp, rob;
	Datum rd = 0;

	tp = PyPgTriggerPull_New(fcinfo);
	if (tp == NULL)
		ereport(ERROR,(
			errmsg("failed to create a TriggerPull object "
				"from the FunctionCallInfo CObject"),
			PythonExceptionDetail
		));

	rob = Py_Call(tp);
	Py_DECREF(tp);
	if (rob == NULL)
	{
		if (PyErr_ExceptionMatches(PyExc_PostgresError))
			PostgresError_FromPythonException();
		else
			ereport(ERROR,(
				errmsg("Python exception occurred"),
				PythonExceptionDetail
			));
	}

	if (rob == Py_None || rob == Py_False)
	{
		/* Skip */
		rd = 0;
	}
	else if (rob == Py_True)
	{
		/* Continue as planned */
		TriggerData *td = (TriggerData *) (fcinfo->context);
		rd = PointerGetDatum(
			td->tg_newtuple ? td->tg_newtuple : td->tg_trigtuple
		);
	}
	else
	{
		/* Replacement HeapTuple returned */
		MemoryContext former = CurrentMemoryContext;
		HeapTuple ht;
		Assert(PyPgHeapTuple_Check(rob));

		MemoryContextSwitchTo(fcinfo->flinfo->fn_mcxt);
		ht = heap_copytuple(PyPgHeapTuple_FetchHeapTuple(rob));
		MemoryContextSwitchTo(former);
		rd = PointerGetDatum(ht);
	}
	Py_DECREF(rob);

	return(rd);
}

static Datum
call_function(PG_FUNCTION_ARGS)
{
	Node *ri = fcinfo->resultinfo;
	ReturnSetInfo *rsi =
		(ri && IsA(ri, ReturnSetInfo)) ? (ReturnSetInfo *) ri : NULL;
	PyObj plc, rob;
	Datum rd;

	plc = (PyObj) fcinfo->flinfo->fn_extra;
	Py_XINCREF(plc);
	if (plc == NULL)
	{
		plc = PyPgProceduralCall_New(fcinfo);
		if (plc == NULL)
			ereport(ERROR,(
				errmsg("failed to create a ProceduralCall object "
					"from the FunctionCallInfo CObject"),
				PythonExceptionDetail
			));
		rob = Py_Call(plc);
	}
	else if (rsi != NULL)
	{
		/* This is VPC-SRF as we have the cache of the last call. */
		/* This is only reached on subsequent calls. (After the first) */

		/* SRFs don't get new arguments, so don't reprocess FCInfo */
		PyPgProceduralCall_FixFCInfo(plc, fcinfo);
		rob = Py_Call(plc);
	}
	else
	{
		PyObj iter = PyPgCall_FetchReturned(plc);

		if (PyObject_HasAttrString(iter, "send"))
		{
			PyObj tdo = PyPgFunction_FetchInput(PyPgCall_FetchObject(plc));
			TupleDesc td = PyPgTupleDesc_FetchTupleDesc(tdo);
			PyObj args;
			if (td->natts > 0)
			{
				HeapTuple ht;
				Datum *fca = fcinfo->arg;
				bool *fcn = fcinfo->argnull;
				ht = HeapTuple_FromTupleDescAndDatumNulls(td, fca, fcn);
				args = PyPgHeapTuple_New(tdo, ht);
			}
			else
			{
				args = EmptyPyPgHeapTuple;
				Py_INCREF(args);
			}
			rob = PyObject_CallMethod(iter, "send", "O", args);
			Py_DECREF(args);
		}
		else
			rob = Py_Call(plc);
	}

	if (rob == NULL)
	{
		if (PyErr_ExceptionMatches(PyExc_PostgresError))
			PostgresError_FromPythonException();
		else
			ereport(ERROR,(
				errmsg("Python exception occurred"),
				PythonExceptionDetail
			));
	}
	if (rsi)
	{
		if (rsi->returnMode == SFRM_ValuePerCall)
		{
			if (rsi->isDone == ExprEndResult)
			{
				if (fcinfo->flinfo->fn_extra != NULL)
				{
					int ind;
					UnregisterExprContextCallback(rsi->econtext,
							srfeccb, PointerGetDatum(fcinfo->flinfo));
					ind = PySequence_Index(fnExtraCalls, plc);
					PySequence_DelItem(fnExtraCalls, ind);
					fcinfo->flinfo->fn_extra = NULL;
				}
			}
			else if (fcinfo->flinfo->fn_extra == NULL)
			{
				RegisterExprContextCallback(rsi->econtext,
						srfeccb, PointerGetDatum(fcinfo->flinfo));
				PyList_Append(fnExtraCalls, plc);
				fcinfo->flinfo->fn_extra = plc;
			}
			fcinfo->flinfo->fn_mcxt = rsi->econtext->ecxt_per_query_memory;
		}
	}
	else if (fcinfo->flinfo->fn_extra == NULL
		&& PyIter_Check(PyPgCall_FetchReturned(plc)))
	{
		int err;
		err = PyList_Append(fnExtraCalls, plc);
		if (err < 0)
		{
			Py_DECREF(plc);
			ereport(ERROR,(
				errmsg("failed to store call in fnExtra"),
				PythonExceptionDetail
			));
		}
		fcinfo->flinfo->fn_extra = plc;
	}

	if (rob == Py_None)
	{
		rd = 0;
		fcinfo->isnull = true;
	}
	else
	{
		TypeCacheEntry *tc = PyPgObject_FetchTypeCache(rob);
		MemoryContext former;
		rd = PyPgObject_FetchDatum(rob);

		former = MemoryContextSwitchTo(fcinfo->flinfo->fn_mcxt);
		rd = datumCopy(rd, tc->typbyval, tc->typlen);
		MemoryContextSwitchTo(former);
	}

	Py_DECREF(rob);
	Py_DECREF(plc);

	return(rd);
}

static Datum
pl_executor(PG_FUNCTION_ARGS)
{
	Datum rd = 0;

	if (CALLED_AS_TRIGGER(fcinfo))
		rd = pull_trigger(fcinfo);
	else
		rd = call_function(fcinfo);

	return(rd);
}

static PyObj InterfaceModule = NULL;

static void *
initialize_main(void)
{
	PyObj ob;

	ob = PyImport_AddModule("__main__");
	if (ob == NULL)
	{
		ereport(ERROR, (
			errmsg("failed to get __main__ dictionary")
		));
	}
	else
	{
		PyObj item;
		ob = PyModule_GetDict(ob);
		PyDict_SetItemString(ob, "Postgres", InterfaceModule);

		item = PyCFunction_New(&ist_begin_def, NULL);
		PyDict_SetItemString(ob, "BEGIN", item);

		item = PyCFunction_New(&ist_release_def, NULL);
		PyDict_SetItemString(ob, "RELEASE", item);

		item = PyCFunction_New(&ist_rollback_def, NULL);
		PyDict_SetItemString(ob, "ROLLBACK", item);

		Py_INCREF(XactDict);
		PyDict_SetItemString(ob, "XD", XactDict);
	}

	return(NULL);
}

static void *
initialize_unicode(void)
{
	if (PyUnicode_SetDefaultEncoding(
				PyEncoding_FromPgEncoding(GetDatabaseEncoding())) < 0)
		ereport(ERROR,(
			errmsg("failed to set the default PyUnicode encoding"),
			PythonExceptionDetail
		));
	return(initialize_main);
}

static void *
initialize_interface(void)
{
	PyObj ob;

	if (InterfaceModule == NULL)
	{
		/*
		 * XXX: Use GUC Variable Here. (?)
		 */
		InterfaceModule = PyImport_ImportModule(DEFAULT_POSTGRES_MODULE);
		if (InterfaceModule == NULL)
			ereport(ERROR,(
				errmsg("failed to import Postgres interface module"),
				PythonExceptionDetail
			));
	}

	ob = PyObject_GetAttrString(InterfaceModule, PyPgCI_CObjectName);
	if (ob == NULL)
		ereport(ERROR,(
			errmsg("failed to get C Interface from extension module"),
			PythonExceptionDetail
		));
	PyPgCI_Initialize(PyCObject_AsVoidPtr(ob));
	Py_DECREF(ob);
	PyPgCI.PL.CodeFromData = PyMarshal_ReadObjectFromString;
	return(initialize_unicode);
}

static void *
initialize_python(void)
{
	if (!Py_IsInitialized())
	{
		_init();
		if (!Py_IsInitialized())
			ereport(ERROR,(
				errmsg("failed to initialize Python")
			));
	}

	return(initialize_interface);
}

Datum pl_handler(PG_FUNCTION_ARGS);
static PGFunction pl;
static Datum
pl_initializor(PG_FUNCTION_ARGS)
{
	static void * (*init_state)(void) = initialize_python;

	while (init_state != NULL)
		init_state = init_state();

	if (initialize_linecache() < 0)
		ereport(INFO, (
			errmsg(
				"failed to overload \"linecache.updatecache\" "
				"for procedure tracebacks"
			),
			PythonExceptionDetail
		));

	if (fnExtraCalls == NULL)
	{
		fnExtraCalls = PyList_New(0);
		if (fnExtraCalls == NULL)
		{
			ereport(ERROR, (
				errmsg("failed to create fnExtraCalls list")
			));
		}
	}

	PyPgCI.PL.Handler = pl_handler;

	/*
	 * Everything is initialized, and will be for the duration of this process.
	 * Set the handler to the executor, and call the executor.
	 */
	pl = pl_executor;
	return(pl_executor(fcinfo));
}
static PGFunction pl = pl_initializor;

/*
 * pl_handler
 *
 * This function just calls pl, which is set to a function which represents
 * its current demand for the function executor.
 *
 * Right now, it is used to avoid a conditional on post initialization.
 */
PG_FUNCTION_INFO_V1(pl_handler);
Datum pl_handler(PG_FUNCTION_ARGS)
{
	static long depth = 0;
	Datum rd;
	MemoryContext fn_mcxt = fcinfo->flinfo->fn_mcxt;

	Assert(depth >= 0);
	++depth;
	PG_TRY();
	{
		rd = pl(fcinfo);
	}
	PG_CATCH();
	{
		--depth;
		PG_RE_THROW();
	}
	PG_END_TRY();
	--depth;

	fcinfo->flinfo->fn_mcxt = fn_mcxt;

	return(rd);
}
/*
 * vim: ts=3:sw=3:noet:
 */
