#!/usr/bin/env tarantool
test = require("sqltester")
test:plan(28 * 4)

local prefix = "unicode-collation-"

local function insert_into_table(tbl_name, data)
    local sql = string.format("INSERT INTO %s VALUES ", tbl_name)
    --local values = {}
    for _, item in ipairs(data) do
        local value = "('"..item.."')"
        local e = sql .. value
        box.execute(e)
    end
end


local collation_entries =
{
    {   -- Afrikaans case sensitive
        "unicode_af_s3", "unstable",
        {"a","A","á","Á","â","Â","b","B","c","C","d","D","e","E","é","É",
         "è","È","ê","Ê","ë","Ë","f","F","g","G","h","H","i","I","î","Î",
         "ï","Ï","j","J","k","K","l","L","m","M","n","N","ŉ","o","O",
         "ô","Ô","ö","Ö","p","P","q","Q","r","R","s","S","t","T","u","U",
         "û","Û","v","V","w","W","x","X","y","Y","z","Z"}},
    {
        -- Albanian
        "unicode_sq_s3", "unstable",
        {"a","A","b","B","c","C","ç","Ç","ḉ","Ḉ","d","D","dh","Dh",
         "DH","dĥ","Dĥ","DĤ","dȟ","Dȟ","DȞ","dḧ","Dḧ","DḦ","dḣ",
         "Dḣ","DḢ","dḩ","Dḩ","DḨ","dḥ","Dḥ","DḤ","dḫ","Dḫ","DḪ",
         "dẖ","Dẖ","e","E","ë","Ë","ȩ̈","Ȩ̈","ę̈","Ę̈","ẹ̈","Ẹ̈","ḙ̈",
         "Ḙ̈","ḛ̈","Ḛ̈","f","F","g","G","gj","Gj","GJ","gĵ","Gĵ",
         "GĴ","gǰ","Gǰ","h","H","i","I","j","J","k","K","l","L",
         "ll","Ll","LL","lĺ","Lĺ","LĹ","lľ","Lľ","LĽ","lļ","Lļ",
         "LĻ","lḷ","Lḷ","LḶ","lḹ","Lḹ","LḸ","lḽ","Lḽ","LḼ","lḻ",
         "Lḻ","LḺ","m","M","n","N","nj","Nj","NJ","nĵ","Nĵ","NĴ",
         "nǰ","Nǰ","o","O","p","P","q","Q","r","R","rr","Rr","RR",
         "rŕ","Rŕ","RŔ","rř","Rř","RŘ","rṙ","Rṙ","RṘ","rŗ","Rŗ",
         "RŖ","rȑ","Rȑ","RȐ","rȓ","Rȓ","RȒ","rṛ","Rṛ","RṚ","rṝ",
         "Rṝ","RṜ","rṟ","Rṟ","RṞ","s","S","sh","Sh","SH","sĥ","Sĥ",
         "SĤ","sȟ","Sȟ","SȞ","sḧ","Sḧ","SḦ","sḣ","Sḣ","SḢ","sḩ",
         "Sḩ","SḨ","sḥ","Sḥ","SḤ","sḫ","Sḫ","SḪ","sẖ","Sẖ","t",
         "T","th","Th","TH","tĥ","Tĥ","TĤ","tȟ","Tȟ","TȞ","tḧ",
         "Tḧ","TḦ","tḣ","Tḣ","TḢ","tḩ","Tḩ","TḨ","tḥ","Tḥ","TḤ",
         "tḫ","Tḫ","TḪ","tẖ","Tẖ","u","U","v","V","x","X","xh",
         "Xh","XH","xĥ","Xĥ","XĤ","xȟ","Xȟ","XȞ","xḧ","Xḧ","XḦ",
         "xḣ","Xḣ","XḢ","xḩ","Xḩ","XḨ","xḥ","Xḥ","XḤ","xḫ","Xḫ",
         "XḪ","xẖ","Xẖ","y","Y","z","Z","zh","Zh","ZH","zĥ","Zĥ",
         "ZĤ","zȟ","Zȟ","ZȞ","zḧ","Zḧ","ZḦ","zḣ","Zḣ","ZḢ","zḩ",
         "Zḩ","ZḨ","zḥ","Zḥ","ZḤ","zḫ","Zḫ","ZḪ","zẖ","Zẖ"}},
    {
        -- Amharic
        "unicode_am_s3", "stable",
        {"ሀ","ሁ","ሂ","ሃ","ሄ","ህ","ሆ","ለ","ሉ","ሊ","ላ","ሌ","ል","ሎ","ሏ","ሐ",
         "ሑ","ሒ","ሓ","ሔ","ሕ","ሖ","ሗ","መ","ሙ","ሚ","ማ","ሜ","ም","ሞ",
         "ሟ","ሠ","ሡ","ሢ","ሣ","ሤ","ሥ","ሦ","ሧ","ረ","ሩ","ሪ","ራ","ሬ","ር",
         "ሮ","ሯ","ሰ","ሱ","ሲ","ሳ","ሴ","ስ","ሶ","ሷ","ሸ","ሹ","ሺ","ሻ","ሼ","ሽ",
         "ሾ","ሿ","ቀ","ቁ","ቂ","ቃ","ቄ","ቅ","ቆ","ቈ","ቊ","ቋ","ቌ","ቍ","በ",
         "ቡ","ቢ","ባ","ቤ","ብ","ቦ","ቧ","ቨ","ቩ","ቪ","ቫ","ቬ","ቭ","ቮ","ቯ",
         "ተ","ቱ","ቲ","ታ","ቴ","ት","ቶ","ቷ","ቸ","ቹ","ቺ","ቻ","ቼ","ች","ቾ",
         "ቿ","ኀ","ኁ","ኂ","ኃ","ኄ","ኅ","ኆ","ኈ","ኊ","ኋ","ኌ","ኍ","ነ","ኑ",
         "ኒ","ና","ኔ","ን","ኖ","ኗ","ኘ","ኙ","ኚ","ኛ","ኜ","ኝ","ኞ","ኟ","አ","ኡ",
         "ኢ","ኣ","ኤ","እ","ኦ","ኧ","ከ","ኩ","ኪ","ካ","ኬ","ክ","ኮ","ኰ","ኲ",
         "ኳ","ኴ","ኵ","ኸ","ኹ","ኺ","ኻ","ኼ","ኽ","ኾ","ወ","ዉ","ዊ","ዋ","ዌ",
         "ው","ዎ","ዐ","ዑ","ዒ","ዓ","ዔ","ዕ","ዖ","ዘ","ዙ","ዚ","ዛ","ዜ","ዝ",
         "ዞ","ዟ","ዠ","ዡ","ዢ","ዣ","ዤ","ዥ","ዦ","ዧ","የ","ዩ","ዪ","ያ",
         "ዬ","ይ","ዮ","ደ","ዱ","ዲ","ዳ","ዴ","ድ","ዶ","ዷ","ጀ","ጁ","ጂ","ጃ",
         "ጄ","ጅ","ጆ","ጇ","ገ","ጉ","ጊ","ጋ","ጌ","ግ","ጎ","ጐ","ጒ","ጓ","ጔ",
         "ጕ","ጠ","ጡ","ጢ","ጣ","ጤ","ጥ","ጦ","ጧ","ጨ","ጩ","ጪ","ጫ","ጬ",
         "ጭ","ጮ","ጯ","ጰ","ጱ","ጲ","ጳ","ጴ","ጵ","ጶ","ጷ","ጸ","ጹ","ጺ","ጻ",
         "ጼ","ጽ","ጾ","ጿ","ፀ","ፁ","ፂ","ፃ","ፄ","ፅ","ፆ","ፈ","ፉ","ፊ","ፋ","ፌ",
         "ፍ","ፎ","ፏ","ፐ","ፑ","ፒ","ፓ","ፔ","ፕ","ፖ","ፗ"}},
    {
        -- Assamese
        "unicode_as_s3", "unstable",
        {"়","অ","আ","ই","ঈ","উ","ঊ","ঋ","এ","ঐ","ও","ঔ","ং ","ঁ ","ঃ ",
         "ক","খ","গ","ঘ","ঙ","চ","ছ","জ","ঝ","ঞ","ট","ঠ","ড","ড়","ঢ","ঢ়",
         "ণ","ৎ ","ত","থ","দ","ধ","ন","প","ফ","ব","ভ","ম","য","য়","ৰ",
         "ল","ৱ","শ","ষ","স","হ","ক্ষ ","া","ি","ী","ু","ূ","ৃ","ে","ৈ",
         "ো","ৌ","্"}},
    {
        -- Austrian German (umlaut primary greater)
        "unicode_de_AT_phonebook_s3", "unstable",
        {"a","A","ä","Ä","ą̈","Ą̈","ǟ","Ǟ","ạ̈","Ạ̈","ḁ̈","Ḁ̈","b","B",
         "c","C","d","D","e","E","f","F","g","G","h","H","i","I",
         "j","J","k","K","l","L","m","M","n","N","o","O","ö","Ö",
         "ǫ̈","Ǫ̈","ȫ","Ȫ","ơ̈","Ơ̈","ợ̈","Ợ̈","ọ̈","Ọ̈","p","P","q","Q",
         "r","R","s","S","ss","ß","ẞ","t","T","u","U","ü","Ü","ǘ",
         "Ǘ","ǜ","Ǜ","ǚ","Ǚ","ų̈","Ų̈","ǖ","Ǖ","ư̈","Ư̈","ự̈","Ự̈","ụ̈",
         "Ụ̈","ṳ̈","Ṳ̈","ṷ̈","Ṷ̈","ṵ̈","Ṵ̈","v","V","w","W","x","X","y",
         "Y","z","Z"}},

    {
        -- Azerbaijani
        "unicode_az_s3", "unstable",
        {"a ","A ","b ","B ","c ","C ","ç ","Ç ","ḉ ","Ḉ ","d ","D ","e ",
         "E ","ə ","Ə ","f ","F ","g ","G ","ğ ","Ğ ","ģ̆ ","Ģ̆ ","h ",
         "H ","x ","X ","ẍ ","Ẍ ","ẋ ","Ẋ ","ı ","I ","Í ","Ì ","Ĭ ",
         "Î ","Ǐ ","Ï ","Ḯ ","Ĩ ","Į ","Ī ","Ỉ ","Ȉ ","Ȋ ","Ị ","Ḭ ",
         "i ","İ ","Į̇ ","Ị̇ ","Ḭ̇ ","j ","J ","k ","K ","q ","Q ","l ",
         "L ","m ","M ","n ","N ","o ","O ","ö ","Ö ","ǫ̈ ","Ǫ̈ ","ȫ ",
         "Ȫ ","ơ̈ ","Ơ̈ ","ợ̈ ","Ợ̈ ","ọ̈ ","Ọ̈ ","p ","P ","r ","R ","s ",
         "S ","ş ","Ş ","t ","T ","u ","U ","ü ","Ü ","ǘ ","Ǘ ","ǜ ",
         "Ǜ ","ǚ ","Ǚ ","ų̈ ","Ų̈ ","ǖ ","Ǖ ","ư̈ ","Ư̈ ","ự̈ ","Ự̈ ","ụ̈ ",
         "Ụ̈ ","ṳ̈ ","Ṳ̈ ","ṷ̈ ","Ṷ̈ ","ṵ̈ ","Ṵ̈ ","v ","V ","y ","Y ","z ",
         "Z ","Ẉ","w ","W ","ẃ ","Ẃ ","ẁ ","Ẁ ","ŵ ","Ŵ ","ẘ ","ẅ ","Ẅ ",
         "ẇ ","Ẇ ","ẉ "}},
    {
        -- Belarusian
        "unicode_be_s3", "unstable",
        {"а","А","б","Б","в","ᲀ","В","г","Г","д","ᲁ","Д","дж","дз","е",
         "Е","ё","Ё","ж","Ж","з","З","і","І","й","Й","к","К","л",
         "Л","м","М","н","Н","о","ᲂ","О","п","П","р","Р","с","ᲃ",
         "С","т","Т","у","У","ў","Ў","ф","Ф","х","Х","ц",
         "Ц","ч","Ч","ш","Ш","ы","Ы","ь","Ь","э","Э","ю","Ю","я","Я"}},
    {
        -- Czech
        "unicode_cs_s3", "unstable",
        {"a","A","á","Á","b","B","c","C","č","Č","ç̌","Ç̌","d","D",
         "ď","Ď","e","E","é","É","ě","Ě","f","F","g","G","h","H",
         "ch","cH","Ch","CH","cĥ","cĤ","Cĥ","CĤ","cȟ","cȞ","Cȟ",
         "CȞ","cḧ","cḦ","Cḧ","CḦ","cḣ","cḢ","Cḣ","CḢ","cḩ","cḨ",
         "Cḩ","CḨ","cḥ","cḤ","Cḥ","CḤ","cḫ","cḪ","Cḫ","CḪ","cẖ",
         "Cẖ","i","I","í","Í","j","J","k","K","l","L","m","M","n",
         "N","ň","Ň","o","O","ó","Ó","p","P","q","Q","r","R","ř",
         "Ř","ŗ̌","Ŗ̌","ṛ̌","Ṛ̌","ṟ̌","Ṟ̌","s","S","š","Š","ṧ","Ṧ","ş̌",
         "Ş̌","ṣ̌","Ṣ̌","ș̌","Ș̌","t","T","ť","Ť","u","U","ú","Ú","ů",
         "Ů","v","V","w","W","x","X","y","Y","ý","Ý","z","Z","ž",
         "Ž","ẓ̌","Ẓ̌","ẕ̌","Ẕ̌"}},
    {
        -- Danish
        "unicode_da_s3", "unstable",
        {"A","a","B","b","C","c","D","d","Đ","đ","Ð","ð","E","e",
         "F","f","G","g","H","h","I","i","J","j","K","k","L","l",
         "M","m","N","n","O","o","Œ","œ","P","p","Q","q","R","r",
         "S","s","T","t","Þ","þ","U","u","V","v","W","w","X","x",
         "Y","y","Ü","ü","Ǘ","ǘ","Ǜ","ǜ","Ǚ","ǚ","Ų̈","ų̈","Ǖ","ǖ",
         "Ư̈","ư̈","Ự̈","ự̈","Ụ̈","ụ̈","Ṳ̈","ṳ̈","Ṷ̈","ṷ̈","Ṵ̈","ṵ̈","Ű","ű",
         "Ų̋","ų̋","Ư̋","ư̋","Ự̋","ự̋","Ụ̋","ụ̋","Ṳ̋","ṳ̋","Ṷ̋","ṷ̋","Ṵ̋","ṵ̋",
         "Z","z","Æ","æ","Ǽ","ǽ","Ǣ","ǣ","Ä","ä","Ą̈","ą̈","Ǟ","ǟ",
         "Ạ̈","ạ̈","Ḁ̈","ḁ̈","Ø","ø","Ǿ","ǿ","Ö","ö","Ǫ̈","ǫ̈","Ȫ","ȫ",
         "Ơ̈","ơ̈","Ợ̈","ợ̈","Ọ̈","ọ̈","Ő","ő","Ǫ̋","ǫ̋","Ơ̋","ơ̋","Ợ̋","ợ̋",
         "Ọ̋","ọ̋","Å","AA","Aa","å","aa","Ǻ","AÁ","Aá","ǻ","aá",
         "AÀ","Aà","aà","AĂ","Aă","aă","AẮ","Aắ","aắ","AẰ","Aằ",
         "aằ","AẴ","Aẵ","aẵ","AẲ","Aẳ","aẳ","AÂ","Aâ","aâ","AẤ",
         "Aấ","aấ","AẦ","Aầ","aầ","AẪ","Aẫ","aẫ","AẨ","Aẩ","aẩ",
         "AǍ","Aǎ","aǎ","AÅ","Aå","aå","AǺ","Aǻ","aǻ","AÄ","Aä",
         "aä","AǞ","Aǟ","aǟ","AÃ","Aã","aã","AȦ","Aȧ","aȧ","AǠ",
         "Aǡ","aǡ","Ą̊","AĄ","Aą","ą̊","aą","AĀ","Aā","aā","AẢ",
         "Aả","aả","AȀ","Aȁ","aȁ","AȂ","Aȃ","aȃ","Ạ̊","AẠ","Aạ",
         "ạ̊","aạ","AẶ","Aặ","aặ","AẬ","Aậ","aậ","Ḁ̊","AḀ","Aḁ",
         "ḁ̊","aḁ"}},
    {
        -- Finnish (v and w are primary equal)
        "unicode_fi_s3", "stable",
        {"a","A","b","B","c","C","d","D","đ","Đ","e","E","f","F","g",
         "G","ǥ","Ǥ","h","H","i","I","j","J","k","K","l","L","m","M",
         "n","N","ŋ","Ŋ","o","O","p","P","q","Q","r","R","s","S","š",
         "Š","t","T","ŧ","Ŧ","u","U","v","V","w","W","x","X","y","Y",
         "ü","Ü","ǘ","Ǘ","ǜ","Ǜ","ǚ","Ǚ","ų̈","Ų̈","ǖ","Ǖ","ư̈","Ư̈","ự̈",
         "Ự̈","ụ̈","Ụ̈","ṳ̈","Ṳ̈","ṷ̈","Ṷ̈","ṵ̈","Ṵ̈","z","Z","ž","Ž","ʒ","Ʒ",
         "ǯ","Ǯ","å","Å","ǻ","Ǻ","ą̊","Ą̊","ạ̊","Ạ̊","ḁ̊","Ḁ̊","ä","Ä","ą̈",
         "Ą̈","ǟ","Ǟ","ạ̈","Ạ̈","ḁ̈","Ḁ̈","æ","Æ","ǽ","Ǽ","ǣ","Ǣ","ö","Ö",
         "ǫ̈","Ǫ̈","ȫ","Ȫ","ơ̈","Ơ̈","ợ̈","Ợ̈","ọ̈","Ọ̈","ø","Ø","ǿ","Ǿ"}},
    {
        -- German (umlaut as 'ae', 'oe', 'ue')
        "unicode_de__phonebook_s3", "stable",
        {"a","A","ä","ǟ","Ǟ","ą̈","Ą̈","ạ̈","Ạ̈","ḁ̈","Ḁ̈","Ä ","b","B","c","C",
         "d","D","e","E","f","F","g","G","h","H","i","I","j","J",
         "k","K","l","L","m","M","n","N","o","O","ȫ","Ȫ","ǫ̈","Ǫ̈",
         "ơ̈","Ơ̈","ợ̈","Ợ̈","ọ̈","Ọ̈","ö ","Ö ","p","P","q","Q","r","R",
         "s","S","ss","ß","t","T","u","U","ǘ","Ǘ","ǜ","Ǜ","ǚ","Ǚ",
         "ǖ","Ǖ","ų̈","Ų̈","ư̈","Ư̈","ự̈","Ự̈","ụ̈","Ụ̈","ṳ̈","Ṳ̈","ṷ̈","Ṷ̈",
         "ṵ̈","Ṵ̈","ü ","Ü ","v","V","w","W","x","X","y","Y","z","Z"}},
    {
        -- Hawaiian
        "unicode_haw_s3", "stable",
        {"a","A","ā","Ā","e","E","é","É","è","È","ĕ","Ĕ","ê","Ê","ế",
         "Ế","ề","Ề","ễ","Ễ","ể","Ể","ě","Ě","ë","Ë","ẽ","Ẽ","ė","Ė",
         "ȩ","Ȩ","ḝ","Ḝ","ę","Ę","ē","Ē","ḗ","Ḗ","ḕ","Ḕ","ẻ","Ẻ","ȅ",
         "Ȅ","ȇ","Ȇ","ẹ","Ẹ","ệ","Ệ","ḙ","Ḙ","ḛ","Ḛ","i","I","í","Í",
         "ì","Ì","ĭ","Ĭ","î","Î","ǐ","Ǐ","ï","Ï","ḯ","Ḯ","ĩ","Ĩ","İ",
         "į","Į","ī","Ī","ỉ","Ỉ","ȉ","Ȉ","ȋ","Ȋ","ị","Ị","ḭ","Ḭ","o",
         "O","ó","Ó","ò","Ò","ŏ","Ŏ","ô","Ô","ố","Ố","ồ","Ồ","ỗ","Ỗ",
         "ổ","Ổ","ǒ","Ǒ","ö","Ö","ȫ","Ȫ","ő","Ő","õ","Õ","ṍ","Ṍ","ṏ",
         "Ṏ","ȭ","Ȭ","ȯ","Ȯ","ȱ","Ȱ","ǫ","Ǫ","ǭ","Ǭ","ō","Ō","ṓ","Ṓ",
         "ṑ","Ṑ","ỏ","Ỏ","ȍ","Ȍ","ȏ","Ȏ","ơ","Ơ","ớ","Ớ","ờ","Ờ","ỡ",
         "Ỡ","ở","Ở","ợ","Ợ","ọ","Ọ","ộ","Ộ","u","U","ú","Ú","ù","Ù",
         "ŭ","Ŭ","û","Û","ǔ","Ǔ","ů","Ů","ü","Ü","ǘ","Ǘ","ǜ","Ǜ","ǚ",
         "Ǚ","ǖ","Ǖ","ű","Ű","ũ","Ũ","ṹ","Ṹ","ų","Ų","ū","Ū","ṻ","Ṻ",
         "ủ","Ủ","ȕ","Ȕ","ȗ","Ȗ","ư","Ư","ứ","Ứ","ừ","Ừ","ữ","Ữ","ử",
         "Ử","ự","Ự","ụ","Ụ","ṳ","Ṳ","ṷ","Ṷ","ṵ","Ṵ","h","H","k","K",
         "l","L","m","M","n","N","p","P","w","W","ʻ"}},
    {
        -- Hebrew
        "unicode_he_s3", "stable",
        {"׳","״","א","ב","ג","ד","ה","ו","ז","ח","ט","י","כ",
         "ך","ל","מ","ם","נ","ן","ס","ע","פ","ף","צ","ץ",
         "ק","ר","ש","ת"} },
    {
        -- Hindi
        "unicode_hi_s3", "stable",
        {"़","ॐ","ं","ँ","ः","अ","आ","इ","ई","उ","ऊ","ऋ","ऌ","ऍ",
         "ए","ऐ","ऑ","ओ","औ","क","ख","ग","घ","ङ","च","छ","ज","झ",
         "ञ","ट","ठ","ड","ढ","ण","त","थ","द","ध","न","प","फ","ब",
         "भ","म","य","र","ल","ळ","व","श","ष","स","ह","ऽ","ा","ि",
         "ी","ु","ू","ृ","ॄ","ॅ","े","ै","ॉ","ो","ौ","्"}},
    {
        -- Icelandic
        "unicode_is_s3", "stable",
        {"a","A","á","Á","ą́","Ą́","ạ́","Ạ́","ḁ́","Ḁ́","b","B","d","D",
         "đ","Đ","ð","Ð","e","E","é","É","ȩ́","Ȩ́","ę́","Ę́","ẹ́","Ẹ́",
         "ḙ́","Ḙ́","ḛ́","Ḛ́","f","F","g","G","h","H","i","I","í","Í",
         "į́","Į́","ị́","Ị́","ḭ́","Ḭ́","j","J","k","K","l","L","m","M",
         "n","N","o","O","ó","Ó","ǫ́","Ǫ́","ớ","Ớ","ợ́","Ợ́","ọ́","Ọ́",
         "p","P","r","R","s","S","t","T","u","U","ú","Ú","ų́","Ų́",
         "ứ","Ứ","ự́","Ự́","ụ́","Ụ́","ṳ́","Ṳ́","ṷ́","Ṷ́","ṵ́","Ṵ́","v","V",
         "x","X","y","Y","ý","Ý","ỵ́","Ỵ́","þ","Þ","æ","Æ","ǽ","Ǽ",
         "ǣ","Ǣ","ä","Ä","ą̈","Ą̈","ǟ","Ǟ","ạ̈","Ạ̈","ḁ̈","Ḁ̈","ö","Ö",
         "ǫ̈","Ǫ̈","ȫ","Ȫ","ơ̈","Ơ̈","ợ̈","Ợ̈","ọ̈","Ọ̈","ø","Ø","ǿ","Ǿ",
         "å","Å","ǻ","Ǻ","ą̊","Ą̊","ạ̊","Ạ̊","ḁ̊","Ḁ̊"}},
    {
        -- Japanese
        "unicode_ja_s3", "stable",
        {"幸","広","庚","康","弘","恒","慌","抗","拘","控","攻","港",
         "溝","甲","皇","硬","稿"}},
    {
        -- Korean
        "unicode_ko_s3", "stable",
        {"伽","佳","假","價","加","可","呵","哥","嘉","嫁","家","暇","架",
         "枷","柯","歌","珂","痂","稼","苛","茄","街","袈","訶","賈","跏",
         "軻","迦","駕","仮","傢","咖","哿","坷","宊","斝","榎","檟","珈",
         "笳","耞","舸","葭","謌","각","刻","却","各","恪","慤","殼","珏",
         "脚","覺","角","閣","卻","咯","埆","搉","擱","桷","갂","갃","간",
         "侃","刊","墾","奸","姦","干","幹","懇","揀","杆","柬","桿","澗",
         "癎","看","磵","稈","竿","簡","肝","艮","艱","諫","間","偘","慳",
         "栞","榦","玕","秆","茛","衎","赶","迀","齦","갅","갆","갇","갈",
         "乫","喝","曷","渴","碣","竭","葛","褐","蝎","鞨","噶","楬","秸",
         "羯","蠍","鶡","갉","갊","갋","갌","갍","갎","갏","감","勘","坎",
         "堪","嵌","感","憾","戡","敢","柑","橄","減","甘","疳","監","瞰",
         "紺","邯","鑑","鑒","龕","坩","埳","嵁","弇","憨","撼","欿","歛",
         "泔","淦","澉","矙","轗","酣","鹻","갑","匣","岬","甲","胛","鉀",
         "걭","걮","걯","거","倨","去","居","巨","拒","据","據","擧","渠",
         "炬","祛","距","踞","遽","鉅","鋸","呿","昛","秬","筥","籧","胠",
         "腒","苣","莒","蕖","蘧","袪","裾","駏","걱","걲","걳","건","乾",
         "件","健","巾","建","愆","楗","腱","虔","蹇","鍵","騫","揵","犍",
         "睷","褰","謇","鞬","걵","걶","걷","걸","乞","傑","杰","桀","乬",
         "朅","榤","걹","걺","걻","걼","걽","걾","걿","검","儉","劍","劒",
         "檢","瞼","鈐","黔","撿","芡","겁","劫","怯","迲","刦","刧","겂",
         "것","겄","겅","겆","겇","겈","겉","겊","겋","게","偈","憩","揭",
         "겍","겎","겏","겐","겑","겒","겓","겔","겕","겖","겗","겘","겙",
         "겚","겛","겜","겝","겞","겟","겠","겡","겢","겣","겤","겥","겦",
         "겧","겨","격","擊","格","檄","激","膈","覡","隔","挌","毄","闃",
         "骼","鬲","鴃","겪","겫","견","堅","牽","犬","甄","絹","繭","肩",
         "見","譴","遣","鵑","樫","狷","畎","筧","縳","繾","羂","蠲","鰹",
         "겭","겮","겯","결","抉","決","潔","結","缺","訣","玦","觖","闋",
         "겱","겲","겳","겴","겵","겶","겷","겸","兼","慊","箝","謙","鉗",
         "鎌","傔","嗛","岒","拑","歉","縑","蒹","黚","鼸","겹","겺","겻",
         "겼","경","京","俓","倞","傾","儆","勁","勍","卿","坰","境","庚",
         "徑","慶","憬","擎","敬","景","暻","更","梗","涇","炅","烱","璟",
         "璥","瓊","痙","硬","磬","竟","競","絅","經","耕","耿","脛","莖",
         "警","輕","逕","鏡","頃","頸","驚","鯨","冏","剄","哽","惸","憼",
         "扃","檠","煢","焭","熲","畊","竸","綆","顈","罄","褧","謦","駉",
         "鯁","黥","겾","겿","곀","곁","곂","곃","계","係","啓","堺","契",
         "季","屆","悸","戒","桂","械","棨","溪","界","癸","磎","稽","系",
         "捆","緄","衮","裍","褌","錕","閫","髡","鵾","鶤","齫","곥","곦",
         "곧","골","汨","骨","搰","榾","矻","鶻","곩","곪","곫","곬","곭",
         "곮","곯","곰","곱","곲","곳","곴","공","供","公","共","功","孔",
         "工","恐","恭","拱","控","攻","珙","空","蚣","貢","鞏","倥","崆",
         "豈","起","錡","錤","飢","饑","騎","騏","驥","麒","僛","剞","墍"}},
    {
        -- Kyrgyz
        "unicode_ky_s3", "unstable",
        {"а","А","б","Б","г","Г","д","ᲁ","Д","е","Е","ё","Ё","ж","Ж",
         "з","З","и","И","й","Й","к","К","л","Л","м","М","н","Н","ң","Ң",
         "о","ᲂ","О","ө","Ө","п","П","р","Р","с","ᲃ","С","т","ᲄ",
         "Т","у","У","ү","Ү","х","Х","ч","Ч","ш","Ш","ъ","ᲆ","Ъ","ы","Ы",
         "э","Э","ю","Ю","я","Я"}},
    {
        -- Kyrgyz (russian codepage)
        "unicode_ky_s3", "unstable",
        {"а","А","б","Б","в","В","г","Г","д","Д","е","Е","ё","Ё","ж","Ж",
         "з","З","и","И","й","Й","к","К","л","Л","м","М","н","Н",
         "о","О","п","П","р","Р","с","С","т","Т","у","У","ф","Ф",
         "х","Х","ц","Ц","ч","Ч","ш","Ш","щ","Щ","ъ","Ъ","ы","Ы",
         "ь","Ь","э","Э","ю","Ю","я","Я"}},
    {
        -- Lithuanian
        "unicode_lt_s3", "stable",
        {"a","A","ą","Ą","b","B","c","C","č","Č","ç̌","Ç̌","d","D",
         "e","E","ę","Ę","ė","Ė","ȩ̇","Ȩ̇","ẹ̇","Ẹ̇","ḙ̇","Ḙ̇","ḛ̇","Ḛ̇",
         "f","F","g","G","h","H","i","I","į","Į","y","Y","ý","Ý",
         "ỳ","Ỳ","ŷ","Ŷ","ẙ","ÿ","Ÿ","ỹ","Ỹ","ẏ","Ẏ","ȳ","Ȳ","ỷ",
         "Ỷ","ỵ","Ỵ","j","J","k","K","l","L","m","M","n","N","o",
         "O","p","P","r","R","s","S","š","Š","ṧ","Ṧ","ş̌","Ş̌","ṣ̌",
         "Ṣ̌","ș̌","Ș̌","t","T","u","U","ų","Ų","ū","Ū","ṻ","Ṻ","ư̄",
         "Ư̄","ự̄","Ự̄","ụ̄","Ụ̄","ṳ̄","Ṳ̄","ṷ̄","Ṷ̄","ṵ̄","Ṵ̄","v","V","z",
         "Z","ž","Ž","ẓ̌","Ẓ̌","ẕ̌","Ẕ̌"} },

    {
        -- Malayalam
        "unicode_ml_s3", "unstable",
        {"ഃ","ഽ","അ","ആ","ഇ","ഈ","ഉ","ഊ","ഋ","ൠ","ഌ","ൡ",
         "എ","ഏ","ഐ","ഒ","ഓ","ഔ","ക","ക്‍","ൿ","ഖ","ഗ","ഘ",
         "ങ","ച","ഛ","ജ","ഝ","ഞ","ട","ഠ","ഡ","ഢ","ണ","ണ്‍",
         "ൺ","ത","ഥ","ദ","ധ","ന","ൻ്","ന്‍","ൻ","പ","ഫ","ബ",
         "ഭ","മ ം","യ","ര","ര്‍","ർ","ല","ല്‍","ൽ","വ","ശ","ഷ",
         "സ","ഹ","ള","ള്‍","ൾ","ഴ","റ","ാ","ി","ീ","ു","ൂ","ൃ",
         "െ","േ","ൈ","ൊ","ോ","ൌ","ൗ"}},

    {
        -- Maltese
        "unicode_mt_s3", "unstable",
        {"A","a","À","à","B","b","Ċ","ċ","Ç̇","ç̇","D","d","E",
         "e","È","è","F","f","Ġ","ġ","Ģ̇","ģ̇","G","g","GĦ","gĦ",
         "Għ","għ","H","h","Ħ","ħ","I","i","Ì","ì","J","j","K",
         "k","L","l","M","m","N","n","O","o","Ò","ò","P","p",
         "Q","q","R","r","S","s","T","t","U","u","Ù","ù","V",
         "v","W","w","X","x","Ż","ż","Ẓ̇","ẓ̇","Ẕ̇","ẕ̇","Z","z"}},
    {
        -- Persian
        "unicode_fa_s3", "unstable",
        {"‫ِ ُ ً ٍ ٌ","ّ","ٔ","آ","ا","ٱ","ء","أ","ٲ","إ","ٳ","ؤ",
         "یٔ","ىٔ","ئ","ب","پ","ت","ث","ج","چ","ح","خ","د","ذ",
         "ر","ز","ژ","س","ش","ص","ض","ط","ظ","ع","غ","ف","ق",
         "ک","ڪ","ګ","ك","ڬ","ڭ","ڮ","گ","ل","م","ن","و",
         "ه","ە","ہ","ۂ","ة","ۃ","ۀ","ھ","ی","ى","ے","ۓ","ي",
         "ې","ۑ","ۍ","ێ","‬"}},
    {
        -- Polish
        "unicode_pl_s3", "stable",
        {"a","A","ą","Ą","b","B","c","C","ć","Ć","ḉ","Ḉ","d","D",
         "e","E","ę","Ę","f","F","g","G","h","H","i","I","j","J",
         "k","K","l","L","ł","Ł","m","M","n","N","ń","Ń","ņ́","Ņ́",
         "ṇ́","Ṇ́","ṋ́","Ṋ́","ṉ́","Ṉ́","o","O","ó","Ó","ǫ́","Ǫ́","ớ","Ớ",
         "ợ́","Ợ́","ọ́","Ọ́","p","P","r","R","s","S","ś","Ś","ṥ","Ṥ",
         "ş́","Ş́","ṣ́","Ṣ́","ș́","Ș́","t","T","u","U","w","W","y","Y",
         "z","Z","ź","Ź","ẓ́","Ẓ́","ẕ́","Ẕ́","ż","Ż","ẓ̇","Ẓ̇","ẕ̇","Ẕ̇"}},
    {
        -- Sinhala
        "unicode_si_s3", "stable",
        {"අ","ආ","ඇ","ඈ","ඉ","ඊ","උ","ඌ","ඍ","එ","ඒ","ඓ","ඔ",
         "ඕ","ඖ","ං","ඃ","ක","ඛ","ග","ඝ","ඞ","ඟ","ච","ඡ","ජ",
         "ඣ","ඥ","ඤ","ට","ඨ","ඩ","ඪ","ණ","ඬ","ත","ථ","ද","ධ",
         "න","ඳ","ප","ඵ","බ","භ","ම","ඹ","ය","ර","ල","ව","ශ","ෂ",
         "ස","හ","ළ","ෆ","ා","ැ","ෑ","ි","ී","ු","ූ","ෘ",
         "ෲ","ෟ","ෙ","ේ","ෛ","ො","ෝ","ෞ","්"}},
    {
        -- Spanish
        "unicode_es_s3", "stable",
        {"a","A","á","Á","b","B","c","C","d","D","e","E","é",
         "É","f","F","g","G","h","H","i","I","í","Í","ï","Ï",
         "j","J","k","K","l","L","m","M","n","N","ñ","Ñ","ņ̃",
         "Ņ̃","ṇ̃","Ṇ̃","ṋ̃","Ṋ̃","ṉ̃","Ṉ̃","o","O","ó","Ó","p","P",
         "q","Q","r","R","s","S","t","T","u","U","ú","Ú","ü",
         "Ü","v","V","w","W","x","X","y","Y","ý","Ý","z","Z"}},
    {
        -- Thai
        "unicode_th_s3", "unstable",
        {"ฯ ๆ ","ก","ข","ฃ","ค","ฅ","ฆ","ง","จ","ฉ","ช","ซ",
         "ฌ","ญ","ฎ","ฏ","ฐ","ฑ","ฒ","ณ","ด","ต","ถ","ท","ธ",
         "น","บ","ป","ผ","ฝ","พ","ฟ","ภ","ม","ย","ร","ฤ","ล","ฦ",
         "ว","ศ","ษ","ส","ห","ฬ","อ","ฮ ํ","ะ ั","า","ๅ ํา","ำ ํๅ ิ ี ึ ื ุ ู","เ",
         "แ","โ","ใ","ไ ฺ"}},
    {
        -- Ukrainian
        "unicode_uk_s3", "unstable",
        {"а","А","б","Б","в","ᲀ","В","г","Г","ґ","Ґ","д","ᲁ",
         "Д","е","Е","є","Є","ж","Ж","з","З","и","И","і","І",
         "ї","ꙶ","Ї","й","Й","к","К","л","Л","м","М","н","Н",
         "о","ᲂ","О","п","П","р","Р","с","ᲃ","С","т",
         "Т","у","У","ф","Ф","х","Х","ц","Ц","ч","Ч",
         "ш","Ш","щ","Щ","ь","Ь","ю","Ю","я","Я","ʼ"}}
}

for _, test_entry in ipairs(collation_entries) do
    -- create title
    local ext_prefix = string.format("%s1.%s.", prefix, test_entry[1])

    test:do_execsql_test(
            ext_prefix.."create_table",
            string.format("create table t1(a varchar(5) collate \"%s\" primary key);", test_entry[1]),
            {})
    test:do_test(
            ext_prefix.."insert_values",
            function()
                return insert_into_table("t1", test_entry[3])
            end, {})
    if test_entry[2] == "stable" then
        test:do_execsql_test(
                ext_prefix.."select",
                string.format("select a from t1 order by a"),
                test_entry[3])
    else -- Instead of unstable test execute a dummy test.
        test:do_execsql_test(
                ext_prefix.."unstable",
                string.format("values(1)"),
                {1})
    end
    test:do_execsql_test(
            ext_prefix.."drop_table",
            "drop table t1",
            {})
end

test:finish_test()
